#!/usr/bin/python3
#
# Convert a source file with embedded G code to a dnc-ready G-code file.
#
# Keep embedded G code, drop source comments starting with "#", and
# convert some embedded tokens to G code with python functions.
# 
# This lets us "hand-code" G code using some higher-level functions to
# generate the actual moves, but does NOT account for tool diameter,
# stock-to-leave, etc.
#
# Take input from stdin, process to stdout. Use shell redirection to
# store result, if desired.
#
# See README file for notes on tokens.

ver_string = "v1.3 (C)2023-2025 P Gettings"
import sys, math, time

out = sys.stderr.write
n = 0;

class MacroSettings:
  def __init__(self):
    self.g43 = False; # true ==> add Z.... G43 H# after tool change
    self.g53 = False; # true ==> add G0 G53 Z0. before M6
    self.tool_g0 = False; # true ==> put G0 on line before G53 Z0. for M6
    self.ztool = 0.0; # Z value for G43 lines setting tool offset
    self.com = True; # true ==> add comments for macro expansion, false ==> strip comments
    self.arcs = False; # true ==> convert arcs to lines
    self.arc_precision = 0.0002; # arc conversion precision or tolerance
    self.plane = 17; # arc plane, 17=XY, 18=ZX, 19=YZ as with g-codes
    self.pos = {'x':0., 'y':0., 'z':0., 'mode':0}
    self.nblocks = False; # true ==> produce N-block to number g-code lines
    self.max_linen = 9999; # maximum line number; reset to n_step when reached
    self.n_step = 2; # increment N-block numbers by n_step each time
    self.line_n = 0; # current line number for output n-blocks
    self.subramp = False; # true ==> ramp down to new z on first move
    self.named_routines = {} # dictionary of named subroutines
settings = MacroSettings()

####
## Functions to print G code from parsed macro line
####

def print_line(line):
  global settings
  if settings.nblocks:
    # check if line begins with %, O, or (
    if line[0] == '(' or line[0] == '%' or line[0] == 'O':
      print(line)
      return
    settings.line_n += settings.n_step
    if settings.line_n > settings.max_linen:
      settings.line_n = settings.n_step
    print("N%d %s"%(settings.line_n, line))
  else:
    print(line)

# helix down from zs to ze, circle with radius R, step down delta per
# revolution, center at x, y
def helix(x, y, zs, ze, delta, R, center_move):
  global settings
  if settings.com: print_line("(HELIX Z%.4f TO %.4f)"%(zs, ze))
  print_line("G1 X%+.4f Y%+.4f Z%+.4f"%(x-R, y, zs))
  n = math.ceil(math.fabs((ze-zs)/delta));
  d = (ze-zs)/n
  for i in range(n+1):
    z = zs + i*d
    print_line("G3 X%+.4f Y%+.4f Z%+.4f I%+.4f"%(x-R, y, z, R))
  print_line("G3 X%+.4f Y%+.4f Z%+.4f I%+.4f"%(x-R, y, ze, R))
  if center_move:
    print_line("G1 X%+.4f Y%+.4f"%(x, y))
  if settings.com: print_line("(END OF HELIX)")

# nibble along X axis, from xs to xe by steps of delta. Each pass, cut
# from ys to ye, then back off and move to ys, and take a new nibble
def nibble_x(xs, xe, ys, ye, delta):
  global settings
  if settings.com: print_line("(NIBBLE X)")
  print_line("G1 X%+.4f Y%+.4f"%(xs, ys))
  print_line("Y%+.4f"%ye)
  print_line("Y%+.4f"%ys)
  n = math.ceil(math.fabs((xe-xs)/delta))
  s = (xe-xs)/n
  for i in range(1,n+1):
    x = xs+i*s
    print_line("X%+.4f"%x)
    print_line("Y%+.4f"%ye)
    print_line("X%+.4f"%(x-s/2))
    print_line("Y%+.4f"%ys)
  print_line("G1 X%+.4f Y%+.4f"%(xs, ys))
  if settings.com: print_line("(END OF NIBBLE X)")

# nibble along Y axis, from ys to ye by steps of delta. Each pass, cut
# from xs to xe, then back off and move to xs, and take a new nibble
def nibble_y(xs, xe, ys, ye, delta):
  global settings
  if settings.com: print_line("(NIBBLE Y)")
  print_line("G1 X%+.4f Y%+.4f"%(xs, ys))
  print_line("X%+.4f"%xe)
  print_line("X%+.4f"%xs)
  n = math.ceil(math.fabs((ye-ys)/delta))
  s = (ye-ys)/n
  for i in range(1,n+1):
    y = ys+i*s
    print_line("Y%+.4f"%y)
    print_line("X%+.4f"%xe)
    print_line("Y%+.4f"%(y-s/2))
    print_line("X%+.4f"%xs)
  print_line("G1 X%+.4f Y%+.4f"%(xs, ys))
  if settings.com: print_line("(END OF NIBBLE Y)")

def angleCalc(x0, y0, x1, y1, radius, flip):
  dy = y1-y0;
  dx = x1-x0;
  alpha = math.atan2(dy, dx) # get angle
  rsa = radius*math.sin(alpha)
  rca = radius*math.cos(alpha)
  if flip:
    rsa *= -1.0
    rca *= -1.0
  return (rsa, rca)

# normalize components so that length is 1
def normalize(dx, dy):
  l = math.sqrt(dx*dx+dy*dy)
  return (dx/l, dy/l)

# offset point i in x, y arrays by radius distance, flip flag for
# left/right control. This algorithm from thinking of taking previous
# and next points, compute normal to previous and next lines, offset
# along normal radius distance, then solve for intersection of both
# offset lines. Test for parallel offset lines, which don't intersect -
# equation explodes to x = infinity; for parallel lines, just offset
# from x[i], y[i] along normal by radius distance. This idea of finding
# the intersection point of the displaced lines should handle the inside
# corner problem. Also have special cases for vertical lines, since
# slope calculations explode.
def offset_pt(x, y, i, radius, flip):
  npts = len(x)
  if flip: ccw = -1.0
  else: ccw = 1.0
  # Find previous point index, next point index, wrapping around end
  # of arrays
  p = (i + npts - 1) % npts # wrap index
  n = (i + 1) % npts

  # compute normal for previous leg
  dx =  x[i] - x[p]
  dy =  y[i] - y[p]
  if (dx*dx+dy*dy)<1e-10: # current and previous are same point!
    p = (i + npts - 2) % npts # so go back 2 points
    dx =  x[i] - x[p]
    dy =  y[i] - y[p]
  if math.fabs(dx) < 1e-10: # vertical special case
    mp = math.nan
  else:
    mp = dy/dx
  u, v = normalize(dx,dy)
  px = v*ccw
  py = -u*ccw

  # compute normal for next leg
  dx =  x[n] - x[i]
  dy =  y[n] - y[i]
  if (dx*dx+dy*dy)<1e-10: # current and next are same point!
    n = (i + 2) % npts # so go forward 2 points
    dx =  x[n] - x[i]
    dy =  y[n] - y[i]
  if math.fabs(dx) < 1e-10: # vertical special case
    mn = math.nan
  else:
    mn = dy/dx
  u, v = normalize(dx,dy)
  nx = v*ccw
  ny = -u*ccw

  # compute offset previous, next points
  xp = x[p]+px*radius
  yp = y[p]+py*radius
  xn = x[n]+nx*radius
  yn = y[n]+ny*radius
  # if next and previous leg have same slope, offset x[i], y[i] along
  # normal by radius
  if math.isnan(mp) and math.isnan(mn): # both lines vertical, so offset along normal
    ix = x[i]+(radius*ccw)
    iy = y[i]
  elif math.isnan(mp): # previous line vertical
    ix = x[i]+radius*px # offset x by radius
    iy = mn*(ix-xn)+yn # use next line eqn
  elif math.isnan(mn): # next line vertical
    ix = x[i]+radius*nx # offset x by radius
    iy = mp*(ix-xp)+yp # use previous line eqn
  elif (mp-mn) < 1e-6: # slopes nearly equal ==> parallel
    ix = x[i]+nx*radius # offset along normal
    iy = y[i]+ny*radius
  else: # find intersection of offset lines
    ix = (yn-yp+mp*xp-mn*xn) / (mp-mn)
    iy = mp*(ix-xp)+yp # could use either previous or next line eqn here
  # return offset coordinates
  return ix, iy
  
# run around a perimeter with X, Y, and optional Z values. Start at the
# first x, y, z and run through each location in order. Does NOT close
# the coordinate list - add starting location to end of list if closed
# perimeter desired!
# Z is optional in input, but x and y need entries for each line. Use
# nan in input to skip that coordinate for that location. End list of
# coordinates with single line containing only END
def perimeter(x, y, z, radius, flip): # arrays of coords, radius of tool
  global settings
  if settings.com: print_line("(PERIMETER WITH %d PTS)"%(len(x)))
  if radius > 1e-4: # radius for tool, so calculate offsets
    if flip: c = 'L'
    else: c='R'
    if settings.com: print_line("(OFFSET BY %.4f TO %c)"%(radius, c))
    print_line("G1");
    for i in range(len(x)):
      outs = ""
      nx, ny = offset_pt(x, y, i, radius, flip)
      outs += " X%+.4f"%nx
      outs += " Y%+.4f"%ny
      if not math.isnan(z[i]): outs += " Z%+.4f"%z[i]
      print_line(outs)
  else: # no radius, so just move around
    print_line("G1")
    for i in range(len(x)):
      outs = ""
      if not math.isnan(x[i]): outs += " X%+.4f"%x[i]
      if not math.isnan(y[i]): outs += " Y%+.4f"%y[i]
      if not math.isnan(z[i]): outs += " Z%+.4f"%z[i]
      print_line(outs)
  if settings.com: print_line("(END OF PERIMETER)")


# cut a slot from xs,ys to xe,ye at various z levels; start at zs, step
# down by delta, until ze. Offset x,y coordinates by radius and cut arc
# at each end, if radius > 0. If ccw is true, climb milling (CCW around
# slot); if false, conventional milling (CW around slot)
def slot(xs,xe, ys,ye, zs,ze, delta, radius, ccw):
  global settings
  if settings.com: print_line("(SLOT)")
  if radius > 1e-4: # radius for slot, so need calculations and arcs
    # compute new start, end coordinates
    rsa, rca = angleCalc(xs, ys, xe, ye, radius, ccw)
    if ccw: gnum = 3
    else: gnum = 2
    print_line("G0 X%+.4f Y%+.4f"%(xs-rsa, ys+rca))
    n = math.ceil(math.fabs((ze-zs)/delta))
    dz = (ze-zs)/n
    for i in range(n+1):
      z = zs+i*dz
      print_line("G1 Z%+.4f"%z)
      print_line("X%+.4f Y%+.4f"%(xe-rsa, ye+rca))
      print_line("G%d X%+.4f Y%+.4f I%+.4f J%+.4f"%(gnum, xe+rsa, ye-rca, rsa, -rca))
      print_line("G1 X%+.4f Y%+.4f"%(xs+rsa, ys-rca))
      print_line("G%d X%+.4f Y%+.4f I%+.4f J%+.4f"%(gnum, xs-rsa, ys+rca, -rsa, rca))
    print_line("G1 X%+.4f Y%+.4f"%(xs, ys))
  else: # no radius, so just move back and forth
    print_line("G0 X%+.4f Y%+.4f"%(xs, ys))
    n = math.ceil(math.fabs((ze-zs)/delta))
    dz = (ze-zs)/n
    for i in range(n+1):
      z = zs+i*dz
      print_line("G1 Z%+.4f"%z)
      print_line("X%+.4f Y%+.4f"%(xe, ye))
      print_line("X%+.4f Y%+.4f"%(xs, ys))
  if settings.com: print_line("(END OF SLOT)")

def ramp(xs,xe, ys,ye, zs,ze, delta, radius, ccw):
  global settings
  if settings.com: print_line("(RAMP)")
  if radius > 1e-4: # radius for slot, so need calculations and arcs
    # compute new start, end coordinates
    rsa, rca = angleCalc(xs, ys, xe, ye, radius, ccw)
    if ccw: gnum = 3
    else: gnum = 2
    print_line("G0 X%+.4f Y%+.4f"%(xs-rsa, ys+rca))
    n = math.ceil(math.fabs((ze-zs)/delta))
    dz = (ze-zs)/n
    print_line("G1 Z%+.4f"%zs)
    for i in range(n):
      z = zs+(i+1)*dz
      print_line("X%+.4f Y%+.4f Z%+.4f"%(xe-rsa, ye+rca, z))
      print_line("G%d X%+.4f Y%+.4f I%+.4f J%+.4f"%(gnum, xe+rsa, ye-rca, rsa, -rca))
      print_line("G1 X%+.4f Y%+.4f"%(xs+rsa, ys-rca))
      print_line("G%d X%+.4f Y%+.4f I%+.4f J%+.4f"%(gnum, xs-rsa, ys+rca, -rsa, rca))
    print_line("X%+.4f Y%+.4f"%(xe-rsa, ye+rca))
    print_line("G%d X%+.4f Y%+.4f I%+.4f J%+.4f"%(gnum, xe+rsa, ye-rca, rsa, -rca))
    print_line("G1 X%+.4f Y%+.4f"%(xs+rsa, ys-rca))
    print_line("G%d X%+.4f Y%+.4f I%+.4f J%+.4f"%(gnum, xs-rsa, ys+rca, -rsa, rca))
    print_line("G1 X%+.4f Y%+.4f Z%+.4f"%(xs, ys, z-dz))
  else: # no radius, so just move back and forth
    print_line("G0 X%+.4f Y%+.4f"%(xs, ys))
    n = math.ceil(math.fabs((ze-zs)/delta))
    dz = (ze-zs)/n
    print_line("G1 Z%+.4f"%zs)
    for i in range(n):
      z = zs+(i+1)*dz
      print_line("X%+.4f Y%+.4f Z%+.4f"%(xe, ye, z))
      print_line("X%+.4f Y%+.4f"%(xs, ys))
    print_line("X%+.4f Y%+.4f"%(xe, ye))
    print_line("X%+.4f Y%+.4f"%(xs, ys))
  if settings.com: print_line("(END OF RAMP)")
  
# repeat a chunk of code from zs to ze by steps of no more than delta
# depth. Like a perimeter code, but can include arcs, etc. If ramp set
# in settings, add move to first line of sub, otherwise move axis and
# then repeat subroutine.
# start, end, axis are arrays; order them the same! Must have same
# lengths!
def subroutine(lines, start, end, delta, axis):
  global settings
  if settings.com: print_line("(START SUB WITH %d LINES)"%len(lines))
  naxis = len(axis)
  if len(start) != naxis or len(end) != naxis:
    return; # mismatch, so give up
  n = 0
  for i in range(naxis):
    steps = math.ceil(math.fabs((end[i]-start[i])/delta))
    if steps > n: n = steps
  dz = []
  for i in range(naxis):
    dz.append((end[i]-start[i])/n)
  for i in range(-1,n): # iterate over steps in axes
    cx = 0.0; cy = 0.0; cz = 0.0;
    outstr = ""
    for j in range(naxis): # iterate over axes
      z = start[j]+(i+1)*dz[j]
      if axis[j] == 1: ltr="X"; cx=z;
      elif axis[j] == 2: ltr="Y"; cy=z;
      elif axis[j] == 3: ltr="Z"; cz=z;
      else: ltr = "Z" # default is Z
      outstr += " %s%+.4f"%(ltr, z)
    # output first line, then rest of subroutine
    if settings.subramp:
      outl = subsub(lines[0], cx, cy, cz)
      print_line("%s %s"%(outl, outstr))
      for i in range(1, len(lines)):
        outl = subsub(lines[i], cx, cy, cz)
        print_line(outl)
    else:
      print_line("G1 %s"%(outstr))
      for line in lines:
        outl = subsub(line, cx, cy, cz)
        print_line(outl)
  if settings.com: print_line("(END OF SUB)") 

# replace x, y, z fields in a subroutine line
def subsub(line, x, y, z):
  outl = line.replace("{x}", "%.4f"%x)
  outl = outl.replace("{y}", "%.4f"%y)
  outl = outl.replace("{z}", "%.4f"%z)
  return outl


# repeat a chunk of code by name
def call(name):
  global settings
  if name not in settings.named_routines:
    return # no named routine!
  sub = settings.named_routines[name]
  if settings.com: print_line("(CALL %s - %d LINES)"%(name, len(sub)))
  for line in sub:
    print_line(line)
  if settings.com: print_line("(END OF '%s')"%name) 


# code a tool change to tool number 't'
# if g53 flag set, print G0 G53 Z0. before tool change. If g43 flag set,
# add Z move with G43 H# after change, to set tool offset in control.
# Set flags with "set g43" and "set g53" before tool change call.
def tool_change(t):
  global settings
  if settings.com: print_line("(TOOL CHANGE TO T%d)"%t)
  if settings.g53:
    if settings.tool_g0:
      print_line("G0")
      print_line("G53 Z0.")
    else:
      print_line("G0 G53 Z0.")
  print_line("M6 T%d"%t)
  if settings.g43:
    print_line("G0 Z%+.4f G43 H%d"%(settings.ztool, t))
  if settings.com: print_line("(END OF TOOL CHANGE)")

####
## Useful utility functions from here down
####
def to_float(s):
  a = s.strip().upper()
  if a == "NAN":
    return(math.nan)
  else:
    return(float(a))

def parse_helix(fields):
  x=0; y=0; zs=0; ze=0; delta=0; radius=0; tool_r = 0.
  center_move = True;
  for f in fields[1:]:
    if f[0].upper()=="X": x=to_float(f[1:]) 
    elif f[0].upper()=="Y": y=to_float(f[1:])
    elif f[0].upper()=="Z":
      g = f[1:].split(",")
      zs=to_float(g[0])
      ze=to_float(g[1])
    elif f[0].upper()=="S": delta=to_float(f[1:])
    elif f[0].upper()=="D": tool_r=to_float(f[1:])/2.0
    elif f[0].upper()=="R": radius=to_float(f[1:])
    elif f[0].upper()=="N": center_move = False;
    else: out("## WARNING - unknown field '%s' in HELIX call, line %d\n"%(f, n))
  out("##%6d HELIX  x %f   y %f   zs %f ze %f    dz %f    r %f  tool %.4f"%
    (n, x, y, zs, ze, delta, radius, tool_r))
  if center_move: out(" end @ center")
  else: out(" end @ edge")
  out("\n")
  if (radius-(tool_r))<0:
    out("## ERROR - TOOL RADIUS (%.4f) > HELIX RADIUS (%.6f) in HELIX call, line %d. Die.\n"%
      (tool_r, radius, n))
    sys.exit(1)
  helix(x, y, zs, ze, delta, radius-tool_r, center_move)

def parse_nibblex(fields):
  for f in fields[1:]:
    if f[0].upper()=="X":
      g = f[1:].split(",")
      xs=to_float(g[0])
      xe=to_float(g[1])
    elif f[0].upper()=="Y":
      g = f[1:].split(",")
      ys=to_float(g[0])
      ye=to_float(g[1])
    elif f[0].upper()=="S": delta=to_float(f[1:])
    else: out("## WARNING - unknown field '%s' in NIBBLE_X call, line %d\n"%(f, n))
  out("##%6d NIBBLE_X   x %f to %f     y %f to %f   stepover %f\n"%(n, xs, xe, ys, ye, delta))
  nibble_x(xs, xe, ys, ye, delta)

def parse_nibbley(fields):
  for f in fields[1:]:
    if f[0].upper()=="X":
      g = f[1:].split(",")
      xs=to_float(g[0])
      xe=to_float(g[1])
    elif f[0].upper()=="Y":
      g = f[1:].split(",")
      ys=to_float(g[0])
      ye=to_float(g[1])
    elif f[0].upper()=="S": delta=to_float(f[1:])
    else: out("## WARNING - unknown field '%s' in NIBBLE_Y call, line %d\n"%(f, n))
  out("##%6d NIBBLE_Y   x %f to %f     y %f to %f   stepover %f\n"%(n, xs, xe, ys, ye, delta))
  nibble_y(xs, xe, ys, ye, delta)

def parse_slot(fields):
  ccw = False;
  for f in fields[1:]:
    if f[0].upper()=="X":
      g = f[1:].split(",")
      xs=to_float(g[0])
      xe=to_float(g[1])
    elif f[0].upper()=="Y":
      g = f[1:].split(",")
      ys=to_float(g[0])
      ye=to_float(g[1])
    elif f[0].upper()=="Z":
      g = f[1:].split(",")
      zs=to_float(g[0])
      ze=to_float(g[1])
    elif f[0].upper()=="S": delta=to_float(f[1:])
    elif f[0].upper()=="R": radius=to_float(f[1:])
    elif f[0].upper()=="C": ccw = True;
    else: out("## WARNING - unknown field '%s' in SLOT call, line %d\n"%(f, n))
  out("##%6d SLOT   x %.4f to %.4f     y %.4f to %.4f    z %.4f to %.4f   z_step %.4f   radius %.4f CCW %s\n"%
    (n, xs,xe, ys,ye, zs,ze, delta, radius, ccw))
  slot(xs, xe, ys, ye, zs, ze, delta, radius, ccw)

def parse_ramp(fields):
  ccw = False;
  for f in fields[1:]:
    if f[0].upper()=="X":
      g = f[1:].split(",")
      xs=to_float(g[0])
      xe=to_float(g[1])
    elif f[0].upper()=="Y":
      g = f[1:].split(",")
      ys=to_float(g[0])
      ye=to_float(g[1])
    elif f[0].upper()=="Z":
      g = f[1:].split(",")
      zs=to_float(g[0])
      ze=to_float(g[1])
    elif f[0].upper()=="S": delta=to_float(f[1:])
    elif f[0].upper()=="R": radius=to_float(f[1:])
    elif f[0].upper()=="C": ccw = True;
    else: out("## WARNING - unknown field '%s' in SLOT call, line %d\n"%(f, n))
  out("##%6d RAMP   x %.4f to %.4f     y %.4f to %.4f    z %.4f to %.4f   z_step %.4f   radius %.4f CCW %s\n"%
    (n, xs,xe, ys,ye, zs,ze, delta, radius, ccw))
  ramp(xs, xe, ys, ye, zs, ze, delta, radius, ccw)

def parse_perimeter(fields):
  global n
  global settings
  radius = 0.0; flip = False;
  for f in fields[1:]:
    if f[0].upper()=="D": radius = float(f[1:])/2.0
    elif f[0].upper()=="L": flip = True;
    else: out("## WARNING - unknown field '%s' in PERIMETER call, line %d\n"%(f, n))
  out("##%6d PERIMETER "%(n))
  x = []; y=[]; z=[]; cx=0; cy=0;
  for line in sys.stdin:
    n+=1;
    line = line.strip()
    if not line: continue
    if line.upper() == "END":
      break;
    coords = line.split(",")
    t = to_float(coords[0]);
    if math.isnan(t): x.append(cx)
    else: cx = t; x.append(t)
    t = to_float(coords[1]);
    if math.isnan(t): y.append(cy)
    else: cy = t; y.append(t)
    if len(coords)>2:
      z.append(to_float(coords[2]))
    else:
      z.append(math.nan)
  # finish, run routine, done
  out(" with %d points in circuit, radius offset %.4f, left flag %s\n"%
    (len(x), radius, flip))
  perimeter(x, y, z, radius, flip)

def parse_named(fields):
  global n
  global settings
  name = fields[1] # get name of subroutine for call line, later
  out("##%6d NAMED SUB '%s'"%(n, name))
  sublines = []
  for line in sys.stdin:
    n+=1
    line = line.strip()
    if not line: continue
    if line.upper() == "END":
      break;
    sublines.append(line)
  settings.named_routines[name] = sublines
  # finish read, return - wait for call command to spew
  out(" w/ %d lines stored\n"%(len(sublines)))
  return

def parse_call(fields):
  global n
  global settings
  name = fields[1] # name of stored sub to call
  if name in settings.named_routines:
    out("##%6d CALL '%s'\n"%(n, name))
    call(name)
  else:
    out("##%6d CALL '%s' - NO SUCH NAME AVAILABLE!\n"%(n, name))

def parse_sub(fields):
  global n
  global settings
  axis = []; start = []; end = []; ltr = ""
  for f in fields[1:]:
    if f[0].upper()=="S": delta = float(f[1:])
    elif f[0].upper()=="X":
      axis.append(1); ltr += "X"
      g = f[1:].split(",")
      start.append(to_float(g[0]))
      end.append(to_float(g[1]))
    elif f[0].upper()=="Y":
      axis.append(2); ltr += "Y"
      g = f[1:].split(",")
      start.append(to_float(g[0]))
      end.append(to_float(g[1]))
    elif f[0].upper()=="Z":
      axis.append(3); ltr += "Z"
      g = f[1:].split(",")
      start.append(to_float(g[0]))
      end.append(to_float(g[1]))
    else: out("## WARNING - unknown field '%s' in SUB call, line %d\n"%(f, n))
  out("##%6d SUBROUTINE"%(n))
  sublines = []
  for line in sys.stdin:
    n+=1
    line = line.strip()
    if not line: continue
    if line[0] == "#": continue # drop comments from inside the subroutine
    if line.upper() == "END":
      break;
    sublines.append(line)
  # finish, run routine, done
  out(" w/ %d lines repeated, stepping over %s\n"%
    (len(sublines), ltr))
  subroutine(sublines, start, end, delta, axis)

def parse_set(fields):
  global n
  global settings
  out("##%6d SET"%n)
  fields = line.split()
  for i in range(1,len(fields)):
    token = fields[i].upper()
    if token == "G43":
      settings.g43 = not settings.g43; out(" G43 to %s\n"%settings.g43)
    elif token == "G53":
      settings.g53 = not settings.g53; out(" G53 to %s\n"%settings.g53)
    elif token == "SPLIT_G53":
      settings.tool_g0 = not settings.tool_g0; out(" SPLIT_G53 to %s\n"%settings.tool_g0)
    elif token == "ZTOOL":
      settings.ztool = float(fields[i+1]);
      out(" ZTOOL to %+.4f\n"%settings.ztool);
      break
    elif token == "NOC":
      settings.com = False; out(" NO COMMENTS\n"); # turn off comments, and strip G code comments
    elif token == "COMMENT":
      settings.com = True; out(" COMMENTS ON\n"); # turn on comments
    elif token == "ARC2LINE":
      settings.arcs = True; # convert arcs to lines
      out(" ARC CONVERSION TO LINES, PRECISION %+.4f\n"%settings.arc_precision);
      continue
    elif token == "ARCP":
      settings.arc_precision = float(fields[i+1]);
      out(" ARC CONVERSION PRECISION %+.4f\n"%settings.arc_precision);
      break;
    elif token == "ARCS":
      settings.arcs = False; out(" ARCS CONVERSION OFF\n"); # leave arcs
    elif token == "NBLOCKS": # if true, output N-block line numbers
      settings.nblocks = True; # true ==> produce N-block to number g-code lines
      out(" LINE NUMBERS\n");
    elif token == "NONBLOCKS": # if true, output N-block line numbers
      settings.nblocks = False; # false ==> no line numbering
      out(" NO LINE NUMBERS\n");
    elif token == "NMAX": # maximum line number; reset to n_step
      settings.max_linen = int(fields[i+1]); # maximum line number; reset to n_step when reached
      out(" MAX LINE NUMBER %d\n"%(settings.max_linen)); break;
    elif token == "NSTEP": # line number delta
      settings.n_step = int(fields[i+1]); # increment N-block numbers by n_step each time
      out(" LINE NUMBER STEP %d\n"%(settings.n_step)); break;
    elif token == "RAMP":
      settings.subramp = True; out(" SUBROUTINE RAMPING ON\n"); # turn on sub ramping
    elif token == "NORAMP":
      settings.subramp = False; out(" SUBROUTINE RAMPING OFF\n"); # turn off sub ramping
    else:
      out("\n## WARNING - unknown argument %s to set!\n"%(token));

def parse_tool(fields):
    global n
    fields = line.split()
    t = int(fields[1])
    tool_change(t)
    out("##%6d TOOL CHANGE to T%d\n"%(n, t))

# 
# convert arc move to linear segments
def arc2lines(mode, plane, xs, ys, zs, xe, ye, ze, xoff, yoff, zoff, nrev, arc_res):
  global settings
  radius = math.sqrt((xoff*xoff)+(yoff*yoff)+(zoff*zoff))
  centerx = xs + xoff;
  centery = ys + yoff;
  centerz = zs + zoff;
  if plane == 18: # XZ plane
    us=xs; ue=xe; centeru = centerx
    vs=zs; ve=ze; centerv = centerz
    ws=ys; we=ye;
    ultr = "X"; vltr = "Z"; wltr = "Y";
  elif plane == 19: # YZ plane
    us=ys; ue=ye; centeru = centery
    vs=zs; ve=ze; centerv = centerz
    ws=xs; we=xe;
    ultr = "Y"; vltr = "Z"; wltr = "X";
  else: # 17 & default = XY plane
    us=xs; ue=xe; centeru = centerx
    vs=ys; ve=ye; centerv = centery
    ws=zs; we=ze;
    ultr = "X"; vltr = "Y"; wltr = "Z";
  alpha_s = math.atan2((vs-centerv), (us-centeru))
  alpha_e = math.atan2((ve-centerv), (ue-centeru))
  if math.fabs(alpha_s - alpha_e) < 1e-6: # same start, end angle ==> full rev
    alpha_e = alpha_s + 2*math.pi
  alpha_e = alpha_e + 2*math.pi*(nrev-1) # add extra revs
  npts = int(math.ceil(math.fabs(radius*(alpha_e-alpha_s))/arc_res))
  dalpha = (alpha_e - alpha_s)/npts;
  # ensure angle change correct for CW, CCW rotation
  if mode == 2: 
    if dalpha > 0: dalpha *= -1.0;
  else:
    if dalpha < 0: dalpha = math.fabs(dalpha)
  dw = (we-ws)/npts;
  if settings.com:
    print_line("(CONVERT ARC TO LINES)")
  print_line("G1")
  for i in range(npts):
    alpha = alpha_s + (i+1)*dalpha
    u = centeru + math.cos(alpha)*radius
    v = centerv + math.sin(alpha)*radius
    w = zs + (i+1)*dz
    print_line("%s%+.4f %s%+.4f %s%+.4f"%(ultr, u, vltr, v, wltr, w))

# break apart blocks run together without spaces
def canonize_blocks(blocks):
  for j in range(len(blocks)):
    # skip all processing once comment found
    if blocks[j][0] == "(": 
      break;
    # check for letters in block after start ==> blocks run together, break apart
    for l in 'ABCDEFGHIJKLMNOPQRSTUVWXYZ':
      if l in blocks[j][1:]:
        [blocks[j], newblock] = blocks[j].split(l, 1)
        blocks.append("%c%s"%(l,newblock))

    if blocks[j] == '': # skip entry entry
      del blocks[j];
  return blocks

def parse_gcode(fields):
  global settings
  cx = settings.pos['x']
  cy = settings.pos['y']
  cz = settings.pos['z']
  arc_x = 0.
  arc_y = 0.
  arc_z = 0.
  mode = settings.pos['mode']
  # break apart smashed blocks
  blocks = canonize_blocks(fields)
  flag = false; nrev=0;
  # find move type and locations
  out_blks = []
  for b in blocks:
    ltr = b[0]
    if ltr == 'X': cx = float(b[1:]); flag=True
    elif ltr == 'Y': cy = float(b[1:]); flag=True
    elif ltr == 'Z': cz = float(b[1:]); flag=True
    elif ltr == 'G':
      gn = int(b[1:])
      if gn < 4: # movement mode
        mode = gn
        out_blks.append(b)
        flag = True
      elif gn == 17: settings.plane = 17
      elif gn == 18: settings.plane = 18
      elif gn == 19: settings.plane = 19
      else:
        flag = False
    elif ltr == 'I': arc_x = float(b[1:]); flag=True
    elif ltr == 'J': arc_y = float(b[1:]); flag=True
    elif ltr == 'K': arc_z = float(b[1:]); flag=True
    elif ltr == 'P': nrev = int(b[1:]); flag=True;
    elif ltr == 'N' and not settings.com: # drop N blocks if com false
      continue
    elif ltr == '(' and not settings.com: # drop comments
      break;
    out_blks.append(b)

  # if an arc move, and converting arcs, convert
  if len(out_blks) > 0:
    if flag and settings.arcs and (mode==2 or mode==3):
      arc2lines(mode, settings.plane,
        settings.pos['x'], settings.pos['y'], settings.pos['z'],
        cx, cy, cz, arc_x, arc_y, arc_z, nrev, settings.arc_precision)
    else:
      print_line(out_blks.join(" "))
  settings.pos['x'] = cx
  settings.pos['y'] = cy
  settings.pos['z'] = cz
  settings.pos['mode'] = mode


####
## Main routine from here....
####

out("##############\n")
out("## CAM Macro Expansion Processor\n");
out("## %s\n"%ver_string);
out("## run as %s\n"%sys.argv[0])
out("## run at %s\n"%time.asctime())
out("##############\n")
out("##Line #       Macro\n")
for line in sys.stdin:
  n+=1
  line = line.strip()
  if not line: continue
  if line[0] == "#":
    out(line); out("\n");
    continue # comment line, send to stderr
  fields = line.upper().split()
  token  = fields[0]
  if token == "HELIX":
    parse_helix(fields);
  elif token == "NIBBLE_X":
    parse_nibblex(fields)
  elif token == "NIBBLE_Y":
    parse_nibbley(fields)
  elif token == "SLOT":
    parse_slot(fields)
  elif token == "RAMP":
    parse_ramp(fields)
  elif token == "PERIMETER":
    parse_perimeter(fields)
  elif token == "SET": # set flags, values for macro processing; 1 per line!
    parse_set(fields)
  elif token == "TOOL": # do a tool change: TOOL num
    parse_tool(fields)
  elif token == "SUB": # run a subroutine from zs to ze by steps delta
    parse_sub(fields)
  elif token == "DEF": # define a named subroutine for later call
    parse_named(fields)
  elif token == "CALL": # call a named subroutine already defined
    parse_call(fields)
  else: # not a macro, so parse as g-code
    # if comments ok & !convert arcs, spit out line
    if settings.com and not settings.arcs:
      print_line(line);
    else:
      parse_gcode(fields)
out("## %d lines in\n"%n)
# done

