#!/usr/bin/python3
#
# Simple DNC program for talking to ancient CNC controls.
#
# Use pyserial for the serial port comm stuff; respect xon/xoff for flow
# control. Check for DC2, DC4 for EOF from yasnac controls.
#
# Note that modern linux kernel serial drivers don't actually respect
# xon/xoff, like they should.  This is not a bug in termios, but in the
# serial port drivers for USB-to-serial adapters and the like; they
# simply don't implement any kind of flow control in the driver itself.
#
# SO, need to not only check for xon/xoff, but do so after every single
# character sent; the MX3 on the 1990-vintage Matsuura mill only has a
# 10 character buffer for sender to respond to xoff!!!

import sys, time
import serial
import argparse

filename = "./default"
portName = "/dev/ttyUSB0"
baud = 9600
parity = serial.PARITY_NONE
hwFlowControl = False # no hardware flow control for USB adapters
xonFlowControl = True # software flow control with xon, xoff characters
charSize = serial.EIGHTBITS # or 8 if no parity
stopBits = serial.STOPBITS_ONE # 1 stop bit
eolTranslate = False # true ==> convert LF to CR LF
dc24Flag = False # true ==> send DC2, 4 to start, stop send
rawFlag = False # true ==> don't check for EOL, etc.

rxFlag = False # if True, receive file; if false, send file TO control

# process command line arguments
parser = argparse.ArgumentParser(prog='pydnc.py',
        description='Simple DNC program for ancient CNC controls',
        epilog='Defaults: port settings 9600 8N1 XON/XOFF; send file ./default')
parser.add_argument('-a', help='toggle raw mode; no EOL checking', action='store_true')
parser.add_argument('-d', help='toggle DC2, DC4 transmission for start, stop', action='store_true')
parser.add_argument('-e', help='toggle translate LF to CR', action='store_true')
parser.add_argument('-f', '--file', help='filename to send or receive')
parser.add_argument('-r', action='store_true', help='receive a file, instead of send')
parser.add_argument('-F', '--device', help='serial port device filename')
parser.add_argument('-b', '--baud', help='baudrate in bps', type=int)
parser.add_argument('-s', '--stopbits', help='number of stop bits', type=int)
parser.add_argument('-c', '--charsize', help='number of data bits', type=int)
parser.add_argument('-p', '--parity',
      help='parity- none even odd mark space',
      choices=['none', 'even', 'odd', 'mark', 'space'])
parser.add_argument('-x', action='store_true', help='turn off all flow control')
parser.add_argument('-w', action='store_true',
      help='turn on RTS/CTS hardware flow control; turn off xon/xoff.')


args = vars(parser.parse_args())

out = sys.stderr.write

for (k,v) in args.items(): # process arguments
  if v: # if there is a value to process
    if k == 'e':
      eolTranslate = not eolTranslate
    elif k == 'a':
      rawFlag = not rawFlag
    elif k == 'd':
      dc24Flag = not dc24Flag
    elif k == 'file':
      filename = v;
    elif k == 'r':
      rxFlag = True
      out("pyDNC: setting to receive mode\n")
    elif k == 'device':
      portName = v
    elif k == 'baud':
      baud = v
    elif k == 'stopbits':
      if v == 1:
        stopBits = serial.STOPBITS_ONE
      elif v == 2:
        stopBits = serial.STOPBITS_TWO
    elif k == 'charsize':
      if v == 7:
        charSize = serial.SEVENBITS
      elif v == 8:
        charSize = serial.EIGHTBITS
    elif k == 'parity':
      if v == 'none':
        parity = serial.PARITY_NONE
      elif v == 'even':
        parity = serial.PARITY_EVEN
      elif v == 'odd':
        parity = serial.PARITY_ODD
      elif v == 'mark':
        parity = serial.PARITY_MARK
      elif v == 'space':
        parity = serial.PARITY_SPACE
    elif k == 'x':
      xonFlowControl = False
      hwFlowControl = False
    elif k == 'w':
      hwFlowControl = True
      xonFlowControl = False

out("pyDNC: opening serial port....\n")
sys.stderr.flush()
# open serial port, set options
port = serial.Serial(port=portName, baudrate=baud, bytesize=charSize,
        parity=parity, stopbits = stopBits, timeout=0,
        xonxoff=xonFlowControl, rtscts=hwFlowControl)
if port.is_open:
  port.reset_input_buffer()
  port.reset_output_buffer()
else:
  out("Port '%s' failed to open. Die.\n"%portName)
  sys.exit(1)
if   parity == serial.PARITY_SPACE: p="S"
elif parity == serial.PARITY_NONE: p="N"
elif parity == serial.PARITY_EVEN: p="E"
elif parity == serial.PARITY_ODD: p="O"
elif parity == serial.PARITY_MARK: p="M"
else: p="unknown"

out("pyDNC: %s @ %d %d%c%d "%(portName, baud, charSize, p, stopBits))
if xonFlowControl: out("XON/XOFF")
elif hwFlowControl: out("RTS/CTS")
else: out("NONE")
out("\n")
out("pyDNC: opening file '%s'....\n"%filename)
# open file for send or receive
if rxFlag: # receive, open and truncate output file
  fp = open(filename, "wt")
  if not fp:
    out("output file '%s' failed to open. Die.\n"%filename);
    sys.exit(1)
else: # send, open for reading
  fp = open(filename, "rb")
  if not fp:
    out("input file '%s' failed to open. Die.\n"%filename);
    sys.exit(1)

# if receiving, grab characters and write, waiting for second '%'
if rxFlag:
  out("pyDNC: starting receive from control; send it....\n")
  if rawFlag:
    out("pyDNC: raw receive mode.\n")
  elif eolTranslate:
    out("pyDNC: translating CR to LF....\n")
  flag = False
  while(1): # will break from loop on DC4
    d = port.read(1)
    if len(d)>0:
      if d[0] == 0x0: # skip nulls
        continue
      if d[0]&0x7f == 0x14: # DC4 = EOT for some controls
        out("\npyDNC: EOT received, finish up...\n")
        if rawFlag:
          fp.write("%c"%chr(d[0]))
          fp.flush()
        break
      elif d[0]&0x7f == 0x12: # DC2 = SOT for some controls
        out("\npyDNC: SOT received, ignoring...\n")
        if rawFlag:
          fp.write("%c"%chr(d[0]))
          fp.flush()
        continue
      else:
        out("%c"%chr(d[0]))
      if rawFlag:
        fp.write("%c"%chr(d[0]))
        fp.flush()
      elif chr(d[0]) == '%': # % starts, stops file write
        flag = not flag
        fp.write("%")
        out("\npyDNC: %% received, file write %s...\n"%str(flag))
        fp.flush()
      elif flag: # write to file if between %s
        if chr(d[0]) == '\r':
          if eolTranslate:
            fp.write("\n")
        else:
          fp.write("%c"%chr(d[0]))
    else:
      fp.flush()
      time.sleep(0.0001)

# if sending, run through file 1 char at a time, checking for xon from control
else:
  out("pyDNC: starting send to control....\n")
  if eolTranslate:
    out("pyDNC: translating LF to CR LF....\n")
  if dc24Flag:
    out("pyDNC: sending DC2 to start transmission....\n")
    port.write(b'\x12') # send DC2 to start transmission
    port.flush()
  c = fp.read(1); # read character from file
  while(c):
    d = port.read(1)
    if len(d)>0: # if something from the control, check for pause
      if d[0]&0x7f == 0x13: # XOFF, DC3 - wait
        out("\n.....buffer full, pause")
        sys.stderr.flush()
        while(1):
          d = port.read(1)
          if len(d)>0:
            if d[0]&0x7f == 0x11:  # XON, DC1 - resume
              out(".....resume.\n")
              sys.stderr.flush()
              break
          time.sleep(0.001)
      else: out(" [0x%02x] "%d[0]) # write received character for debugging
    if c[0] == 0x0a and eolTranslate: # if line feed & translation
      port.write(b'\r') # add CR before LF
      port.write(b'\n')
    else:
      port.write(c)
    port.flush()
    out("%c"%chr(c[0]))
    c = fp.read(1);
  if dc24Flag:
    out("pyDNC: sending DC4 to end transmission....\n")
    port.write(b'\x14') # send DC4 to end transmission
    port.flush()
out("pyDNC: close file, "); sys.stderr.flush()
fp.close()
out("close port, "); sys.stderr.flush()
port.flush()
port.close()
out("quit....\n"); sys.stderr.flush()
sys.exit(0)


