/*
 * vis.c
 *
 * Converts block geometry to VRML 2.0 for visualization on a
 * workstation.
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include "model.h"

#define NCOLORS 255 /* # of entries in the colormap */

typedef struct ColorscaleStruct {
  double r[NCOLORS];
  double g[NCOLORS];
  double b[NCOLORS];
  double p[NCOLORS];
  } Colorscale;

void usage(void);
void write_header(char *filename);
Colorscale colorscale(GravityModel model);
void write_blocks(GravityModel model, Colorscale c, float transparency);

void usage(void)
{
  printf("usage: vis [-t transparency] model_file\n");
  printf("VRML code is sent to stdout.\n");
}

int main(int argc, char *argv[])
{
  GravityModel model;
  Colorscale colors;
  int i;
  float transparency=0.5;
  char opt;
  char *optstring="t:";

  if(argc<2) {
    usage();
    exit(1);
    }

  /* parse command options */
  while((opt = getopt(argc, argv, optstring)) != -1) {
    switch(opt) {
      case 't': /* set transparency */
        transparency = (float)atof(optarg);
        fprintf(stderr, "transparency being set to %f\n", transparency);
        break;
      case '?': /* unknown option */
        usage();
        exit(1);
        break;
      default:
        break;
      }
    }

  /* load the model geometry */
  model = loadModel(argv[optind]);

  /* compute colorscale */
  colors = colorscale(model);

  /* write the VRML header */
  write_header(argv[1]);

  /* write the blocks */
  write_blocks(model, colors, transparency);

  /* done */
  printf("# done\n");
}

/* Print a VRML 2.0 header and WorldInfo node */
void write_header(char *filename)
{
  printf("#VRML V2.0 utf8\n\n");
  printf("WorldInfo {\n");
  printf("  title \"Gravity Model %s\"\n", filename);
  printf("  info [\"Created by vis\"]\n}\n");
}

/* Create a colormap from the range of densities in the model */
Colorscale colorscale(GravityModel model)
{
  long i;
  double rho_step, Pmin, Pmax;
  double ar, ag, bg, ab;
  Colorscale c;

  /* find min/max density of model */
  Pmin = Pmax = 0.0;
  for(i=0; i<model.nelem; i++) {
    if(Pmax < model.elem[i].rho) {
      Pmax = model.elem[i].rho;
      }
    if(Pmin > model.elem[i].rho) {
      Pmin = model.elem[i].rho;
      }
    }

  rho_step = (Pmax - Pmin)/NCOLORS;

  ar =  1.0 / (NCOLORS*NCOLORS);
  ag = -4.0 / (NCOLORS*NCOLORS);
  bg =  4.0 / NCOLORS;
  ab = -1.0 / (NCOLORS*NCOLORS);

  for(i = 0; i<NCOLORS; i++) {
    c.r[i] = ar * i*i;
    c.g[i] = ag * i*i + bg * i;
    c.b[i] = ab * i*i + 1.0;
    c.p[i] = Pmin + rho_step * i;
    }

  return(c);
}

/* Print the VRML code for the blocks of the model */
/* Could probably save a bit of file space by using defined blocks and
 * colors, but that is only useful if we wanted to edit the file by
 * hand.
 */
void write_blocks(GravityModel model, Colorscale c, float transparency)
{
  long i;
  int j, k;
  double d, od;

  double fov, vpos, D;
  double xmax, xmin, ymax, ymin, zmax;

  xmax = xmin = model.elem[0].x;
  ymax = ymin = model.elem[0].y;
  zmax = model.elem[0].z;
  printf("\n# Start of model elements\n");
  for(i=0; i<model.nelem; i++) {
    printf("Transform {\n");
    printf("  translation %lf %lf %lf\n", model.elem[i].x,
      model.elem[i].y, model.elem[i].z);
    printf("  children [\n");
    printf("    Shape  {\n");
    printf("      appearance Appearance {\n");
    printf("        material Material {\n");
    /* find the right color for the box */
    k = 0; od = fabs(c.p[0] - model.elem[i].rho);
    for(j=0; j<NCOLORS; j++) {
      d = fabs(c.p[j] - model.elem[i].rho);
      if(d < od) {
        k = j; od = d;
        }
      }
    printf("          emissiveColor %lf %lf %lf\n", c.r[k], c.g[k],
      c.b[k]);
    printf("          diffuseColor %lf %lf %lf\n", c.r[k], c.g[k],
      c.b[k]);
    printf("          transparency %lf\n", transparency);
    printf("        }\n");
    printf("      }\n");
    printf("      geometry Box {\n");
    printf("        size %lf %lf %lf\n", model.elem[i].xlen,
      model.elem[i].ylen, model.elem[i].zlen);
    printf("      }\n");
    printf("    }\n");
    printf("  ]\n");
    printf("}\n\n");

    if(model.elem[i].x > xmax) xmax = model.elem[i].x;
    if(model.elem[i].x < xmin) xmin = model.elem[i].x;
    if(model.elem[i].y > ymax) ymax = model.elem[i].y;
    if(model.elem[i].y < ymin) ymin = model.elem[i].y;
    if(model.elem[i].z > zmax) zmax = model.elem[i].z;
    }

  /* Create a viewpoint to see the whole model */
  printf("\n# Viewpoint\n");
  fov = 0.78; /* radians */
  /* Choose z value for camera that puts all of model in view */
  /* so choose the maximum value of all (x,y) element coords,
     and then camera z >= 2*D / tan(fov/2)
  */
  D = fabs(xmax);
  if(fabs(ymax) > D) D = fabs(ymax);
  if(fabs(ymin) > D) D = fabs(ymin);
  if(fabs(xmin) > D) D = fabs(xmin);
  vpos = 2*D / tan(fov/2.0);
  if(vpos < 2*zmax) vpos = 2*zmax;
  printf("Viewpoint {\n");
  printf("  fieldOfView %lf\n", fov);
  printf("  position 0 0 %lf\n", -vpos);
  printf("  orientation 1 0 0 3.14\n");
  printf("}\n");

  /* Create a set of axes for reference */
  printf("\n# Axes\n");
  /* want axes from origin to (xmax, ymax, vpos) */
  printf("#   X axis (green)\n");
  printf("Transform {\n");
  printf("  rotation 0 0 1 1.57\n");
  printf("  translation %lf 0 0\n", (vpos/2.0));
  printf("  children [\n");
  printf("    Shape  {\n");
  printf("      appearance Appearance {\n");
  printf("        material Material {\n");
  printf("          emissiveColor 0.0 1.0 0.0\n");
  printf("          diffuseColor 0.0 1.0 0.0\n");
  printf("        }\n");
  printf("      }\n");
  printf("      geometry Cylinder {\n");
  printf("        height %lf\n", vpos);
  printf("      }\n");
  printf("    }\n");
  printf("  ]\n");
  printf("}\n\n");
  printf("#   Y axis (blue)\n");
  printf("Transform {\n");
  printf("  rotation 0 0 0 0\n");
  printf("  translation 0 %lf 0\n", (vpos/2.0));
  printf("  children [\n");
  printf("    Shape  {\n");
  printf("      appearance Appearance {\n");
  printf("        material Material {\n");
  printf("          emissiveColor 0.0 0.0 1.0\n");
  printf("          diffuseColor 0.0 0.0 1.0\n");
  printf("        }\n");
  printf("      }\n");
  printf("      geometry Cylinder {\n");
  printf("        height %lf\n", vpos);
  printf("      }\n");
  printf("    }\n");
  printf("  ]\n");
  printf("}\n\n");
  printf("#   Z axis (red)\n");
  printf("Transform {\n");
  printf("  rotation 1 0 0 1.57\n");
  printf("  translation 0 0 %lf\n", (vpos/2.0));
  printf("  children [\n");
  printf("    Shape  {\n");
  printf("      appearance Appearance {\n");
  printf("        material Material {\n");
  printf("          emissiveColor 1.0 0.0 0.0\n");
  printf("          diffuseColor 1.0 0.0 0.0\n");
  printf("        }\n");
  printf("      }\n");
  printf("      geometry Cylinder {\n");
  printf("        height %lf\n", vpos);
  printf("      }\n");
  printf("    }\n");
  printf("  ]\n");
  printf("}\n\n");
}
