/*
 * simanneal.cpp
 *
 * Simulated Annealing inversion code for 3d grav problems
 *
 * Be sure to link with the math library - the code uses pow(), exp(),
 * etc.!
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000, 2001 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <time.h>
#include "model.h"
#include "mt.h"
#include "gaussrand.h"

static char *version_id = "1.2.0";

/*
 * DATA TYPES
 * GLOBAL VARS
 */
typedef struct {
  double sigma;		// step size to perturb densities
  double s0;		// constant to compute step size from current 'time'
  double sc;		// decay constant for sigma
  double smin;		// decay constant for sigma
  double T0;		// initial 'temperature' for Simul. Anneal process
  double D;		// constant in T decay equation
  double T;		// current 'temperature'
  unsigned long maxiter;// maximum # of iterations
  double obserr;	// error of observed gravity, in mGal
  double errthresh;	// error where fit acceptable
  unsigned int regularize; // regularization flag
  double alpha;		// regularization parameter
  } AnnealParms;

AnnealParms parameters;

GravityModel m, mstar;	// m is current model, mstar is best model
GravityModel mapr;	// mpr is the a priori model, for comparison,
	 	 	// etc.
/*
 * FUNCTION PROTOTYPES
 */
void perturbModel(double sigma, Position *op, unsigned long np, double
  gobs[], double gp[]);
		// Perturbs the existing best-fit
		// model to search for a better one.
		// Specialize to the constraints of
		// a problem!
void updateT(unsigned long k);
void updateSigma(unsigned long k);
unsigned long updateModel(unsigned long k);
double E(double gobs[], double gp[], unsigned long np); 
unsigned long anneal(Position op[], double gobs[], unsigned long np);
void usage(void);

/*
 * FUNCTION SOURCE
 */
/*
 * Perturb the existing best-fit model to find
 * a better one.
 *
 * This function should be tweaked and redone for
 * particular problems.  The function gets lots of arguments,
 * in case they are useful for some problem. Note the current
 * function uses only "sigma" and none of the other args....
 *
 * The function should update global GravityModel "m" using global
 * GravityModel "mstar" and whatever magic pertains to the problem.
 *
 * This is only a simple example.
 *
 * THIS FUNCTION WILL CONVERGE TO A RANDOM MODEL WITHOUT ANY PHYSICAL
 * PLAUSIBILITY!!!  It requires additional constraints to maintain
 * geologic meaning.  Alternatively, regularize the problem by changing
 * the error function (E()) to include a measure of the distance of the
 * current model from the a priori model!
 */
void perturbModel(double sigma, Position *op, unsigned long np, double
  gobs[], double gp[])
{
  unsigned long i;

  // determine perturbation
  for(i=0; i<mstar.nelem; i++) {
    m.elem[i].rho = gaussrand(mstar.elem[i].rho, sigma);
    }
  return;
}

/*
 * Update the 'temperature' - determines how often an error-increasing
 * model is adopted
 *
 * 2 possible methods of changing the 'temperature' with 'time'
 * T = T0/log(k) guaranteed to converge to global min, but slow
 * so normally use T=T0*D^k
 */
void updateT(unsigned long k)
{
  parameters.T = parameters.T0*pow(parameters.D, k);
    // guaranteed to find global minimum(?), but slower
  //parameters.T = parameters.T0 / log(k);
}

/*
 * Heuristic to update the random perturbation step of the model:
 *
 * Take the initial perturbation and decrease as a function of
 * 'time'
 */
void updateSigma(unsigned long k)
{
  // new sigma
  parameters.sigma = parameters.s0 * pow(parameters.sc, k);
  // set a minimum randomness
  if(parameters.sigma < parameters.smin) parameters.sigma = parameters.smin;
}

/*
 * Update the best-fit model to the working model
 *
 * Also updates T, sigma, etc.
 * Anything that needs to change with the best-fit model should
 * go in here.
 */
unsigned long updateModel(unsigned long k)
{
  copyModel(m, &mstar);
  updateT(k);
  updateSigma(k);
  return(++k);
}

/*
 * Error/cost function.
 *
 * This is the function that is minimized.
 */
double E(double gobs[], double gp[], unsigned long np)
{
  unsigned long i;
  double err=0, dg;

  for(i=0; i<np; i++) {
    dg = (gobs[i] - gp[i]);
    err += dg*dg;
    }

  // Tikhonov regularization
  if(parameters.regularize) {
    for(i=0; i<m.nelem; i++)
      err += parameters.alpha*fabs(m.elem[i].rho - mapr.elem[i].rho);
      }

  return(err);
}

/*
 * Inversion routine
 *
 * op	observed gravity positions, as Position array w/np entries
 * gobs	observed gravity anomalies, in same order as op
 * np	# of gravity positions
 * gp	predicted gravity array; created and filled by anneal()
 */
unsigned long anneal(Position op[], double gobs[], unsigned long np)
{
  unsigned long i, j, k, l;
  double err, olderr, dE=0;

  double *gp;

  char run = 1;

  if((gp = (double *)malloc(np*sizeof(double))) == NULL) {
    fprintf(stderr, "cannot allocate memory for %ld predicted gravity values\n",
      np);
    return(0);
    }

  fprintf(stderr,
    "Iteration  Model_Num.          Error           Error_change Perturbation  Temperature\n"); 

  k = l = 0;

  /* compute starting error */
  for(i=0; i<np; i++) {
    gp[i] = gravity(m, op[i]);
    gp[i] *= 1e5; // convert to mGal
    }
  err = E(gobs, gp, np);
  olderr = err;
  // set initial perturbation size
  updateSigma(k);

  fprintf(stderr, "%10d %10d %24.9lf %12.6lf %12.6lf %12.6lf\n",
    l, k, err, dE, parameters.sigma, parameters.T);

  while(run) {
    l++;
    fprintf(stderr, "%10lu\b\b\b\b\b\b\b\b\b\b", l);

    /* test for sufficient fit, failsafe */
    if(err < parameters.errthresh) {
      run=0;
      fprintf(stderr, "Convergence - error < threshhold\n");
      continue;
      }
    else if(l > parameters.maxiter) {
      run = 0;
      fprintf(stderr, "Bailout - maximum iterations reached!\n");
      continue;
      }

    /* current model not good enough, so pick a new model */
    perturbModel(parameters.sigma, op, np, gobs, gp);

    /* compute gravity effect */
    for(i=0; i<np; i++) {
      gp[i] = gravity(m, op[i]);
      gp[i] *= 1e5; // convert to mGal
      }

    /* compute new error */
    err = E(gobs, gp, np);

    /* compute dE */
    dE = err - olderr;

    /* accept new model? */
    if(dE < 0) {
      k=updateModel(k);
      olderr = err;
      fprintf(stderr, "%10d %10d %24.9lf %12.6lf %12.6lf %12.6lf\n",
	l, k, err, dE, parameters.sigma, parameters.T);
      }
    else {
      if(genrand() < exp(-1.0 * dE / parameters.T)) {
	k=updateModel(k);
	olderr = err;
	fprintf(stderr, "%10d %10d %24.9lf %12.6lf %12.6lf %12.6lf\n",
	  l, k, err, dE, parameters.sigma, parameters.T);
        }
      }
    }
  fprintf(stderr, "Inversion terminated after %lu iterations, %lu updates; final step error=%lf\n",
    l, k, err);
  /*
   * write gravity error grid for plotting
   */
  fprintf(stderr, "Writing gravity error surface to standard out.\n");
  //printf("# gravity error surface\n");
  for(i=0; i<np; i++) {
    printf("%lf %lf %lf %lf\n", op[i].x, op[i].y, op[i].z, gobs[i]-gp[i]);
    }
  return(l);
}

/*
 * program usage printer
 */
void usage(void)
{
  fprintf(stderr, "usage: simanneal [options] model_file grid_file\n");
  fprintf(stderr, "where options are:\n");
  fprintf(stderr, " -S sigma0         set initial perturbation size; currently %lf\n",
    parameters.s0);
  fprintf(stderr, " -s sigma_constant set decay constant for perturbation size; currently %lf\n",
    parameters.sc);
  fprintf(stderr, " -m min_sigma      set minimum perturbation size; currently %lf\n",
    parameters.smin);
  fprintf(stderr, " -o output_file    output file; default is model.out\n");
  fprintf(stderr, " -T T0             set initial 'temperature' of S.A. process\n");
  fprintf(stderr, "                   default=%lf\n", parameters.T0);
  fprintf(stderr, " -D D              set constant in T-decay equation; currently %f\n",
    parameters.D);
  fprintf(stderr, " -E thresh         set observation error level (mGal); currently %lf\n",
    parameters.obserr);
  fprintf(stderr, " -k bailout        set maximum number of iterations; currently %lu\n",
    parameters.maxiter);
  fprintf(stderr, " -R                regularize the inversion using Tikhonov metric\n");
  fprintf(stderr, " -A                set regularization parameter; currently %lf\n",
    parameters.alpha);
  fprintf(stderr, "model_file is a standard utah-g3d model geometry file.\n");
  fprintf(stderr, "grid_file is a blank-delimited 4-column file, in x y z obs_gravity order\n");
  fprintf(stderr, "Observed gravity should be in mGal.\n");
  fprintf(stderr, "Both files allow blank and comment lines ");
  fprintf(stderr, "(put # as the first non-whitespace character).\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "Output is the new best-fit model, and the gravity error surface.\n");
  fprintf(stderr, "Error surface is sent to stdout, as 4 column data: x y z dg (in mGal)\n");
}

/*
 * Note that the inversion routine is not here
 */
int main(int argc, char *argv[])
{
  char *optstr = "S:s:m:o:T:D:E:k:RA:hH";
  int opt;

  unsigned long i, k;

  Position *op;	// array of observation positions
  double *gobs; // observed gravity array
  unsigned long ngp; // # of grid/obs. gravity points

  FILE *fp;
  char *record;
  char outname[1025];

  // setup default parameters
  parameters.s0 = 500;
  parameters.sc = 0.9;
  parameters.smin = 10.0;
  parameters.T0 = 1.5;
  parameters.D = 0.8;
  parameters.obserr = 5e-3;
  parameters.maxiter = 1000000;
  parameters.regularize=0;
  parameters.alpha = 1.0;
  strcpy(outname, "model.out");

  if(argc < 2) {
    usage();
    exit(1);
    }
  while((opt = getopt(argc, argv, optstr)) != -1) {
    switch(opt) {
      case 'S': parameters.s0 = atof(optarg); break;
      case 's': parameters.sc = atof(optarg); break;
      case 'm': parameters.smin = atof(optarg); break;
      case 'E': parameters.obserr = atof(optarg); break;
      case 'k': parameters.maxiter = (unsigned long)atol(optarg); break;
      case 'o': // output file name
	strncpy(outname, optarg, 1024);
	outname[1024] = '\0';
        break;
      case 'T': parameters.T0 = atof(optarg); break;
      case 'D': parameters.D = atof(optarg); break;
      case 'R': parameters.regularize = 1; break;
      case 'A': parameters.alpha = atof(optarg); break;
      case 'h':
      case 'H':
      default:
        usage();
        exit(1);
        break;
      }
    }
  parameters.T = parameters.T0;

  /*
   * start banner, etc.
   */
  fprintf(stderr, "----------------------------------------\n");
  fprintf(stderr, "Simulated Annealing 3D Gravity Inversion\n");
  fprintf(stderr, "          v%s\n", version_id);
  fprintf(stderr, "\n");
  fprintf(stderr, "    Copyright (C)2001\n");
  fprintf(stderr, "    Paul Gettings, Dep't of Geology & Geophysics\n");
  fprintf(stderr, "    University of Utah\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "    This program is part of utah-g3d.\n");
  fprintf(stderr, "----------------------------------------\n");

  fprintf(stderr, "Inversion parameters:\n");
  fprintf(stderr, " Initial perturbation step:%24lf\n", parameters.s0);
  fprintf(stderr, " Step decay constant:      %24lf\n", parameters.sc);
  fprintf(stderr, " Minimum step:             %24lf\n", parameters.smin);
  fprintf(stderr, " T0:%24lf\n", parameters.T0);
  fprintf(stderr, " D: %24lf\n", parameters.D);
  fprintf(stderr, " Observation error:%24lf mGal\n", parameters.obserr);
  fprintf(stderr, " Max. Iterations: %24ld\n", parameters.maxiter);
  fprintf(stderr, " Regularize? ");
  if(parameters.regularize) fprintf(stderr, "YES\n");
  else fprintf(stderr, "no\n");
  fprintf(stderr, " Regularization parameter: %24lf\n", parameters.alpha);

  /*
   * load the a priori model
   */
  m = loadModel(argv[optind]);
  mstar.nelem = 0;
  copyModel(m, &mstar); // automatically reallocates mstar.elem!
  mapr.nelem = 0;
  copyModel(m, &mapr); // automatically reallocates mapr.elem!
  fprintf(stderr, "Loaded a priori model from '%s' with %ld elements.\n",
    argv[optind], mstar.nelem);

  /*
   * load the computation grid & observed gravity anomalies
   */
  if((fp = fopen(argv[optind+1], "rt")) == NULL) {
    printf("Cannot open %s for read.\n", argv[optind+1]);
    exit(0);
    }
  k=0;
  while(!feof(fp)) {
    record = getrecord(fp);
    if(record == NULL) continue;
    k++;
    }
  rewind(fp);
  if((op = (Position *)malloc(k*sizeof(Position))) == NULL) {
    fprintf(stderr, "cannot allocate memory for %ld positions\n", k);
    exit(1);
    }
  if((gobs = (double *)malloc(k*sizeof(double))) == NULL) {
    fprintf(stderr, "cannot allocate memory for %ld observed gravity values\n", k);
    exit(1);
    }
  ngp = k;
  i=0;
  while(!feof(fp)) {
    record = getrecord(fp);
    if(record == NULL) continue;
    if(i == ngp) continue;
    sscanf(record, "%lf %lf %lf %lf", &op[i].x, &op[i].y, &op[i].z, &gobs[i]);
    i++;
    }
  fclose(fp);
  fprintf(stderr, "Loaded computation grid and observed gravity from '%s':\n",
    argv[optind+1]);
  fprintf(stderr, "  %ld grid positions\n", ngp);

  /* Set the error threshhold, depending on the number of obs pts and
     measurement errors. */
  parameters.errthresh = (parameters.obserr*parameters.obserr)*ngp;

  fprintf(stderr, "  setting error threshhold to %lf mGal\n",
    parameters.errthresh);

  /*
   * Invert
   */
  init_gaussrand(); // get a new sequence of random numbers each time

  k = anneal(op, gobs, ngp);

  /*
   * write ending model
   */
  fprintf(stderr, "Writing best-fit model to '%s'.\n", outname);
  saveModel(outname, mstar);


  // done
  fprintf(stderr, "done.\n");
  exit(0);
}

/*
 * END OF FILE
 */
