/*
 * openglvis.cpp
 *
 * Visualizes model geometry in an OpenGL window.
 * Faster than vis.c and a VRML viewer, but no transparency
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000, 2001 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */
#include <GL/glut.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include "model.h"

// version string
static char *VERSION="1.6.0";

// viewport position, look vector, and model rotation
GLfloat vx, vy, vz;
GLfloat lx, ly, lz;
GLfloat rx, ry, rz;
  // half-angle of Field of View, in radians
GLfloat FOV = 0.34;

// clipping variables
GLdouble MINX_CLIP, MAXX_CLIP;
GLdouble MINY_CLIP, MAXY_CLIP;
GLdouble NEAR_CLIP,  FAR_CLIP, MINFAR_CLIP;
GLdouble MINX, MAXX;
GLdouble MINZ, MAXZ;

// viewport size
GLint viewportWidth, viewportHeight;
#define TEXT_HEIGHT	20	// height of parameter subwindow

// Movement step sizes for pan, zoom, etc.
GLfloat Xstep=1;
GLfloat Ystep=1;
GLfloat Zstep=1;
  // these are in degrees!
GLfloat Astep=1.0;//angle step for model rotations

// main window and subwindow IDs
GLint mainWin, subWin;

// other global vars
GravityModel model;
GLuint *block;
GLuint xAxisList, yAxisList, zAxisList; // display lists
GLuint positionList;			// 
unsigned char positionFlag;	// flag if there are grid positions to
				// plot
int rotateAxisFlag=0;	// flag that determines what axis we rotate
			// around
int zoomAxisFlag=0;	// flag for whether zoom is clip or zoom

// Colormap defines and vars
#define NCOLORS 255 /* # of entries in the colormap */
typedef struct ColorscaleStruct {
  double r[NCOLORS];
  double g[NCOLORS];
  double b[NCOLORS];
  double p[NCOLORS]; //density!
  } Colorscale;


/*
 * Function prototypes
 */
static void reshape(GLint w,GLint h);
static void draw(void);
static void reshapeParms(GLint w,GLint h);
static void drawParms(void);
static void idle(void);
static void update();
void keyPress(GLint key, GLint x, GLint y);
void normalKeyPress(unsigned char key, GLint x, GLint y);
void mouseEvent(GLint button, GLint state, GLint x, GLint y);
void passiveMouseMotion(GLint x, GLint y);
void mouseMotion(GLint x, GLint y);
void setClippingPlanes(void);
GLuint blockDisplayList(float xs, float ys, float zs, GLfloat color[4]);
void createAxisLists(void);
GLuint createPositionList(Position *p, long N, GLfloat color[4]);
Colorscale colorscale(GravityModel model);
void putText(float x, float y, float z, char *string);
Position *loadPositions(char *file, long *np);

/*
 * Window reshape callback
 */
static void reshape(GLint w,GLint h) {
  if(w == 0) w = 1;

  GLfloat r=(GLfloat)h/w;

  // reset the viewport size
  glViewport(0,0,w,h);
  viewportWidth = w;
  viewportHeight= h;

  // set the projection matrix for perspective
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  // set horiz. clipping planes
  MINY_CLIP = r*MINX_CLIP; MAXY_CLIP = r*MAXX_CLIP;
  //glFrustum(MINX_CLIP,MAXX_CLIP,MINY_CLIP,MAXY_CLIP,NEAR_CLIP,FAR_CLIP);
  glOrtho(MINX_CLIP,MAXX_CLIP,MINY_CLIP,MAXY_CLIP,NEAR_CLIP,FAR_CLIP);

  // set the viewpoint
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  //glTranslatef(vx, vy, vz);
  gluLookAt(vx, vy, vz, vx+lx, vy+ly, vz+lz, 0, 1, 0);

  // set the directional light
  static GLfloat pos[4]={0,0,0,1};
  pos[0] = vx; pos[1] = vy; pos[2] = vz;
  glLightfv(GL_LIGHT1,GL_POSITION,pos);

  reshapeParms(w, TEXT_HEIGHT);

#ifdef DEBUG
  printf("reshape: clipping planes-\n");
  printf("reshape: x:%f/%f y:%f/%f n/f:%f/%f\n",MINX_CLIP, MAXX_CLIP,
    MINY_CLIP, MAXY_CLIP, NEAR_CLIP, FAR_CLIP);
  printf("reshape: viewpoint parms-\n");
  printf("reshape: v: %f, %f, %f\n", vx, vy, vz);
  printf("reshape: l: %f, %f, %f\n", lx, ly, lz);
  printf("reshape: r: %f, %f, %f\n", rx, ry, rz);
#endif
}

/*
 * Window reshape callback
 */
static void reshapeParms(GLint w,GLint h) {
  if(w == 0) w = 1;

  GLfloat r=(GLfloat)h/w;

  glutSetWindow(subWin);
  // reset the viewport size
  glViewport(0,0,w,h);

  // set the projection matrix for perspective
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  // set horiz. clipping planes
  //MINY_CLIP = r*MINX_CLIP; MAXY_CLIP = r*MAXX_CLIP;
  glOrtho(0,w,0,TEXT_HEIGHT,0,100);

  // set the viewpoint
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  // set the viewpoint
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  gluLookAt(0, 0, 50, 0, 0, 0, 0, 1, 0);

  // set the directional light
  static GLfloat pos[4]={0,0,-50,1};
  glLightfv(GL_LIGHT1,GL_POSITION,pos);
  glutSetWindow(mainWin);
}

/*
 * Renders the scene
 */
static void draw(void) {
  long i;

#ifdef DEBUG
  GLint s=glutGet(GLUT_ELAPSED_TIME);
#endif

  glutSetWindow(mainWin);
  // clear screen & depth buffer
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // save the current matrix
  glPushMatrix();
  // rotate the model space
  glRotatef(rx, 1, 0, 0);
  glRotatef(ry, 0, 1, 0);
  glRotatef(rz, 0, 0, 1);
  // draw model blocks
  for(i=0; i<model.nelem; i++) {
    glPushMatrix();
      //glBlendFunc(GL_ONE, GL_ZERO);
      // translate the current matrix
      glTranslatef(model.elem[i].x, model.elem[i].y, model.elem[i].z);
      glScalef(model.elem[i].xlen/2.0, model.elem[i].ylen/2.0,
	model.elem[i].zlen/2.0);
      glCallList(block[i]);
    // restore previous matrix
    glPopMatrix();
    }
  // draw computation grid positions
  if(positionFlag) {
    glCallList(positionList);
    }
  // draw axes
  glCallList(xAxisList);
  glCallList(yAxisList);
  glCallList(zAxisList);
  // restore original matrix
  glPopMatrix();

  // swap buffers - allows animation
  // also forces a flush
  glutSwapBuffers();

#ifdef DEBUG
  GLint e=glutGet(GLUT_ELAPSED_TIME);

  printf("end draw: %d ms\n", e-s);
#endif

}

/*
 * Renders the scene
 */
static void drawParms(void) {
  // clear screen & depth buffer
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // plot text label
  char buf[256], buf2[512], axisLetter, zoomLetter;
  GLfloat color[4] = {1, 1, 1, 1};
  glMaterialfv(GL_FRONT,GL_AMBIENT_AND_DIFFUSE,color);
  glColor4fv(color);
  axisLetter = 'N';
  if(rotateAxisFlag == 1) axisLetter = 'X';
  if(rotateAxisFlag == 2) axisLetter = 'Y';
  if(rotateAxisFlag == 4) axisLetter = 'Z';
  if(!zoomAxisFlag) zoomLetter = 'Z';
  else zoomLetter = 'C';
  snprintf(buf, 255, "rx:%6.1f ry:%6.1f rz:%6.1f rotate:%c zoom/clip:%c",
    rx, ry, rz, axisLetter, zoomLetter);
  snprintf(buf2, 255, "%s n/f:%8.3f/%8.3f",
    buf, NEAR_CLIP, FAR_CLIP);
  putText(1, 0, 0, buf2);

  // swap buffers - allows animation
  // also forces a flush
  glutSwapBuffers();
}

/*
 * put text string "string" at position x, y, z
 */
void putText(float x, float y, float z, char *string)
{
  char *c;
  float pos[3];

  pos[0] = x; pos[1] = y; pos[2] = z;

  glRasterPos3fv(pos);
  for(c=string; *c != '\0'; c++) {
    glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, *c);
    }
}

/*
 * idle loop callback
 */
static void idle(void) {
  // redisplay
  glutPostRedisplay();
}

/*
 * Update projection and redraw the scene
 */
static void update()
{
  glutSetWindow(mainWin);
  reshape(viewportWidth, viewportHeight);
  glutPostRedisplay();
  glutSetWindow(subWin);
  glutPostRedisplay();
  glutSetWindow(mainWin);
}

/*
 * Keyboard event callback
 */
void keyPress(GLint key, GLint x, GLint y)
{
  int params[2];
  // handle a keypress event
  switch(key) {
    case 'q':
    case 'Q':
    case 27 : //escape
      exit(0);
      break;
    case 'r': //reset viewdirection
      setClippingPlanes();
      rx = 0; ry = 0; rz = 0;
      break;
    case 'n':
      rotateAxisFlag = 0;
      break;
    case 'x':
      rotateAxisFlag = 1;
      break;
    case 'y':
      rotateAxisFlag = 2;
      break;
    case 'z':
      rotateAxisFlag = 4;
      break;
    case 'c':
      zoomAxisFlag = !zoomAxisFlag;
      break;
    case 'w':
      glGetIntegerv(GL_POLYGON_MODE, params);
      if(params[0] == GL_FILL || params[1] == GL_FILL)
	glPolygonMode(GL_FRONT_AND_BACK, GL_LINE); //wireframe
      else
	glPolygonMode(GL_FRONT_AND_BACK, GL_FILL); //solid
      break;
    default:
      break;
    }
  update();
}

/*
 * callback for normal key presses
 */
void normalKeyPress(unsigned char key, GLint x, GLint y)
{
  keyPress(key, x, y);
}

// Mouse button states
unsigned char buttonMask=0;
#define LEFTBUTTON	0x01
#define MIDDLEBUTTON	0x02
#define RIGHTBUTTON	0x04

// previous mouse (x,y)
GLint oldx, oldy;

/*
 * mouse button click callback
 */
void mouseEvent(GLint button, GLint state, GLint x, GLint y)
{
  switch(state) {
    case GLUT_DOWN:
      if(button == GLUT_LEFT_BUTTON) buttonMask |= LEFTBUTTON;
      if(button == GLUT_MIDDLE_BUTTON) buttonMask |= MIDDLEBUTTON;
      if(button == GLUT_RIGHT_BUTTON) buttonMask |= RIGHTBUTTON;
      break;
    case GLUT_UP:
      if(button == GLUT_LEFT_BUTTON) buttonMask &= ~LEFTBUTTON;
      if(button == GLUT_MIDDLE_BUTTON) buttonMask &= ~MIDDLEBUTTON;
      if(button == GLUT_RIGHT_BUTTON) buttonMask &= ~RIGHTBUTTON;
      break;
    }
}

/*
 * mouse motion w/o button callback
 */
void passiveMouseMotion(GLint x, GLint y)
{
  oldx = x;
  oldy = y;
}

/* 
 * mouse motion w/button held down
 */
void mouseMotion(GLint x, GLint y)
{
  GLint dx, dy, delta;
  GLfloat theta, phi;
  GLfloat dX, dY, dZ, L;

  dx = x-oldx; dy = y-oldy;
  delta = dx;
  if(abs(dy) > abs(delta)) delta = dy;
  if(buttonMask & LEFTBUTTON) { //rotate
    if(rotateAxisFlag == 1) {
      rx += delta*Astep;
      }
    if(rotateAxisFlag == 2) {
      ry += delta*Astep;
      }
    if(rotateAxisFlag == 4) {
      rz -= delta*Astep;
      }
#ifdef DEBUG
    printf("rotate: %f %f %f\n", rx, ry, rz);
#endif
    }
  if(buttonMask & MIDDLEBUTTON) { //zoom or clip
    if(!zoomAxisFlag) {
      // zoom by changing the clipping planes
      FAR_CLIP += delta*Zstep;
      if(FAR_CLIP < MINFAR_CLIP) {
	FAR_CLIP = MINFAR_CLIP;
	}
      MINX_CLIP += -delta*Zstep*tan(FOV);
      MAXX_CLIP +=  delta*Zstep*tan(FOV);
      if(MINX_CLIP > MINX) {
	MINX_CLIP = MINX;
	}
      if(MAXX_CLIP < MAXX) {
	MAXX_CLIP = MAXX;
	}
      }
    else { // clip
      NEAR_CLIP += -1.0*delta*Zstep;
      if(NEAR_CLIP < 0)
	NEAR_CLIP = 0;
      if(NEAR_CLIP > FAR_CLIP)
	NEAR_CLIP = FAR_CLIP - 1;
      }
#ifdef DEBUG
    printf("zoom: dx=%d dy=%d delta=%d\n", dx, dy, delta);
#endif
    }
  if(buttonMask & RIGHTBUTTON) { //pan
    /*if(lx == 0) theta = 0.0;
    else theta = atan(ly/lx);
    if(ly == 0) phi = 0.0;
    else phi = atan(lz/ly);
    */
    dX = dx*Xstep;
    dY = dy*Ystep;
    dZ = 0;
#ifdef DEBUG
    printf("pan: dx=%d dy=%d delta=(%f, %f, %f)\n", dx, dy, dX, dY, dZ);
#endif
    vx += dX;
    vy += dY;
    vz += dZ;
    }
  oldx=x;
  oldy=y;
  // redraw the scene
  update();
}

/*
 * Set clipping plane variables
 *
 * also sets MINX, MAXX, MINZ, MAXZ
 */
void setClippingPlanes(void)
{
  long i;
  float minx, miny, minz;
  float maxx, maxy, maxz;

  minx = maxx = model.elem[0].x;
  miny = maxy = model.elem[0].y;
  minz = maxz = model.elem[0].z;

  for(i=1; i<model.nelem; i++) {
    if(model.elem[i].x > maxx) maxx = model.elem[i].x;
    if(model.elem[i].y > maxy) maxy = model.elem[i].y;
    if(model.elem[i].z > maxz) maxz = model.elem[i].z;
    if(model.elem[i].x < minx) minx = model.elem[i].x;
    if(model.elem[i].y < miny) miny = model.elem[i].y;
    if(model.elem[i].z < minz) minz = model.elem[i].z;
    }
  vx = (maxx+minx)*0.5;
  vy = (maxy+miny)*0.5;
  vz = -(maxx-minx)/tan(FOV);
  lx = 0; ly = 0; lz = 1.0;
  MINX_CLIP = vx - fabs(vz)*tan(FOV);
  MAXX_CLIP = vx + fabs(vz)*tan(FOV);
  // MINY_CLIP, MAXY_CLIP set in reshape()
  NEAR_CLIP = 1;
  FAR_CLIP = (maxz - vz);
  MINFAR_CLIP = FAR_CLIP;
  MINX = minx;
  MAXX = maxx;
  MINZ = minz;
  MAXZ = maxz;
#ifdef DEBUG
  printf("setClippingPlanes: X:[%f,%f] Y:[%f,%f] Z:[%f,%f]\n", minx,
    maxx, miny, maxy, minz, maxz);
#endif
}

/*
 * create a display list for a single block
 * because of the glScale call in draw(),
 * we can normally use (1,1,1) for the (xs,ys,zs) triple
 */
GLuint blockDisplayList(float xs, float ys, float zs, GLfloat color[4])
{
  GLuint index=glGenLists(1);

  glNewList(index,GL_COMPILE);
  glColor4fv(color);
  glMaterialfv(GL_FRONT,GL_AMBIENT_AND_DIFFUSE,color);

  glBegin(GL_QUADS);

    glNormal3f(-1,0,0);            // left
    glVertex3f(-xs, ys, zs);
    glVertex3f(-xs, ys,-zs);
    glVertex3f(-xs,-ys,-zs);
    glVertex3f(-xs,-ys, zs);

    glNormal3f( 1,0,0);            // right
    glVertex3f( xs, ys, zs);
    glVertex3f( xs,-ys, zs);
    glVertex3f( xs,-ys,-zs);
    glVertex3f( xs, ys,-zs);

    glNormal3f(0,0,1);             // front
    glVertex3f( xs, ys, zs);
    glVertex3f(-xs, ys, zs);
    glVertex3f(-xs,-ys, zs);
    glVertex3f( xs,-ys, zs);

    glNormal3f(0,0,-1);             // back
    glVertex3f( xs, ys,-zs);
    glVertex3f( xs,-ys,-zs);
    glVertex3f(-xs,-ys,-zs);
    glVertex3f(-xs, ys,-zs);

    glNormal3f(0,1,0);             // top
    glVertex3f( xs, ys, zs);
    glVertex3f( xs, ys,-zs);
    glVertex3f(-xs, ys,-zs);
    glVertex3f(-xs, ys, zs);

    glNormal3f(0,-1,0);            // bottom
    glVertex3f( xs,-ys, zs);
    glVertex3f(-xs,-ys, zs);
    glVertex3f(-xs,-ys,-zs);
    glVertex3f( xs,-ys,-zs);

  glEnd();

  glEndList();

  return(index);
}

/*
 * create display lists for axis lines
 */
void createAxisLists(void)
{
  GLfloat color[4] = {0, 0, 0, 1};

  xAxisList = glGenLists(1);
  glNewList(xAxisList, GL_COMPILE);
  color[0] = 1;
  glColor4fv(color);
  glMaterialfv(GL_FRONT,GL_AMBIENT_AND_DIFFUSE,color);
  color[0] = 0;
  glBegin(GL_LINES);
    glVertex3f(        0, 0, 0);
    glVertex3f(MAXX_CLIP, 0, 0);
  glEnd();
  glEndList();

  yAxisList = glGenLists(1);
  glNewList(yAxisList, GL_COMPILE);
  color[1] = 1;
  glColor4fv(color);
  glMaterialfv(GL_FRONT,GL_AMBIENT_AND_DIFFUSE,color);
  color[1] = 0;
  glBegin(GL_LINES);
    glVertex3f(0,         0, 0);
    glVertex3f(0, MAXX_CLIP, 0);
  glEnd();
  glEndList();

  zAxisList = glGenLists(1);
  glNewList(zAxisList, GL_COMPILE);
  color[2] = 1;
  glColor4fv(color);
  glMaterialfv(GL_FRONT,GL_AMBIENT_AND_DIFFUSE,color);
  color[2] = 0;
  glBegin(GL_LINES);
    glVertex3f(0, 0,    0);
    glVertex3f(0, 0, MAXZ);
  glEnd();
  glEndList();
}

/*
 * create display list for grid positions
 */
GLuint createPositionList(Position *p, long N, GLfloat color[4])
{
  long i;
  GLuint index;

  index = glGenLists(1);
  glNewList(index, GL_COMPILE);
    glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE,color);
    glColor4fv(color);
    glBegin(GL_POINTS);
      for(i=0; i<N; i++) {
	glVertex3f(p[i].x, p[i].y, p[i].z);
	}
    glEnd();
  glEndList();
  return(index);
}

/*
 * Create a colormap from the range of densities in the model
 */
Colorscale colorscale(GravityModel model)
{
  long i;
  double rho_step, Pmin, Pmax;
  double ar, ag, bg, ab;
  Colorscale c;

  /* find min/max density of model */
  Pmin = Pmax = 0.0;
  for(i=0; i<model.nelem; i++) {
    if(Pmax < model.elem[i].rho) {
      Pmax = model.elem[i].rho;
      }
    if(Pmin > model.elem[i].rho) {
      Pmin = model.elem[i].rho;
      }
    }

  printf("Max/Min density: %lg/%lg\n", Pmax, Pmin);

  rho_step = (Pmax - Pmin)/NCOLORS;

  ar =  1.0 / (NCOLORS*NCOLORS);
  ag = -4.0 / (NCOLORS*NCOLORS);
  bg =  4.0 / NCOLORS;
  ab = -1.0 / (NCOLORS*NCOLORS);

  for(i = 0; i<NCOLORS; i++) {
    c.r[i] = ar * i*i;
    c.g[i] = ag * i*i + bg * i;
    c.b[i] = ab * i*i + 1.0;
    c.p[i] = Pmin + rho_step * i;
    }

  return(c);
}

void usage(void)
{
  printf("openglvis [-f geom_file] [-p position_file] [-hv]\n");
}

void version(void)
{
  printf("OpenGLvis v%s\n", VERSION);
  printf("\t(c)2001 Paul Gettings\n");
  printf("\t        Dep't of Geology & Geophysics\n");
  printf("\t        University of Utah\n");
  printf("\tPart of the utah-g3d package; ");
  printf("released under the GPL\n");
}

/*
 * Main routine
 */
int main(int argc, char** argv) {
  char *opts = "f:p:hv";
  char opt;
  char *defaultFilename = "geometry";
  char buf[1025], pbuf[1025];

  // initialize vars
  rx = ry = rz = 0;

  // process OpenGL cmd-line args
  glutInit(&argc,argv);

  // process other args
  positionFlag = 0;
  strncpy(buf, defaultFilename, 1024);
  while((opt = getopt(argc, argv, opts)) != -1) {
    switch(opt) {
      case 'f':
        strncpy(buf, optarg, 1024);
        break;
      case 'p':
        strncpy(pbuf, optarg, 1024);
        positionFlag++;
        break;
      case '?':
      case 'h':
        usage();
        exit(0);
        break;
      case 'v':
        version();
        exit(0);
        break;
      }
    }

  model = loadModel(buf);
  printf("loaded model geometry from file '%s' (%ld blocks)\n", buf, model.nelem);

  Position *cgrid;
  long npoints; // # of compute grid points

  if(positionFlag) {
    cgrid = loadPositions(pbuf, &npoints);
    printf("loaded %ld grid positions from file '%s'\n", npoints, pbuf);
    }

  setClippingPlanes();

  if((block = (GLuint *)malloc(model.nelem*sizeof(GLuint))) == NULL) {
    fprintf(stderr, "cannot allocate memory for block[].\n");
    exit(1);
    }

  /*
   * OpenGL
   */
  // ask for color buffer, depth buffer, double buffered
  glutInitDisplayMode(GLUT_RGB | GLUT_DEPTH | GLUT_DOUBLE);
  // window in upper left corner
  glutInitWindowPosition(0,0);

  // Main window
  // window 500x500 pixels
  glutInitWindowSize(500,500);
  mainWin = glutCreateWindow("utah-g3d vis");

  // Vis info sub window
  subWin = glutCreateSubWindow(mainWin, 0, 0, 500, TEXT_HEIGHT);
  glutDisplayFunc(drawParms);
  glutReshapeFunc(reshapeParms);

  glutSetWindow(mainWin);

  static GLfloat dim[4]={0.5,0.5,0.5,0};
  // set a light source
  glLightfv(GL_LIGHT0,GL_AMBIENT,dim);
  // enable face culling, lighting, lights 0&1, etc.
  glEnable(GL_CULL_FACE);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glEnable(GL_LIGHT1);
  //glEnable(GL_BLEND);
  glEnable(GL_DEPTH_TEST); //depth testing - use that depth buffer
  glEnable(GL_NORMALIZE); //auto-normalize normal vectors

  // other OpenGL settings
  glDisable(GL_LINE_STIPPLE);//solid lines only
  glShadeModel(GL_FLAT);//fast, if ugly
  glPolygonMode(GL_FRONT, GL_FILL);//only fill front of poly
  glLineWidth(3.0);//thick axis lines

  // create display lists for blocks, axes, and grid pts
  GLfloat color[4] = {0.8, 0.8, 0.8, 1.0};
  GLuint b[NCOLORS];
  Colorscale c;
  long i;
  int j, k;
  double d, od;

  c = colorscale(model);
  for(i=0; i<NCOLORS; i++) {
    color[0] = c.r[i]; color[1] = c.g[i]; color[2] = c.b[i];
    b[i] = blockDisplayList(1, 1, 1, color);
    }
  for(i=0; i<model.nelem; i++) {
    // find the right color for the block
    k = 0; od = fabs(c.p[0] - model.elem[i].rho);
    for(j=0; j<NCOLORS; j++) {
      d = fabs(c.p[j] - model.elem[i].rho);
      if(d < od) {
	k = j; od = d;
	}
      }
    block[i] = b[k];
    }
  createAxisLists();
  if(positionFlag) {
    color[0] = 1; color[1] = 1; color[2] = 1;
    positionList = createPositionList(cgrid, npoints, color);
    }
  printf("X axis == RED\nY axis == GREEN\nZ axis == BLUE\n");


  // set callbacks
  glutDisplayFunc(draw);
  glutReshapeFunc(reshape);
  glutSpecialFunc(keyPress);
  glutKeyboardFunc(normalKeyPress);
  glutPassiveMotionFunc(passiveMouseMotion);
  glutMotionFunc(mouseMotion);
  glutMouseFunc(mouseEvent);
  //glutIdleFunc(idle);

  glutMainLoop();
}

Position *loadPositions(char *file, long *np)
{
  long i;
  char *rec;
  FILE *fp;
  Position *p;

  if((fp = fopen(file, "rt")) == NULL) {
    fprintf(stderr, "loadPositions: cannot open %s for read.\n", file);
    exit(1);
    }

  i=0;
  while(!feof(fp)) {
    if(getrecord(fp) != NULL)
      i++;
    }

  rewind(fp);
  if((p = (Position *)malloc((i+1)*sizeof(Position))) == NULL) {
    fprintf(stderr, "loadPositions: cannot allocate memory for %ld positions\n", i);
    exit(1);
    }
  i = 0;
  while(!feof(fp)) {
    if((rec = getrecord(fp)) != NULL) {
      sscanf(rec, "%lf %lf %lf", &p[i].x, &p[i].y, &p[i].z);
      i++;
      }
    }
  *np = i;
  return(p);
}

/*
 * Functions we don't currently use, but might want later
 */
/*
void rotatexz(float *x, float *y, float *z, float theta, float phi)
{
  float X, Y, Z;
  GLfloat ct, st, cp, sp;

  // rotate (x,y,z) to new position
  ct = cos(theta); st = sin(theta);
  cp = cos(phi); sp = sin(phi);
  // compound rotation - first about z, then about x
  X = (*x)*ct + (*y)*st;
  Y = (*y)*ct - (*x)*st;
  Z = (*z);
  *x = X;
  *y = Y*cp + Z*sp;
  *z = Z*cp - Y*sp;
}

void rotatexy(float *x, float *y, float *z, float theta, float phi)
{
  float X, Y, Z;
  GLfloat ct, st, cp, sp;

  // rotate (x,y,z) to new position
  ct = cos(theta); st = sin(theta);
  cp = cos(phi); sp = sin(phi);
  // compound rotation - first about y, then about x
  X = (*x)*ct + (*z)*st;
  Z = (*z)*ct - (*x)*st;
  Y = (*y);
  *x = X;
  *y = Y*cp + Z*sp;
  *z = Z*cp - Y*sp;
}

*/
