/*
 * newton.cpp
 *
 * Gravity inversion using Newton's method
 *
 * This program solves the 3d gravity inverse using a regularized
 * Newton's method.
 * The Frechet derivative matrix is computed analytically (see
 * frechet()).  This method converges rapidly, but should be
 * regularized.  The method tends to fit the data with implausible
 * models if left alone; regularization and excellent starting models
 * help maintain plausibility in the output.
 *
 * This code implements Tikhonov regularization, which depends on a
 * "regularization parameter", alpha, that is normally set in an
 * empirical manner.  alpha can be considered the strength of
 * regularization, and determines how closely the model is allowed to
 * stray from the a priori model guess.
 *
 * To maintain consistent scale between alpha and the standard error
 * misfit, it is often necessary to use very small alpha values, on the
 * order of 1e-16.  To facilitate this, an input alpha value is scaled
 * by 1e-15.
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000, 2001 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include "model.h"
#include "matrix.h"

static char *VERSION="1.2.0";

int frechet(Matrix F, GravityModel m, Position p[])
{
  unsigned long i, j;
  static double r;

  for(i=0; i<F.m; i++) {
    for(j=0; j<F.n; j++) {
      r  = sqrt(
           (p[i].z - m.elem[j].z)*(p[i].z - m.elem[j].z) +
           (p[i].y - m.elem[j].y)*(p[i].y - m.elem[j].y) +
           (p[i].x - m.elem[j].x)*(p[i].x - m.elem[j].x)
            );
      // F = A*dz/r^3; A = G*xlen*ylen*zlen
      F.array[i][j]  = GAMMA*m.elem[j].xlen*m.elem[j].ylen*m.elem[j].zlen;
      F.array[i][j] *= (p[i].z - m.elem[j].z)/(r*r*r);
      }
    }
  return(1);
}

double error(GravityModel m, Position *p, double *d, unsigned long n)
{
  double E=0;
  unsigned long i;

  for(i=0; i<n; i++) {
    E += fabs(gravity(m, p[i]) - d[i]);
    }
  return(E);
}

void usage(void)
{
  fprintf(stderr, "usage: newton [options] <a priori model> <data> <output file> <alpha>\n");
  fprintf(stderr, "where [options] are:\n");
  fprintf(stderr, "   -e dE threshhold       set error change convergence criterion\n");
  fprintf(stderr, "   -i max iterations      set maximum # of iterations\n");
}

int main(int argc, char *argv[])
{
  GravityModel m, map;
  double *d;
  Position *p;
  unsigned long M, N, np;
  long i=0, j, k, maxiter;
  Matrix H, F, FT, I;
  double *L, *dm;
  double e, ep, delta;

  char *optstr="e:i:hH";
  int opt;

  double alpha;

  // check command line
  if(argc < 5) {
    usage();
    exit(1);
    }

  // set threshholds, etc.
  delta = 1e-8;
  maxiter = 10;

  while((opt = getopt(argc, argv, optstr)) != -1) {
    switch(opt) {
      case 'e': delta = atof(optarg); break;
      case 'i': maxiter = atoi(optarg); break;
      case 'h':
      case 'H': usage(); exit(1); break;
      default: usage(); exit(1); break;
      }
    }

  // load gravity model - a priori model
  m = loadModel(argv[optind]);
  fprintf(stderr, "loaded a priori model geometry; %ld blocks\n", m.nelem);
  N = m.nelem;
  if(N < 1) {
    exit(1);
    }
  //   copy initial model to backup
  map.nelem = -1;
  copyModel(m, &map); // copyModel allocates a buffer, etc. if
		      // map.nelem < 1

  // load data, allocate memory for arrays
  FILE *fp;
  char *record;

  if((fp = fopen(argv[optind+1], "rt")) == NULL) {
    fprintf(stderr, "cannot open data file.\n");
    exit(1);
    }
  //   count number of observations
  i=0;
  while(!feof(fp)) {
    record = getrecord(fp);
    if(record == NULL) continue;
    i++;
    }
  rewind(fp);
  M = np = i;
  if(M < 1) {
    fprintf(stderr, "# of observation points < 1.\n");
    exit(1);
    }
  fprintf(stderr, "loading %ld observations...", M);
  //   allocate matrices
  F = newMatrix(M, N);
  FT = newMatrix(N, M);
  if(F.array == NULL) {
    fprintf(stderr, "cannot allocate space for Frechet matrix.\n");
    exit(1);
    }
  if(FT.array == NULL) {
    fprintf(stderr, "cannot allocate space for transposed Frechet matrix.\n");
    exit(1);
    }
  H = newMatrix(N, N);
  if(H.array == NULL) {
    fprintf(stderr, "cannot allocate space for pseudo-Hessian.\n");
    exit(1);
    }
  if((L = allocateVector(N)) == NULL) {
    fprintf(stderr, "cannot allocate space for L vector.\n");
    exit(1);
    }
  if((dm = allocateVector(N)) == NULL) {
    fprintf(stderr, "cannot allocate space for dm vector.\n");
    exit(1);
    }
  if((d = allocateVector(M)) == NULL) {
    fprintf(stderr, "cannot allocate space for data vector.\n");
    exit(1);
    }
  if((p = (Position *)malloc(sizeof(Position)*M)) == NULL) {
    fprintf(stderr, "cannot allocate space for position vector.\n");
    exit(1);
    }
  //  get observation records
  i=0;
  while(!feof(fp)) {
    record = getrecord(fp);
    if(record == NULL) continue;
    sscanf(record, "%lf %lf %lf %lf\n", &p[i].x, &p[i].y, &p[i].z, &d[i]);
    d[i] /= 1e5; // convert from mGal to m/s/s
    i++;
    if(i>=np) break;
    }
  fclose(fp);
  fprintf(stderr, "done.\n");

  // set alpha
  fprintf(stderr, "setting and scaling regularization parameter, alpha.\n");
  alpha = atof(argv[optind+3]);
  alpha *= 1e-15;

  // compute starting error
  e = error(m, p, d, np);
  ep = e;

  /* NOTE!!!!
   * Frechet derivative of our gravity operator does not depend on the density
   * distribution; only on the block and observation positions!  Since these
   * are invariant in this inversion scheme, we can compute F, F' and also H
   * exactly once for the entire process.  This should greatly speed the
   * iteration process.
   */
  // compute F, F'
  frechet(F, m, p);
  transpose(F, FT);

  // compute H = (F'*F + alpha*I)^-1 with alpha as the regularization parameter
  Matrix h;
  h = newMatrix(N, N);
  if(h.array == NULL) {
    fprintf(stderr, "cannot allocate space for pseudo-Hessian matrix.\n");
    exit(1);
    }
  mm(FT, F, h);
  I=eye(h.m); // regularization bits
  mscale(I, alpha);
  ma(h, I, h);
  if(!inv(h, H)) {
    fprintf(stderr, "error computing pseudo-Hessian inverse\n");
    exit(1);
    }

  // start iteration
  fprintf(stderr, "starting iterative inversion\n");
  fprintf(stderr, "max iterations = %ld\n", maxiter);
  fprintf(stderr, "dE  threshhold = %lg\n", delta);
  fprintf(stderr, "regularization parameter = %lg\n", alpha);
  fprintf(stderr, " Iter #    Current Error    Error Change\n");
  fprintf(stderr, "-------- ---------------- ----------------\n");
  i = 1;
  while(i < maxiter) {
    fprintf(stderr, "%8ld %16lg %16lg\n", i, e, ep-e);
    // compute L = F'*(A(m) - d) + a*|m - map|
    for(j=0; j<N; j++) {
      L[j] = 0;
      for(k=0; k<M; k++) {
	L[j] += FT.array[j][k]*(gravity(m, p[k]) - d[k]);
	}
      L[j] += alpha*(m.elem[i].rho - map.elem[i].rho);
      }

    // compute dm = H*L
    for(j=0; j<N; j++) {
      dm[j] = 0;
      for(k=0; k<N; k++) {
        dm[j] += H.array[j][k]*L[k];
        }
      }

    // compute m' = m + dm
    for(j=0; j<N; j++) {
      m.elem[j].rho += dm[j];
      }

    // compute e' = |A(m') - d|
    ep = error(m, p, d, N);

    // |e' - e| < delta? yes ==> stop
    if(fabs(ep - e) < delta) break;
    // e' > e? yes ==> stop
    if( (ep-e) > 0) {
      for(j=0; j<N; j++) {
	m.elem[j].rho -= dm[j];
	}
      break;
      }

    // next iteration
    e = ep;
    i++;
    }

  // output final model, error
  fprintf(stderr, "finished.\n");
  fprintf(stderr, "# iterations: %d\n", i);
  fprintf(stderr, "final error: %lg\n", e);
  fprintf(stderr, "writing final model to %s\n", argv[optind+2]);
  saveModel(argv[optind+2], m);

  exit(0);
}
