/*
 * model.h
 *
 * header file for computation engine; include this in applications
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */
#ifndef GRAVITYMODEL
#define GRAVITYMODEL

#include <stdio.h>
#include <math.h>

/*
 * Constants
 */
#define GAMMA	6.671e-11 /* constant of gravitation, in m and kg */

/*
 * Data Types
 */
/* Structure that defines a model element
 * Contains:
 *   x, y, z		(x,y,z) position of center of element
 *   rho		density, normally in kg/m^3
 *   xlen, ylen, zlen	length of element in each axis
 * Note that using this data, it is possible to compute:
 *   Volume = xlen*ylen*zlen
 *   Mass = Volume*rho = xlen*ylen*zlen*rho
 *
 * The elements are defined this way for a simple reason - this allows
 * for a model to only need to track those elements that have anomalous
 * density - all the information for each element is in the struct.
 * Moreover, elements need not be equal, making it possible to scale
 * elements to larger sizes at greater depth, etc.  This should allow
 * for fewer elements in a model than if all were the same size; hence,
 * overall using less memory and CPU cycles.
 *
 * If have lots of RAM and little CPU cycles, then add a volume and/or
 * mass entry to the struct and compute these only when element
 * attributes change.  Otherwise, trade the space for cycles to compute
 * to temporary storage when computing gravity effect.
 */
typedef struct ModelElementStruct {
  double x;
  double y;
  double z;
  double rho;
  double xlen;
  double ylen;
  double zlen;
  } Element;

/* Structure that defines a whole model
 * Contains:
 *   Element* elem	Pointer to an array of elements; each entry is
 *   			a struct of type Element
 *   nelem		Number of x, y, and z elements.  The array of
 *			elements (elem) should have nelem entries.
 *			This is a long int, so only guarantee 4 billion
 *			addressable entries. :)
 */
typedef struct GravityModelStruct {
  Element *elem;
  long nelem;
  } GravityModel;


/* Structure defining a 3d position
 * Makes it easier to pass around positions where to compute gravity
 * anomalies.
 */
typedef struct PositionStruct {
  double x;
  double y;
  double z;
  } Position;

/*
 * Function Prototypes
 */
GravityModel loadModel(char *filename);
int saveModel(char *filename, GravityModel model);
double gravity(GravityModel model, Position pos);
char *getrecord(FILE *fptr);
void copyElement(Element src, Element *dst);
int copyModel(GravityModel src, GravityModel *dst);

#endif
