/*
 * model.c
 *
 * computation engine; start here for building applications
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */

/* Be sure to link with the math library - the code uses pow()! */

#include <stdlib.h>
#include "model.h"

/*
 * Prototypes for internal functions
 */
double _block(double x, double y, double z,
  double a, double b, double c, double rho);

/* Read a model geometry file.
 * Bails to the system on error!
 *
 * File structure is very simple, and straight ASCII:
 *	nelem
 *	elements(x y z rho xlen ylen zlen))
 *    There is no required order, since every element has all its data
 *    inherently!
 *
 *    Empty lines and those beginning with # are ignored.
 */
GravityModel loadModel(char *filename) {
  FILE *fptr;
  char *record;
  long i;
  GravityModel model;
  if((fptr = fopen(filename, "rt")) == NULL) {
   fprintf(stderr, "Cannot open model file '%s' for read.\n", filename);
   exit(1);
   }
  
  /* read the first line */
  while( (record = getrecord(fptr)) == NULL && !feof(fptr)); /* do nothing */
  if(feof(fptr)) {
    fprintf(stderr, "EOF reached before expected; empty file?\n");
    exit(1);
    }
  sscanf(record, "%ld", &model.nelem);

  /* create and allocate memory */
  model.elem = (Element *)malloc(model.nelem*sizeof(Element));
  if(model.elem == NULL) {
    fprintf(stderr, "Cannot allocate memory for %ld elements\n", model.nelem);
    exit(1);
    }
  
  /* read the rest of the file into the element array */
  for(i=0; i<model.nelem; i++) {
    record = getrecord(fptr);
    if(record == NULL) {
      i--;
      }
    else {
      sscanf(record, "%lg %lg %lg %lg %lg %lg %lg",
	     &model.elem[i].x, &model.elem[i].y, &model.elem[i].z,
	     &model.elem[i].rho,
	     &model.elem[i].xlen, &model.elem[i].ylen, &model.elem[i].zlen);
      }
    if(feof(fptr)) {
      fprintf(stderr, "EOF reached before %ld elements read; truncated file?\n",
	model.nelem);
      exit(1);
      }
    }
  fclose(fptr);
  /* return model struct */
  return(model);
}

/* Write gravity model to file.
 * Returns 0 on failure, 1 on success.
 */
int saveModel(char *filename, GravityModel model) {
  FILE *fptr;
  long i;
  if((fptr = fopen(filename, "wt")) == NULL) {
   printf("Cannot open file '%s' for write.\n", filename);
   return(0);
   }
  
  /* write the first line */
  fprintf(fptr, "%ld\n", model.nelem);

  /* write the rest of the model */
  for(i=0; i<model.nelem; i++) {
    fprintf(fptr, "%lf %lf %lf %lf %lf %lf %lf\n",
	    model.elem[i].x, model.elem[i].y, model.elem[i].z,
	    model.elem[i].rho,
	    model.elem[i].xlen, model.elem[i].ylen, model.elem[i].zlen);
    }
  fclose(fptr);
  /* return success */
  return(1);
}

/* Compute gravity effect of a single model element block.
 * Uses the "moment method" that expands the gravitational potential in
 * terms of Legendre polynomials, truncates the series at l=2, sets the
 * element center of mass (COM) at the origin, and then takes the
 * derivative of the potential with respect to z to find the vertical
 * gravitational force.  This has been specialized to a rectangular
 * block with side lengths 2a, 2b, and 2c.
 *
 * Note that for a cube, B02 and B22 are 0 ==> can treat a cube like a
 * sphere and compress the mass to the COM and use Newton's law!
 *
 * Taken from:  Interpretation Theory in Applied Geophysics, by Grant &
 * West. 1965 - pages 222-227
 */
double _block(double x, double y, double z,
  double a, double b, double c, double rho)
{
  double B00, B02, B22;
  double r, r2, r3, r5, r7;
  double F;

  /*
     This computation wants z positive up, but
     rest of model system has z positive down!
  */
  z = -z;

  r  = sqrt(x*x + y*y + z*z);
  if(r == 0) return(0.0); /* observation point at COM of element
  			     so ignore this element */
  r2 = r*r;
  r3 = r2*r;
  r5 = r3*r2;
  r7 = r5*r2;

  B00 = 8*GAMMA*rho*a*b*c;
  B02 = B00 * (2*c*c - a*a - b*b) / 6.0;
  B22 = B00 * (a*a - b*b) / 24.0;

  F  = B00*z / r3;
  F += ( B02/(2*r5) ) * ( (5*z*(3*z*z-r2)/r2) - 4*z );
  F += B22*15*z*(x*x-y*y) / r7;
  return F;
}

/* Compute vertical gravity effect at position "pos" due to gravity model
 * "model".
 */
double gravity(GravityModel model, Position pos)
{
  /* compute gravity effect at position pos due to gravity model */
  long i;
  double dx, dy, dz, m, r;
  double dg;

  dg = 0.0;

  for(i=0; i<model.nelem; i++) {
    /*m = model.elem[i].xlen*model.elem[i].ylen*model.elem[i].zlen*model.elem[i].rho;
    */
    dx = pos.x - model.elem[i].x;
    dy = pos.y - model.elem[i].y;
    dz = pos.z - model.elem[i].z;
    /* r = dx*dx + dy*dy + dz*dz; */ /* actually, this is r^2 */
    /* if(r == 0) continue; /* r==0 ==> pos is at COM of the block, so neglect it */
    /*dg += (GAMMA * m * dz) / pow(r, 1.5);*/
    dg += _block(dx, dy, dz, model.elem[i].xlen/2, model.elem[i].ylen/2,
      model.elem[i].zlen/2, model.elem[i].rho);
    }
  return(dg);
}

/* Get a single record from a file.
 * Returns NULL on EOF, or for comment/blank lines.
 */
char *getrecord(FILE *fptr) {
  static char record[1025];
  char str[1025];
  int i;

  fgets(record, 1025, fptr);
  if(feof(fptr)) {
    return(NULL);
    }
  i = sscanf(record, "%1024s", str);

  if(i == EOF) {
    return(NULL); /* empty line/EOF */
    }
  if(str[0] == '#') {
    return(NULL); /* comment */
    }
  return(record);
}

/* Copy a model Element from src to dst
 */
void copyElement(Element src, Element *dst)
{
  dst->x    = src.x;
  dst->y    = src.y;
  dst->z    = src.z;
  dst->xlen = src.xlen;
  dst->ylen = src.ylen;
  dst->zlen = src.zlen;
  dst->rho  = src.rho;
}

/* Copy model in src to dst.
 * If dst does not have a suitable # of elements, a new element array
 * is allocated.
 * Returns 0 if memory unavailable
 * Returns 1 on success
 */
int copyModel(GravityModel src, GravityModel *dst)
{
  long i;

  if(dst->nelem != src.nelem) {
    if(dst->nelem > 0) /* assume an existing model */
      free(dst->elem);
    dst->elem = (Element *)malloc(src.nelem*sizeof(Element));
    if(dst->elem == NULL)
      return(0);
    }
  dst->nelem = src.nelem;
  for(i=0; i<src.nelem; i++) {
    copyElement(src.elem[i], &dst->elem[i]);
    }
  return(1);
}
