/*
 * matrix.c
 *
 * Utility functions for dealing with matrix math.  This file contains
 * some functions to do simple matrix operations.  They are used in the
 * inversion codes.  This is far from a complete L.A. package.  It is
 * unoptimized, and probably less-than-perfectly implemented.
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000, 2001 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include "matrix.h"

#define max(a, b)	( (a>b) ? a : b )
#define min(a, b)	( (a<b) ? a : b )

double sign(double a, double b)
{
  int s;
  s = ( (b >= 0) ? 1 : -1 );
  return(fabs(a)*s);
}

/* make a Matrix from nothing */
Matrix newMatrix(unsigned long m, unsigned long n)
{
  Matrix M;
  double **a;

  if(m == 0 || n == 0) {
    return(makeMatrix(NULL, 0, 0));
    }

  a = allocateArray(m, n);
  return(makeMatrix(a, m, n));
}

/* free a matrix; make it unusable */
void delMatrix(Matrix M)
{
  free(M.array);
  M.array = NULL;
  M.m = M.n = 0;
}

/* make a Matrix from a double[][] */
Matrix makeMatrix(double **a, unsigned long m, unsigned long n)
{
  Matrix M;

  if(a == NULL) {
    /* empty Matrix */
    M.m = 0;
    M.n = 0;
    M.array = NULL;
    return(M);
    }

  M.m = m;
  M.n = n;
  M.array = a;
  return(M);
}

double *allocateVector(unsigned long l)
{
  return((double *)malloc(sizeof(double)*l));
}

/* allocate 2d array
 * returns NULL on error; frees allocated memory if error occurs
 * partway through
 */
double **allocateArray(unsigned long m, unsigned long n)
{
  double **a;
  unsigned long i, j;

  if((a = (double **)malloc(sizeof(double *)*m)) == NULL)
    return(a);
  for(i=0; i<m; i++) {
    if((a[i] = (double *)malloc(sizeof(double)*n)) == NULL) {
      for(j=0; j<i; j++) {
        free(a[j]);
        }
      free(a);
      return(NULL);
      }
    }
  return(a);
}

/* copy 2d array from src to dst
 * arrays must be of suitable size before calling copyArray()!
 */
int copyArray(double **src, double **dst, unsigned long m, unsigned long n)
{
  unsigned long i, j;

  for(i=0; i<m; i++)
    for(j=0; j<n; j++)
      dst[i][j] = src[i][j];

  return(0);
}

/* copy Matrix to Matrix
 */
int copyMatrix(Matrix src, Matrix dst)
{
  unsigned long i, j;

  /* check sizes */
  if(src.m != dst.m) return(-1);
  if(src.n != dst.n) return(-1);

  /* copy */
  for(i=0; i<src.m; i++)
    for(j=0; j<src.n; j++)
      dst.array[i][j] = src.array[i][j];

  return(0);
}

/* Clear the array of a Matrix */
int clearMatrix(Matrix M)
{
  unsigned long i, j;

  for(i=0; i<M.m; i++)
    for(j=0; j<M.n; j++)
      M.array[i][j] = 0;
  return(0);
}

/* Create an identity matrix */
Matrix eye(unsigned long m)
{
  Matrix a;
  unsigned long i;

  a = newMatrix(m, m);
  if(a.array == NULL) return(a);

  clearMatrix(a);

  for(i=0; i<m; i++)
    a.array[i][i] = 1.0;
  return(a);
}

/* Print the size and array of a matrix */
int printMatrix(Matrix m)
{
  printf("[%dx%d]\n", m.m, m.n);
  return(printArray(m.array, m.m, m.n));
}

/* Print a double ** array */
int printArray(double **a, unsigned long m, unsigned long n)
{
  unsigned long i, j;

  for(i=0; i<m; i++) {
    for(j=0; j<n; j++) {
      printf("%15lg ", a[i][j]);
      }
    printf("\n");
    }
  return(0);
}

/* scale matrix elements by scalar */
int mscale(Matrix a, double scale)
{
  unsigned long i, j;

  for(i=0; i<a.m; i++) {
    for(j=0; j<a.n; j++) {
      a.array[i][j] *= scale;
      }
    }
  return(0);
}

/* Scale a single row of a matrix */
int rscale(Matrix a, unsigned long row, double scale) 
{
  unsigned long i;

  if(row > a.m) return(-1);
  if(row < 0) return(-1);

  for(i=0; i<a.n; i++)
    a.array[row][i] *= scale;

  return(0);
}

/* Scale a single column of a matrix */
int cscale(Matrix a, unsigned long col, double scale) 
{
  unsigned long i;

  if(col > a.n) return(-1);
  if(col < 0) return(-1);

  for(i=0; i<a.m; i++)
    a.array[i][col] *= scale;

  return(0);
}

/* matrix multiply */
/* compute c=a*b */
int mm(Matrix a, Matrix b, Matrix c)
{
  unsigned long i, j, k;

  /* check sizes */
  if(a.n != b.m) return(-1);
  if(c.m != a.m) return(-1);
  if(c.n != b.n) return(-1);

  for(i=0; i<a.m; i++) {
    for(j=0; j<b.n; j++) {
      c.array[i][j] = 0;
      for(k=0; k<a.n; k++) {
        c.array[i][j] += a.array[i][k]*b.array[k][j];
        }
      }
    }
  return(1);
}

/* matrix add */
int ma(Matrix a, Matrix b, Matrix c)
{
  unsigned long i, j;

  /* check sizes */
  if(a.m != b.m) return(-1);
  if(a.n != b.n) return(-1);
  if(c.m != a.m) return(-1);
  if(c.n != a.n) return(-1);

  for(i=0; i<a.m; i++) {
    for(j=0; j<b.n; j++) {
      c.array[i][j] = a.array[i][j]+b.array[i][j];
      }
    }
  return(1);
}

/* matrix subtract */
int ms(Matrix a, Matrix b, Matrix c)
{
  unsigned long i, j;

  /* check sizes */
  if(a.m != b.m) return(-1);
  if(a.n != b.n) return(-1);
  if(c.m != a.m) return(-1);
  if(c.n != a.n) return(-1);

  for(i=0; i<a.m; i++) {
    for(j=0; j<b.n; j++) {
      c.array[i][j] = a.array[i][j]-b.array[i][j];
      }
    }
  return(1);
}

int transpose(Matrix a, Matrix aT)
{
  unsigned int i, j;

  if(a.m != aT.n) return(-1);
  if(a.n != aT.m) return(-1);

  for(i=0; i<a.m; i++) {
    for(j=0; j<a.n; j++) {
      aT.array[j][i] = a.array[i][j];
      }
    }
  return(1);
}



int ludecomp(Matrix M, int *indx)
{
  long i, imax, j, k;
  double big, dum, sum, temp;
  double *vv;
  double **a;
  int d;

  if(M.n == 0) return(0);
  if(M.array == NULL) return(0);

  a = M.array;

  if((vv = allocateVector(M.n)) == NULL) return(0);

  d = 1;
  for(i=0; i<M.n; i++) {
    big = 0;
    for(j=0; j<M.n; j++) {
      if((temp = fabs(a[i][j])) > big) big = temp;
      }
    if(big == 0.0) return(0);
    vv[i] = 1.0/big;
    }
  for(j=0; j<M.n; j++) {
    for(i=0; i<j; i++) {
      sum = a[i][j];
      for(k=0; k<i; k++) sum -= a[i][k]*a[k][j];
      a[i][j] = sum;
      }
    big = 0.0;
    for(i=j; i<M.n; i++) {
      sum = a[i][j];
      for(k=0; k<j; k++) {
        sum -= a[i][k]*a[k][j];
        }
      a[i][j] = sum;
      if((dum=vv[i]*fabs(sum)) >= big) {
        big = dum;
        imax = i;
        }
      }
    if(j != imax) {
      for(k=0; k<M.n; k++) {
        dum = a[imax][k];
        a[imax][k] = a[j][k];
        a[j][k] = dum;
        }
      d = -d;
      vv[imax] = vv[j];
      }
    indx[j] = imax;
    if(a[j][j] == 0) return(0); /* singular matrix */
    if(j < M.n-1) {
      dum = 1.0/a[j][j];
      for(i=j+1; i<M.n; i++) a[i][j] *= dum;
      }
    }
  free(vv);
  return(d);
}

int lubacksub(Matrix M, int *indx, double *b)
{
  long i, ii=-1, ip, j; /* original code from Num. Recipes sets ii=0,
  			   but i=0 is a valid index.  Set to -1 and
  			   change test below */
  double sum;
  double **a;

  if(M.n == 0) return(0);
  if(M.array == NULL) return(0);

  a = M.array;

  for(i=0; i<M.n; i++) {
    ip = indx[i];
    sum = b[ip];
    b[ip] = b[i];
    if(ii != -1) /* find a non-vanishing element yet? */
      for(j=ii; j<=i-1; j++)
	sum -= a[i][j]*b[j];
    else if(sum)
      ii = i;
    b[i] = sum;
    }
  for(i=M.n-1; i>=0; i--) {
    sum = b[i];
    for(j=i+1; j<M.n; j++)
      sum -= a[i][j]*b[j];
    b[i] = sum/a[i][i];
    }
  return(1);
}

int inv(Matrix M, Matrix Mi)
{
  Matrix A;
  double *col;
  long i, j;
  int *indx;

  if(M.m != M.n) return(0);
  if(Mi.m != M.m || Mi.n != M.m) return(0);

  if((indx = (int *)malloc(sizeof(int)*M.m)) == NULL) return(0);
  if((col = (double *)malloc(sizeof(double)*M.m)) == NULL) return(0);
  A = newMatrix(M.m, M.n);
  if(A.m == 0 || A.array == NULL) return(0);

  copyMatrix(M, A);

  ludecomp(A, indx);
  for(j=0; j<M.n; j++) {
    for(i=0; i<M.n; i++) col[i] = 0.0;
    col[j] = 1;
    lubacksub(A, indx, col);
    for(i=0; i<M.n; i++) Mi.array[i][j] = col[i];
    }
  return(1);
}

