#!/usr/bin/python
#
# generate_model.py
#
# Take a text description file and parse it into a bunch of blocks.
#
# This file is part of utah-g3d.
#
# utah-g3d is copyright (c) 2000 by
# Paul Gettings,
# Department of Geology & Geophysics,
# University of Utah.
#
# All Rights Reserved.
# 
# utah-g3d is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License, version 2, as
# published by the Free Software Foundation.
# 
# utah-g3d is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, in the file
# COPYING.  If that file is not present, write to the Free
# Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
# 02111-1307, USA

version_string = "*              v1.0.1                 *"

import string, sys, math;

## Model element generator name class
class NameElement:
  def __init__(self, name):
    self.name = name;

## Model element class
class Element:
  def __init__(self, X, Y, Z, P, Xl, Yl, Zl):
    self.x = X
    self.y = Y
    self.z = Z
    self.rho = P
    self.xlen = Xl
    self.ylen = Yl
    self.zlen = Zl

def main():
  global version_string
  ## Parse command line options
  if len(sys.argv) < 3:
    print "usage: %s description_file output_file"%sys.argv[0]
    sys.exit(0);

  ## Hello banner
  print "**************************************"
  print "*      Model Geometry Generator      *"
  print version_string
  print "*            Paul Gettings           *"
  print "* Department of Geology & Geophysics *"
  print "*        University of Utah          *"
  print "**************************************"

  infilename = sys.argv[1];
  outfilename= sys.argv[2];

  infile = open(infilename, "rt");

  ## setup defaults, in case the infile doesn't set them

  # start with 1x1x1 m cubes...
  xlen=1.0; ylen=1.0; zlen=1.0


  elems = []
  print "Parsing description file '%s':"%infilename
  ## Parse the input file into an array of blocks
  nline = 0;
  while(1):
    line = infile.readline();
    if not line: break;
    nline = nline+1;
    line = string.strip(line);
    if not line: continue;
    if line[0] == '#': continue;

    fields = string.split(line);

    # what to do?
    if string.upper(fields[0]) == "SPHERE":
      # sphere of blocks
      # SPHERE x y z r rho
      sys.stdout.write("  processing SPHERE: ")
      if len(fields)<6:
        print "\n** Incorrect SPHERE directive at line number %d **"%nline
        continue
      x=float(fields[1])
      y=float(fields[2])
      z=float(fields[3])
      r=float(fields[4])
      p=float(fields[5])
      elems.append(NameElement("SPHERE"));
      sys.stdout.write("(%f, %f, %f)/%f/%f\n"%(x, y, z, r, p))
      blockify_sphere(x, y, z, r, p, xlen, ylen, zlen, elems)

    elif string.upper(fields[0]) == "HDISC":
      # horizontal disc of blocks
      # HDISC x y z r h rho
      sys.stdout.write("  processing horizontal DISC: ")
      if len(fields)<7:
        print "\n** Incorrect HDISC directive at line number %d **"%nline
        continue
      x=float(fields[1])
      y=float(fields[2])
      z=float(fields[3])
      r=float(fields[4])
      h=float(fields[5])
      p=float(fields[6])
      elems.append(NameElement("HDISC"));
      sys.stdout.write("(%f, %f, %f)/%f/%f/%f\n"%(x, y, z, r, h, p))
      blockify_hdisc(x, y, z, r, h, p, xlen, ylen, zlen, elems)

    elif string.upper(fields[0]) == "BLOCK":
      # rectangular solid
      # BLOCK x y z rho xsize ysize zsize
      sys.stdout.write("  processing BLOCK: ")
      if len(fields)<8:
        print "\n** Incorrect BLOCK directive at line number %d **"%nline
        continue
      x=float(fields[1])
      y=float(fields[2])
      z=float(fields[3])
      p=float(fields[4])
      xsize=float(fields[5])
      ysize=float(fields[6])
      zsize=float(fields[7])
      elems.append(NameElement("BLOCK"));
      sys.stdout.write("(%f, %f, %f)/%f/%fx%fx%f\n"%(x, y, z, p, xsize, ysize, zsize))
      blockify_block(x, y, z, xsize, ysize, zsize, p, xlen, ylen, zlen, elems)

    elif string.upper(fields[0]) == "BLOCKSIZE":
      # change dimensions of blocks
      # BLOCKSIZE xlen ylen zlen
      sys.stdout.write("  processing BLOCKSIZE: ")
      if len(fields)<4:
        print "\n** Incorrect BLOCKSIZE directive at line number %d **"%nline
        continue
      xlen=float(fields[1])
      ylen=float(fields[2])
      zlen=float(fields[3])
      sys.stdout.write("%f %f %f\n"%(xlen, ylen, zlen))

    else:
      # ignore it; print warning
      print "Unknown directive '%s' in description file."%fields[0]

  nelem = 0L
  for e in elems:
    if isinstance(e, Element):
      nelem = nelem+1

  print "Resolving duplicate elements (checking %d elements)"%nelem
  ## Find duplicate blocks, and remove them
  duplicate = []
  interval = int(round(nelem/100.0));
  sys.stdout.write("  checking element #");
  sys.stdout.flush()
  n=0; cubes = {}
  for i in xrange(len(elems)):
    if isinstance(elems[i], Element):
      n = n+1
      if (n%interval) == 0 or n == 1:
	sys.stdout.write("%8d\b\b\b\b\b\b\b\b"%n)
	sys.stdout.flush()
      key = "%g,%g,%g,%g,%g,%g"%(elems[i].x, elems[i].y, elems[i].z, elems[i].xlen, elems[i].ylen, elems[i].zlen);
      if cubes.has_key(key):
        #print "Adding key %s"%key
        duplicate.append(cubes[key]);
        cubes[key] = i
      else:
        cubes[key] = i
  sys.stdout.write("%8d\b\b\b\b\b\b\b\b"%n)
  print ""
  duplicate.sort()
  duplicate.reverse()
  for i in duplicate:
    del elems[i]

  nelem = 0L
  for e in elems:
    if isinstance(e, Element):
      nelem = nelem+1

  ## write out model geometry file
  print "Writing output file '%s'"%outfilename
  outfile = open(outfilename, "wt");
  outfile.write("# Model geometry file generated from %s\n"%infilename);
  outfile.write("# number of model elements:\n");
  outfile.write("%d\n"%nelem);
  outfile.write("# Model element definitions\n");
  for e in elems:
    if isinstance(e, NameElement):
      outfile.write("# %s\n"%e.name);
    else:
      outfile.write("%10g %10g %10g %10g %10g %10g %10g\n"%\
      		(e.x, e.y, e.z, e.rho, e.xlen, e.ylen, e.zlen));

  outfile.close();
  infile.close();
  print "done."


def blockify_sphere(x, y, z, r, p, xlen, ylen, zlen, elems):
  ## create all the blocks inside a sphere
  # start from the center @ (x,y,z), and determine how many blocks will
  # fit in each axis direction.  Then, iterate over all the
  # possibilities, testing if R=sqrt(dx^2+dy^2+dz^2)<r.  If so, then
  # add the block.
  nxblock = int(round(r/xlen));
  nyblock = int(round(r/ylen));
  nzblock = int(round(r/zlen));
  total_elem = (2L*nxblock+1)*(2L*nyblock+1)*(2L*nzblock+1);

  print "    Sphere bounding box = %dx%dx%d elements (%g total)"%(2*nxblock+1,
    2*nyblock+1, 2*nzblock+1, total_elem);
  print "    Checking all elements in bounding box."
  sys.stdout.write("    Progress: ");

  n = 0L; interval = long(math.ceil(total_elem / 100.0));
  for i in xrange(-nxblock, nxblock+1):
    for j in xrange(-nyblock, nyblock+1):
      for k in xrange(-nzblock, nzblock+1):
        R = pow(i*xlen, 2) + pow(j*ylen, 2) + pow(k*zlen, 2)
        R = pow(R, 0.5)
        if R <= r:
          elems.append(Element(x+(i*xlen), y+(j*ylen), z+(k*zlen), p, xlen, ylen, zlen));
	n = n+1
	if (n%interval) == 0 or n == 1:
	  sys.stdout.write("%3.0f %%\b\b\b\b\b"%(n/interval))
	  sys.stdout.flush()
  print ""

def blockify_hdisc(x, y, z, r, h, p, xlen, ylen, zlen, elems):
  ## create all the blocks inside a horizontal disc
  # start from the center @ (x,y,z), and determine how many blocks will
  # fit in each axis direction.  Then, iterate over all the
  # possibilities, testing if R=sqrt(dx^2+dy^2)<r.  If so, then
  # add the block.
  nxblock = int(round(r/xlen));
  nyblock = int(round(r/ylen));
  nzblock = int(round(h/(2*zlen)));
  total_elem = (2L*nxblock+1)*(2L*nyblock+1)*(2L*nzblock+1);

  print "    Disc bounding box = %dx%dx%d elements (%g total)"%(2*nxblock+1,
    2*nyblock+1, 2*nzblock+1, total_elem);
  print "    Checking all elements in bounding box."
  sys.stdout.write("    Progress: ");

  n = 0L; interval = long(math.ceil(total_elem / 100.0));
  for k in xrange(-nzblock, nzblock+1):
    for i in xrange(-nxblock, nxblock+1):
      for j in xrange(-nyblock, nyblock+1):
        R = pow(i*xlen, 2) + pow(j*ylen, 2)
        R = pow(R, 0.5)
        if R <= r:
          elems.append(Element(x+(i*xlen), y+(j*ylen), z+(k*zlen), p, xlen, ylen, zlen));
	n = n+1
	if (n%interval) == 0 or n == 1:
	  sys.stdout.write("%3d %%\b\b\b\b\b"%int(round(n/interval)))
	  sys.stdout.flush()
  print ""

def blockify_block(x, y, z, xsize, ysize, zsize, p, xlen, ylen, zlen, elems):
  ## Create the blocks inside a large rectangular solid

  nxblock = int(round(xsize/(2*xlen)))
  nyblock = int(round(ysize/(2*ylen)))
  nzblock = int(round(zsize/(2*zlen)))
  total_elem = (2L*nxblock+1)*(2L*nyblock+1)*(2L*nzblock+1);

  print "    Block bounding box = %dx%dx%d elements (%g total)"%(2*nxblock+1,
    2*nyblock+1, 2*nzblock+1, total_elem);
  print "    Adding all elements in bounding box."
  sys.stdout.write("    Progress: ");

  n = 0L; interval = long(math.ceil(total_elem / 100.0));
  for i in xrange(-nxblock, nxblock+1):
    for j in xrange(-nyblock, nyblock+1):
      for k in xrange(-nzblock, nzblock+1):
	elems.append(Element(x+(i*xlen), y+(j*ylen), z+(k*zlen), p, xlen, ylen, zlen));
	n = n+1
	if (n%interval) == 0 or n == 1:
	  sys.stdout.write("%3d %%\b\b\b\b\b"%int(round(n/interval)))
	  sys.stdout.flush()
  print ""

######
## GO!
######
main()
