/*
 * genModel.c
 *
 * Generate geometry of blocks from higher-level shapes.
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#include "model.h" /* for getrecord() */
#include "blocktree.h"
#include "genModel.h"

char *version="v1.0.2";

int main(int argc, char *argv[])
{
  long nelem, nl;
  long i, j, interval;
  int nf;
  BlockList *blocks, *cblock, *iter1, *iter2;
  FILE *fptr;
  char *line, field[1024];
  double xlen, ylen, zlen;
  double oxlen, oylen, ozlen;
  double a, b, c, d, e, f, g;

  if(argc < 3) {
    fprintf(stderr, "usage: %s model_file out_file\n", argv[0]);
    exit(1);
    }

  /* open the model file */
  if((fptr = fopen(argv[1], "rt")) == NULL) {
    fprintf(stderr, "cannot open '%s' for read\n", argv[1]);
    exit(1);
    }

  /* setup vars */
  if((blocks = allocateBlock((void *) NULL)) == NULL) {
    fprintf(stderr, "cannot allocate memory for first Block\n");
    exit(1);
    }
  cblock = blocks;
  xlen = ylen = zlen = 1.0;
  nl = 0;

  /* read and parse */
  printf("Processing file %s\n", argv[1]);
  while(!feof(fptr)) {
    line = getrecord(fptr);
    if(feof(fptr)) continue;
    nl++;
    
    if(line == NULL) continue;

    sscanf(line, "%s", field);

    uppercase(field);

    if(!strcasecmp(field, "BLOCKSIZE")) {
      /* blocksize */
      printf(" BLOCKSIZE\n");
      oxlen = xlen; oylen = ylen; ozlen = zlen;
      nf = sscanf(line, "%s %lf %lf %lf", field, &xlen, &ylen, &zlen);
      if(nf != 4) {
        fprintf(stderr, "Illegal BLOCKSIZE directive at line %ld\n", nl);
	xlen = oxlen; ylen = oylen; zlen = ozlen;
        }
      printf("  xlen=%lf ylen=%lf zlen=%lf m\n", xlen, ylen, zlen);
      }
    else if(!strcasecmp(field, "BLOCK")) {
      /* block x y z rho xs ys zs */
      printf(" BLOCK\n");
      nf = sscanf(line, "%s %lf %lf %lf %lf %lf %lf %lf", field, &a, &b, &c, &d, &e, &f, &g);
      if(nf != 8) {
        fprintf(stderr, "Illegal BLOCK directive at line %ld\n", nl);
        continue;
        }
      else {
        cblock = blockify_block(a, b, c, d, e, f, g, xlen, ylen, zlen, cblock);
        }
      }
    else if(!strcasecmp(field, "SPHERE")) {
      /* sphere x y z r rho */
      printf(" SPHERE\n");
      nf = sscanf(line, "%s %lf %lf %lf %lf %lf", field, &a, &b, &c, &d, &e);
      if(nf != 6) {
        fprintf(stderr, "Illegal SPHERE directive at line %ld\n", nl);
        continue;
        }
      else {
        cblock = blockify_sphere(a, b, c, d, e, xlen, ylen, zlen, cblock);
        }
      }
    else if(!strcasecmp(field, "HDISC")) {
      /* hdisc x y z r h rho */
      printf(" HDISC\n");
      nf = sscanf(line, "%s %lf %lf %lf %lf %lf %lf", field, &a, &b, &c, &d, &e, &f);
      if(nf != 7) {
        fprintf(stderr, "Illegal HDISC directive at line %ld\n", nl);
        continue;
        }
      else {
        cblock = blockify_hdisc(a, b, c, d, e, f, xlen, ylen, zlen, cblock);
        }
      }
    else {
      fprintf(stderr, "Unknown directive %s at line %ld\n", field, nl);
      }
    }
  fclose(fptr);

  /* count # of elements */
  printf("Counting total # of created elements...");
  nelem = 0;
  iter1 = blocks;
  while(iter1 != NULL) {
    if(iter1->flag) nelem++;
    iter1 = (BlockList *)iter1->next;
    }
  printf(" %ld\n", nelem);

  interval = (long)ceil(nelem/100.0);

  /* resolve duplicate elements */
  printf("Resolving duplicated elements\n");
  fflush(stdout);

  iter1 = delDuplicate(blocks);
  free(blocks);
  blocks = iter1;

  /* count # of elements */
  nelem = 0;
  iter1 = blocks;
  while(iter1 != NULL) {
    if(iter1->flag) nelem++;
    iter1 = (BlockList *)iter1->next;
    }

  printf("Created %ld unique elements.  Writing to file %s\n", nelem, argv[2]);

  /* print output file */
  if((fptr = fopen(argv[2], "wt")) == NULL) {
    fprintf(stderr, "cannot open %s for output.\n", argv[2]);
    exit(1);
    }
  fprintf(fptr, "# Model geometry generated from %s\n", argv[1]);
  fprintf(fptr, "# by %s %s\n", argv[0], version);
  fprintf(fptr, "# Number of model elements\n");
  fprintf(fptr, "%ld\n", nelem);
  fprintf(fptr, "# Element definitions\n");
  iter1 = blocks;
  while(iter1 != NULL) {
    if(iter1->flag) {
      fprintf(fptr, "%lf %lf %lf %lf %lf %lf %lf\n",
		iter1->elem.x, iter1->elem.y, iter1->elem.z,
		iter1->elem.rho, iter1->elem.xlen, iter1->elem.ylen,
		iter1->elem.zlen);
      }
    iter1 = (BlockList *)iter1->next;
    }
  fclose(fptr);
  printf("Done\n");
}

void uppercase(char *string)
{
  char *ptr;

  ptr = string;
  while(*ptr != '\0') {
    *ptr = (char)toupper(*ptr);
    *ptr++;
    }
}

/* Allocate a block from memory, setting a few fields as well.
 * Returns NULL on failure.
 */
BlockList *allocateBlock(void *p)
{
  BlockList *b;
  if((b = (BlockList *)malloc(sizeof(BlockList))) == NULL) {
    return(NULL);
    }
  b->next = NULL;
  b->prev = p;
  b->flag = TRUE;
  return(b);
}

/* Set the element fields of a Block.  Returns a pointer to the
 * next block.
 */
BlockList *setBlock(double x, double y, double z, double p,
			double xs, double ys, double zs,
			BlockList *block)
{
  block->elem.x = x;
  block->elem.y = y;
  block->elem.z = z;
  block->elem.rho = p;
  block->elem.xlen = xs;
  block->elem.ylen = ys;
  block->elem.zlen = zs;
  if((block->next = allocateBlock((void *) block)) == NULL) {
    fprintf(stderr, "cannot allocate memory for block.\n");
    exit(1);
    }
  return((BlockList *)block->next);
}


BlockList *blockify_block(double X, double Y, double Z, double p, double xs,
				double ys, double zs, double xlen, double ylen,
				double zlen, BlockList *blocks)
{
  unsigned long i, j, k;
  unsigned long nx, ny, nz;
  double x, y, z;

  nx = (unsigned long)ceil(xs/xlen);
  ny = (unsigned long)ceil(ys/ylen);
  nz = (unsigned long)ceil(zs/zlen);

  X -= ((double)(nx-1)/2.0)*xlen;
  Y -= ((double)(ny-1)/2.0)*ylen;
  Z -= ((double)(nz-1)/2.0)*zlen;

  printf("  Block bounding box= %ldx%ldx%ld\n", nx, ny, nz); 
  for(i=0; i<nx; i++) {
    for(j=0; j<ny; j++) {
      for(k=0; k<nz; k++) {
        x = X + i*xlen;
        y = Y + j*ylen;
        z = Z + k*zlen;
	blocks = setBlock(x, y, z, p, xlen, ylen, zlen, blocks);
        }
      }
    }
  printf("  done with block\n");
  return(blocks);
}

BlockList *blockify_sphere(double X, double Y, double Z, double r, double p,
				double xlen, double ylen, double zlen,
				BlockList *blocks)
{

  unsigned long i, j, k;
  unsigned long nx, ny, nz;
  double x, y, z;
  double X0, Y0, Z0;
  double dx, dy, dz;

  nx = (unsigned long)ceil(2*r/xlen);
  ny = (unsigned long)ceil(2*r/ylen);
  nz = (unsigned long)ceil(2*r/zlen);

  X0=X; Y0=Y; Z0=Z;

  X -= ((double)(nx-1)/2.0)*xlen;
  Y -= ((double)(ny-1)/2.0)*ylen;
  Z -= ((double)(nz-1)/2.0)*zlen;

  printf("  Sphere bounding box= %ldx%ldx%ld\n", nx, ny, nz); 
  for(i=0; i<nx; i++) {
    for(j=0; j<ny; j++) {
      for(k=0; k<nz; k++) {
        x = X + i*xlen;
        y = Y + j*ylen;
        z = Z + k*zlen;
        dx = x - X0;
        dy = y - Y0;
        dz = z - Z0;
        if( (dx*dx + dy*dy + dz*dz) <= r*r)
	  blocks = setBlock(x, y, z, p, xlen, ylen, zlen, blocks);
        }
      }
    }
  printf("  done with sphere\n");
  return(blocks);
}

BlockList *blockify_hdisc(double X, double Y, double Z, double r, double h,
				double p, double xlen, double ylen,
				double zlen, BlockList *blocks)
{

  unsigned long i, j, k;
  unsigned long nx, ny, nz;
  double x, y, z;
  double X0, Y0, Z0;
  double dx, dy;

  nx = (unsigned long)ceil(2*r/xlen);
  ny = (unsigned long)ceil(2*r/ylen);
  nz = (unsigned long)ceil(h/zlen);

  X0=X; Y0=Y; Z0=Z;

  X -= ((double)(nx-1)/2.0)*xlen;
  Y -= ((double)(ny-1)/2.0)*ylen;
  Z -= ((double)(nz-1)/2.0)*zlen;

  printf("  Hdisc bounding box= %ldx%ldx%ld\n", nx, ny, nz); 
  for(i=0; i<nx; i++) {
    for(j=0; j<ny; j++) {
      for(k=0; k<nz; k++) {
        x = X + i*xlen;
        y = Y + j*ylen;
        z = Z + k*zlen;
        dx = x - X0;
        dy = y - Y0;
        if( (dx*dx + dy*dy) <= r*r)
	  blocks = setBlock(x, y, z, p, xlen, ylen, zlen, blocks);
        }
      }
    }
  return(blocks);
}

BlockList *delDuplicate(BlockList *block)
{
  BlockList *newlist;
  BlockTree *tree;

  /* generate the oct tree */
  printf(" Generating oct-tree\n");
  tree = genBlockTree(block);
  if(tree == NULL) { /* bad things happened down there */
    exit(1);
    }

  /* traverse the tree to build a new Block list */
  printf(" Traversing tree to build element list\n");
  newlist = blockTreeToList(tree);

  return(newlist);
}
