/*
 * compute.c
 *
 * Computes the gravity anomaly (vertical component only) at a set of
 * (x,y,z) positions.  Output values are in microGal (10^-6 Gal = 10^-8
 * m/s/s)
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */

#include <stdio.h>
#include "model.h"

/* #define DEBUG */

void usage(void);
void banner(void);

void usage(void)
{
  printf("usage: compute model_file position_file output_file\n");
}

void banner(void)
{
  printf("*******************************************\n");
  printf("* Block Gravity Model Computation Program *\n");
  printf("*                  v1.1a                  *\n");
  printf("*              Paul Gettings              *\n");
  printf("*      Dep't of Geology & Geophysics      *\n");
  printf("*            University of Utah           *\n");
  printf("*******************************************\n");
}

int main(int argc, char *argv[])
{
  Position pos;
  GravityModel model;
  double dg;
  FILE *fptr, *out;
  char *record;
  unsigned long np;

  if(argc<4) {
    usage();
    exit(0);
    }

  /* Hello banner */
  banner();

  model = loadModel(argv[1]);
  printf("Model geometry loaded.\n");
  if((fptr = fopen(argv[2], "rt")) == NULL) {
    printf("Cannot open %s for read.\n", argv[2]);
    exit(0);
    }
  if((out = fopen(argv[3], "wt")) == NULL) {
    printf("Cannot open %s for write.\n", argv[3]);
    exit(0);
    }

#ifdef DEBUG
  printf("Computing gravity at:\n");
#else
  printf("Computing gravity effect (in mGal) of %d elements\n", model.nelem);
  printf(" at position #");
  fflush(stdout);
#endif
  np = 0;
  while(!feof(fptr)) {
    record = getrecord(fptr);
    if(record == NULL) continue;
    sscanf(record, "%lg %lg %lg", &pos.x, &pos.y, &pos.z);
#ifdef DEBUG
    printf("  (%10lg, %10lg, %10lg)\n", pos.x, pos.y, pos.z);
#else
    printf("%8ld\b\b\b\b\b\b\b\b", np);
    fflush(stdout);
#endif
    dg = gravity(model, pos);
    dg *= 1e5; /* 100 is m/s -> cm/s, 1e3 is Gal -> mGal */
    fprintf(out, "%lg %lg %lg %lg\n", pos.x, pos.y, pos.z, dg);
    np++;
    }
  printf("\n");
  fclose(out);
  fclose(fptr);
  printf("Computed gravity at %lu points.\n", np);
}
