/*
 * blocktree.c
 *
 * Creates an oct-tree from an element list, or vice-versa
 *
 * This file is part of utah-g3d.
 *
 * utah-g3d is copyright (c) 2000 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * utah-g3d is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * utah-g3d is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */
#include <stdlib.h>
#include "blocktree.h"
#include "genModel.h"

/* tree counters */
int depth=0;
long leaves=0;

/* Convert a tree to a list of elements
 * Returns a pointer to the new list
 */
BlockList *blockTreeToList(BlockTree *tree)
{
  BlockList *list, *iter;

  if((list = allocateBlock((void *)NULL)) == NULL)
    return(NULL);

  depth=0;
  iter = list;
  iter = traverseTree(tree, iter);
  /* drop the last block */
  iter = (BlockList *)iter->prev;
  free(iter->next);
  iter->next = NULL;
  return(list);
}

/* Traverses a tree, creating a list of elements from each leaf
 * Returns the list of elements, which are malloc'd from free memory
 */
BlockList *traverseTree(BlockTree *tree, BlockList *blocks)
{
  int i;
  Element *e;

  depth++;

  if(tree->Leaf) {
    leaves++;
    e = (Element *)tree->branches[0];
    copyElement(*e, &blocks->elem);
    if((blocks->next = allocateBlock((void *)blocks)) == NULL) {
      fprintf(stderr, "cannot allocate memory for next block.\n");
      exit(1);
      }
    blocks = (BlockList *)blocks->next;
    return(blocks);
    }
  else {
    for(i=0; i<8; i++) {
      if(tree->branches[i] == NULL) {
        continue;
        }
      else {
        blocks = traverseTree((BlockTree *)tree->branches[i], blocks);
        }
      }
    }
  depth--;
  return(blocks);
}

/* Allocates a new branch for the oct-tree, setting some parameters
 * Returns a pointer to the new branch node
 */
BlockTree *allocateBranch()
{
  BlockTree *b;
  int i;

  if((b = (BlockTree *)malloc(sizeof(BlockTree))) == NULL) {
    return(NULL);
    }
  for(i=0; i<8; i++) {
    b->branches[i] = NULL;
    }
  b->Leaf = 0;
  return(b);
}

/* Generates an oct-tree from the list of elements
 * Returns NULL on error
 */
BlockTree *genBlockTree(BlockList *blocks)
{
  Range extents;
  BlockTree *tree;
  BlockList *biter;
  long nb;


  if((tree = allocateBranch()) == NULL) {
    fprintf(stderr, "cannot allocate root branch.\n");
    return(NULL);
    }

  /* get range of interest */
  biter = blocks;
  extents.xmax = extents.xmin = biter->elem.x;
  extents.ymax = extents.ymin = biter->elem.y;
  extents.zmax = extents.zmin = biter->elem.z;
  extents.minResX = biter->elem.xlen;
  extents.minResY = biter->elem.ylen;
  extents.minResZ = biter->elem.zlen;
  while(biter->next != NULL) {
    if(biter->elem.x > extents.xmax)
      extents.xmax = biter->elem.x;
    if(biter->elem.x < extents.xmin)
      extents.xmin = biter->elem.x;
    if(biter->elem.y > extents.ymax)
      extents.ymax = biter->elem.y;
    if(biter->elem.y < extents.ymin)
      extents.ymin = biter->elem.y;
    if(biter->elem.z > extents.zmax)
      extents.zmax = biter->elem.z;
    if(biter->elem.z < extents.zmin)
      extents.zmin = biter->elem.z;
    if(biter->elem.xlen < extents.minResX)
      extents.minResX = biter->elem.xlen;
    if(biter->elem.ylen < extents.minResY)
      extents.minResY = biter->elem.ylen;
    if(biter->elem.zlen < extents.minResZ)
      extents.minResZ = biter->elem.zlen;

    biter = (BlockList *)biter->next;
    }
  extents.minResX /= 2.0;
  extents.minResY /= 2.0; 
  extents.minResZ /= 2.0;


  /* build the tree */
  nb = 0; depth=0;
  biter = blocks;
  while(biter->next != NULL) {
    nb++;
    if(!insertLeaf(*biter, tree, extents)) {
      fprintf(stderr, "error inserting element %ld into tree.\n", nb);
      return(NULL);
      }
    biter = (BlockList *)biter->next;
    }

  return(tree);
}

/* Inserts a leaf node into the tree, creating branches as necessary.
 * Returns 0 on failure.
 */
int insertLeaf(BlockList block, BlockTree *tree, Range r)
{
  Range R;
  int o, rc;
  double midx, midy, midz;

  depth++;
  if((r.xmax - r.xmin)<=r.minResX && (r.ymax-r.ymin)<=r.minResY &&
    (r.zmax-r.zmin)<=r.minResZ) {
    /* leaf node */
    if(tree->branches[0] != NULL) {
      free(tree->branches[0]);
      tree->branches[0] = NULL;
      }
    if((tree->branches[0] = malloc(sizeof(Element))) == NULL) {
      depth--;
      return(0);
      }
    copyElement(block.elem, (Element *)tree->branches[0]);
    tree->Leaf = 1;
    depth--;
    return(1);
    }
  else {
    /* another branch */
    R.xmax = r.xmax; R.ymax = r.ymax; R.zmax = r.zmax;
    R.xmin = r.xmin; R.ymin = r.ymin; R.zmin = r.zmin;
    R.minResX = r.minResX; R.minResY = r.minResY; R.minResZ = r.minResZ;
    midx = (r.xmax + r.xmin)/2.0;
    midy = (r.ymax + r.ymin)/2.0;
    midz = (r.zmax + r.zmin)/2.0;

    /* find the octant */
    o = 0;
    if(block.elem.x > midx) o += 1;
    if(block.elem.y > midy) o += 2;
    if(block.elem.z > midz) o += 4;

    if(tree->branches[o] == NULL) {
      /* allocate a branch */
      if((tree->branches[o] = allocateBranch()) == NULL) {
       depth--;
       return(0);
       }
      }
    switch(o) {
      case 0: R.xmax = midx; R.ymax = midy; R.zmax = midz;
	      break;
      case 1: R.xmin = midx; R.ymax = midy; R.zmax = midz;
	      break;
      case 2: R.xmax = midx; R.ymin = midy; R.zmax = midz;
	      break;
      case 3: R.xmin = midx; R.ymin = midy; R.zmax = midz;
	      break;
      case 4: R.xmax = midx; R.ymax = midy; R.zmin = midz;
	      break;
      case 5: R.xmin = midx; R.ymax = midy; R.zmin = midz;
	      break;
      case 6: R.xmax = midx; R.ymin = midy; R.zmin = midz;
	      break;
      case 7: R.xmin = midx; R.ymin = midy; R.zmin = midz;
	      break;
      default:
              depth--;
              return(0);
              break;
      }
    rc = insertLeaf(block, (BlockTree *)tree->branches[o], R);
    depth--;
    return(rc);
    }
}
