/* compute terrain correction from integration of DGG diamonds
 *
 * use Gauss-Legendre integration of a spherical diamond, from
 * Asgharzadeh et al 2007
 *
 * Use a discrete global grid @ 500 m center-center distance filled
 * with DEM values from SRTM (-60 to 60), GLOBE (poles, land), and
 * ETOPO1 (oceans).  Ocean mask derived from cells that needed filling
 * from ETOPO1.
 *
 * Read exploration format and replace terrain correction entries with
 * updated values....
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>

#include "global_grid/global_grid.h"
#include "spherical.h"
#include "constants.h"
#include "gausslegquad.h"
#include "fileio.h"
#include "octree.h"

#define DGG_MIN_RES 468 /* m; distance between adjacent polygons */
#define MAXDIST	166735 /* m for maximum distance of terrain correction */

typedef struct gravityStationStruct {
  char name[26];
  double gravity, stderr;
  double elev, iztc, ttc;
  LonLatPoint location;
  } GravStation;

typedef struct gravityStnCollectionStruct {
  unsigned long n;
  GravStation *info;
  } GravStations;

// if 1, print lots of helpful info to stderr
char terrain_verbose=1;
// if 1, output files with local_dems for each station
char map_debug=0;
// if 1, read DEM for each station - for doing global station coverage
char global_stns=0;

/* Compute gravity effect triple integral of a diamond-shaped spherical
 * polygon.
 *
 * order[3] specifies the number of GLQ points in each axis:
 * [radius, latitude, longitude].
 * Set the order such that the distance between GLQ points is less than the
 * distance from the prism to the observation point (O, r)!  Note that GLQ
 * points are roughly equally spaced inside the prism, so divide the max
 * dimension of the prism by the observation distance to get order requirement.
 *
 * Latitudes and longitudes are in RADIANS!!!!
 */
double integrateDiamond(LonLatPoint P[], LonLatPoint O, double r, double r1, double r2, double rho, int order[3])
{
  int i, j, k;
  double *nl, *nr, *np; /* values along [-1,1] interval in lambda, phi, radius */
  double *wl, *wr, *wp; /* Gauss-Legendre Quad. weights for lambda, phi, radius */
  double top, bot, lats[12], lons[12], rads[12];
  double cosdelta, R, K, C;
  double cosOlat, cosSlat, sinOlat, sinSlat;
  double cosDeltaLon, sinDeltaLon;
  double maxlon, minlon;
  double I, sum, gr, gl, gp;
  LonLatPoint A;

  // get GLQ abcissas (integration points in [0,1] space) and weights
  nr = GLQinterval(order[0]); nl = GLQinterval(order[1]); np = GLQinterval(order[2]);
  wr = GLQweights(order[0]); wl = GLQweights(order[1]); wp = GLQweights(order[2]);

  // fiddle with longitude signs to handle prime meridian and -PI/PI transitions
  j=0; // count of negative longitudes
  for(i=0; i<4; i++) { if(P[i].lon<0) j++; }
  if(j || O.lon<0) { // if a negative longitude present, fiddle with signs
    k=0;
    I=fabs(P[0].lon-P[2].lon);
    if(I<1) I=fabs(P[1].lon-P[3].lon);
    if(I>1) {  // if I large, then stations near +-180 deg ==> add to negatives only
      for(i=0; i<4; i++)
        if(P[i].lon<0) P[i].lon+=2*PI;
      if(O.lon<0) O.lon+=2*PI;
      }
    else { // if I small, then stations at meridian ==> add to all stations.
      for(i=0; i<4; i++) P[i].lon+=2*PI;
      O.lon += 2*PI;
      }
    }

  /* cos() and sin() of colatitude of observation point fixed, so compute once */
  cosOlat = sin(O.lat); // cos of colat is sin of lat!
  sinOlat = cos(O.lat);

  // Scale coordinates from [0,1] to radii and angles on the Earth
  for(i=0; i<order[0]; i++) { // radii points
    rads[i] = ( nr[i]*(r2-r1)+(r2+r1) )/2;
    }
  minlon=P[0].lon; maxlon=P[2].lon;
  if(P[1].lon<minlon) { minlon=P[1].lon; maxlon=P[3].lon; }
  if(P[2].lon<minlon) { minlon=P[2].lon; maxlon=P[0].lon; }
  if(P[3].lon<minlon) { minlon=P[3].lon; maxlon=P[1].lon; }
  for(i=0; i<order[2]; i++) { // longitudes (in radians!)
    lons[i] = ( np[i]*(maxlon-minlon)+(maxlon+minlon) )/2;
    }
  /* N.B. - latitudes need to be set for each longitude point; discrete
   * global grid diamonds are not N-S oriented, so do latitude limits at
   * each longitude; thus compute lats[] inside the quadrature loop
   */
  I=0;
  for(i=0; i<order[0]; i++) {	/* radius */
    for(j=0; j<order[2]; j++) { /* longitude (phi) */
      /* determine min, max lats at current longitude points */
      if(P[0].lon<=lons[j] && lons[j]<=P[1].lon) { // which leg of poly is this longitude on?
	A = greatCirclePtLon(P[0], P[1], lons[j]);}
      else if(P[1].lon<=lons[j] && lons[j]<=P[0].lon) {
	A = greatCirclePtLon(P[1], P[0], lons[j]);}
      else if(P[3].lon<=lons[j] && lons[j]<=P[0].lon) {
	A = greatCirclePtLon(P[3], P[0], lons[j]);}
      else if(P[0].lon<=lons[j] && lons[j]<=P[3].lon) {
	A = greatCirclePtLon(P[0], P[3], lons[j]);}
      top = A.lat;
      if(P[1].lon<=lons[j] && lons[j]<=P[2].lon) { // which leg of poly is this longitude on?
	A = greatCirclePtLon(P[1], P[2], lons[j]);}
      else if(P[2].lon<=lons[j] && lons[j]<=P[1].lon) {
	A = greatCirclePtLon(P[2], P[1], lons[j]);}
      else if(P[2].lon<=lons[j] && lons[j]<=P[3].lon) {
	A = greatCirclePtLon(P[2], P[3], lons[j]);}
      else if(P[3].lon<=lons[j] && lons[j]<=P[2].lon) {
	A = greatCirclePtLon(P[3], P[2], lons[j]);}
      bot = A.lat;
      /* check that top is N of bottom; swap if not */
      if(top<bot) { A.lat=top; top=bot; bot=A.lat; }
      /* compute scaled lats at current longitude */
      for(k=0; k<order[1]; k++) {
	lats[k] = ( nl[k]*(top-bot)+(top+bot) )/2;
	}
      /* integrate via Gauss-Legendre Quadrature; integral becomes sums
       * See Asgharzadeh et al 2007 for the equations implemented here
       */
      cosDeltaLon = cos(O.lon - lons[j]);
      sinDeltaLon = sin(O.lon - lons[j]);
      sum=0; // sum over current lat limits
      for(k=0; k<order[1]; k++) { /* latitude (lambda) */
	cosSlat = sin(lats[k]); // cos of source colatitude == sin source latitude
        sinSlat = cos(lats[k]);
        cosdelta = cosOlat*cosSlat + sinOlat*sinSlat*cosDeltaLon;
        /* R is only ever used as 1/R^3, so we do division and cubing at same time
         * as sqrt() by raising to -1.5 power
         */
        R = pow(r*r + rads[i]*rads[i] - 2*r*rads[i]*cosdelta, -1.5);
        K = BIGG*rho * rads[i]*rads[i]*sinSlat; /* rho is kg/m^3 (density) */
        C = wr[i]*wl[j]*wp[k]*R*K;
          /* WARNING - HERE BE MAGIC
           * The fabs() call is here to handle the 2 cases in a TERRAIN CORRECTION CALCULATION:
           * 1) air instead of rock BELOW stn elevation, and
           * 2) rock instead of air ABOVE stn elevation.
           * In case 1, rho is negative, in case 2, rho positive; BUT radial difference will flip
           * sign between case 1 and 2! Which means that either change sign of rho for 2 cases, OR
           * TAKE FABS(DELTA RADIUS) AND ALWAYS USE POSITIVE RHO!!!!!!
           * So, take fabs() and use rho=2670 FOR ALL CASES.
           * This does NOT apply to a general calculation of the gravity effect of a prism! Drop the
           * fabs() and correct rho!
           */
        gr = C*fabs(rads[i]*cosdelta - r);	/* radial component */
        //gl = C* rads[i]*(cosOlat*sinSlat*cosDeltaLon  - sinOlat*cosSlat); /* latitude component */
        //gp = C* -1*rads[i]*sinSlat*sinDeltaLon; /* longitude component */
        sum += gr; // only want radial component 
        }
      I += sum*(top-bot)/2;	/* scale integral sum by current latitude range */
      //if(I<0) { fprintf(stderr, "DEBUG: neg contribution: %lf %lf %lf\n", I,sum,top-bot); }
      }
    }
  I *= fabs(P[2].lon-P[0].lon)*fabs(r2-r1)/4;	/* scale integral sum by integration limits */
  return(I*1e5); /* convert to mGal from m/s^2 */
}

/* Load polygon geometry, elevation, and masks from disk files
 *
 * select only polys in lon, lat box rounding to nearest deg
 * so can use index files rather than scanning 2.7 billion polys
 * in 220 GB disk file....
 */
GlobalGrid loadDEM(float lon[], float lat[])
{
  FILE *fp;
  int i,j,k, n;
  GlobalGrid dem;
  GlobalGridMaskValue M;
  GlobalGridValue V;
  PolygonList polys;

  if(terrain_verbose)fprintf(stderr,"->> Load DEM data for bbox around %.3f,%.3f to %.3f,%.3f\n", lon[0],lat[0],lon[1],lat[1]);
  if(terrain_verbose)fprintf(stderr,"--> Find polygons in bounding box...\n");

  // get polygon list from lat, lon bounds 
  polys = getPolysFromIndex(lat[0], lon[0], lat[1], lon[1]);

  if(terrain_verbose)fprintf(stderr,"--> found %lu polygons in bbox; loading DEM data...\n", polys.npolys);

  // setup the GlobalGrid for these polygons
  dem.npolys = polys.npolys;
  dem.polys = (GlobalGridPoly *)malloc(sizeof(GlobalGridPoly)*dem.npolys);
  dem.vals = (float *)malloc(sizeof(float)*dem.npolys);
  dem.mask = (short int *)malloc(sizeof(short int)*dem.npolys);

  fp=fopen(GEOM_FILE_500m, "rb");
  for(i=0; i<dem.npolys; i++) {
    dem.polys[i] = getPolyById(fp, polys.ids[i]);
    }
  fclose(fp);
  fp=fopen(MASK_FILE_500m, "rb");
  for(i=0; i<dem.npolys; i++) {
    M = getMaskById(fp, polys.ids[i]);
    dem.mask[i] = M.value;
    }
  fclose(fp);
  fp=fopen(VALUE_FILE_500m, "rb");
  for(i=0; i<dem.npolys; i++) {
    V = getValueById(fp, polys.ids[i]);
    dem.vals[i] = V.value;
    }
  fclose(fp);

  free(polys.ids); // free polygon list allocated in getPolysFromIndex()

  if(terrain_verbose)fprintf(stderr,"--> loaded DEM data for polygons\n");

  return(dem);
}

extern char *optarg;
extern int optind, opterr, optopt;

int main(int argc, char *argv[])
{
  FILE *fp, *mapfp;
  char *line;

  GlobalGrid dem, local_dem;
  OctTree *tree;
  LonLatPoint P[4], A, *loc;
  XyzPoint U,V;
  GravStations stns;
  GravStation *stn;
  unsigned long n, i,j, ncells;
  long offset=0; // number of stations to skip at start
  float lat[2], lon[2]; // min, max lat & lon bdy
  double dg, re, re_stn, dist, zmin, zmax;
  double g, scale;
  char opt;
  char *optstring="qmvsh?o:";

  int order[3];

  while((opt=getopt(argc, argv, optstring)) != -1) {
    switch(opt) {
      case 'q': terrain_verbose=0; break;
      case 'v': terrain_verbose=1; break;
      case 'm': map_debug=1; break;
      case 's': global_stns=1; break;
      case 'o': offset=atol(optarg); break;
      case 'h':
      case '?':
        fprintf(stderr, "usage: %s [-q][-v] [-m] [-s] [-o offset] input_file > output_file\n", argv[0]);
        fprintf(stderr, "  -q	quiet; minimal output\n");
        fprintf(stderr, "  -v	verbose; normal output - opposite of -q\n");
        fprintf(stderr, "  -m	map debug; save x,y,z files for full DEM and each station\n");
        fprintf(stderr, "  -s	large station extent; load DEM from disk for each station. Less memory, slower than single DEM for all stations\n");
        fprintf(stderr, "  -o	skip first 'offset' stations in input file\n");
        exit(3);
        break;
      default:
        break;
      }
    }

  if(terrain_verbose)fprintf(stderr,"***************************************************************************\n");
  if(terrain_verbose)fprintf(stderr,"** Terrain correction using spherical diamonds of a discrete global grid **\n");
  if(terrain_verbose)fprintf(stderr,"** DEM data from SRTM, GLOBE, and ETOPO1 data sets                       **\n");
  if(terrain_verbose)fprintf(stderr,"** v1.5 (Feb 2013)  Paul Gettings, University of Utah                    **\n");
  if(terrain_verbose)fprintf(stderr,"***************************************************************************\n");

  fprintf(stderr, ">>> Loading station data from '%s'...\n", argv[optind]);

  if((fp=fopen(argv[optind], "rt")) == NULL) { fprintf(stderr, "can't open '%s' for reading.\n", argv[1]); exit(2); }
  n=0;
  // count stations
  while(!feof(fp)) {
    line=getrecord(fp);
    if(line==NULL) { continue; }
    n++;
    }
  if(n<1) { fprintf(stderr, "--> found no stations; stop.\n"); exit(0); }
  if(terrain_verbose && offset>0)fprintf(stderr,"->> skipping %ld stations at start.\n", offset);
  // read stations
  stns.n = n-offset;
  stns.info=(GravStation *)malloc(sizeof(GravStation)*stns.n);
  n=0;
  rewind(fp);
  while(!feof(fp)) {
    line=getrecord(fp);
    if(line==NULL) { continue; }
    if(n<offset) {n++; continue;} // skip first 'offset' stations
    // name lon lat z gravity sigma iztc; ignore ttc and comment fields on line
    stn = &(stns.info[n-offset]); // for shorter lines below
    sscanf(line, "%25s %lf %lf %lf %lf %lf %lf", &stn->name, &stn->location.lon,&stn->location.lat,
      &stn->elev, &stn->gravity, &stn->stderr, &stn->iztc);
    n++;
    }
  fclose(fp);
  if(terrain_verbose)fprintf(stderr,"--> got %u stations; using maximum distance of %g km.\n", stns.n, MAXDIST/1000.0);

  if(global_stns) {
    if(terrain_verbose)fprintf(stderr,">>> Global station flag set; load DEM from disk for each station.\n");
    }
  // if not global_stns flag, load single DEM for all stations
  else {
    if(terrain_verbose)fprintf(stderr,">>> Load DEM for all stations to max distance...\n");
    lat[0] = stns.info[0].location.lat;
    lon[0] = stns.info[0].location.lon;
    lat[1] = stns.info[0].location.lat;
    lon[1] = stns.info[0].location.lon;
    for(n=1; n<stns.n; n++) {
      if(stns.info[n].location.lat < lat[0]) {
	lat[0] = stns.info[n].location.lat;
	}
      if(stns.info[n].location.lat > lat[1]) {
	lat[1] = stns.info[n].location.lat;
	}
      if(stns.info[n].location.lon < lon[0]) {
	lon[0] = stns.info[n].location.lon;
	}
      if(stns.info[n].location.lon > lon[1]) {
	lon[1] = stns.info[n].location.lon;
	}
      }
    // clip to world, check for -180 to 180
    if(lat[0] < -90) { lat[0] = -90; }
    if(lon[0] < -180) { lon[0] = -180; }
    if(lat[1] > 90) { lat[1] = 90; }
    if(lon[1] > 180){ lon[1] -= 360; }
    if(lon[0] > lon[1]) { A.lon=lon[1]; lon[1]=lon[0]; lon[0]=A.lon; } 


    // move out MAXDIST from max, min....
    dist = MAXDIST* 360/(2*PI*WGS84_b); // 166.7 km in degrees of lat
    lat[0] -= dist;
    lat[1] += dist;
    P[0].lat = lat[0]+(lat[1]-lat[0])/2;
    P[0].lon = lon[0]+(lon[1]-lon[0])/2;
    A = deg2rad(P[0]);
    re = cos(P[0].lat*DEG2RAD) * ellipsoid_radius(A);
    dist = MAXDIST* 360/(2*PI*re); // 166.7 km in deg lon @ this lat
    lon[0] -= dist;
    lon[1] += dist;

    // clip to the world
    if(lat[0] < -90) lat[0] = -90;
    if(lat[1] > 90) lat[1] = 90;
    if(lon[0] < -180) lon[0] = -180;
    if(lon[1] > 180) lon[1] = 180;

    // load DGG polys in the box
    lon[0] = floor(lon[0]); lon[1] = ceil(lon[1]);
    lat[0] = floor(lat[0]); lat[1] = ceil(lat[1]);
    dem = loadDEM(lon, lat);

    // output a copy of the whole local DEM
    if(map_debug) {
      mapfp = fopen("TOTAL_DEM", "wt");
      if(mapfp==NULL) map_debug=0;
      }
    if(map_debug) {
      fprintf(stderr, "*** Map debug: output full DEM to TOTAL_DEM.\n");
      for(j=0; j<dem.npolys; j++) {
	fprintf(mapfp, "%ld %.3f %.7f %.7f %.7f %.7f %.7f %.7f %.7f %.7f %.7f %.7f\n",
	  dem.polys[j].id, dem.vals[j], dem.polys[j].center.lon, dem.polys[j].center.lat,
	  dem.polys[j].verts[0].lon, dem.polys[j].verts[0].lat,
	  dem.polys[j].verts[1].lon, dem.polys[j].verts[1].lat,
	  dem.polys[j].verts[2].lon, dem.polys[j].verts[2].lat,
	  dem.polys[j].verts[3].lon, dem.polys[j].verts[3].lat);
	}
      fclose(mapfp);
      }
    /* build cartesian oct-tree for selecting polys in range; set x,y,z
     * resolutions to 100 m less than DGG inter-cell-center distance, so
     * 1 poly per leaf node in oct-tree */
    if(terrain_verbose)fprintf(stderr,"->> build Cartesian octree for polygon searches\n");
    tree = genOctTree(dem, DGG_MIN_RES-100, DGG_MIN_RES-100, DGG_MIN_RES-100);
    if(terrain_verbose)fprintf(stderr,"--> built octree\n");
    }  // end if(!global_stns)

  if(terrain_verbose)fprintf(stderr,">>> Processing stations....\n");
  if(map_debug) fprintf(stderr, "*** Map Debug: Output 'local' map for each station to station name files....\n");
  // run through each station, checking polys in range to compute terrain correction
  for(i=0; i<stns.n; i++) {
    dg=0; ncells=0;
    re_stn = ellipsoid_radius(deg2rad(stns.info[i].location));

    if(global_stns) { // load DEM for this station
      lon[0] = stns.info[i].location.lon;
      lon[1] = stns.info[i].location.lon;
      lat[0] = stns.info[i].location.lat;
      lat[1] = stns.info[i].location.lat;
      dist = MAXDIST* 360/(2*PI*WGS84_b); // 166.7 km in degrees of lat
      lat[0] -= dist;
      lat[1] += dist;
      re = cos(P[0].lat*DEG2RAD) * re_stn;
      dist = MAXDIST* 360/(2*PI*re); // 166.7 km in deg lon @ this lat
      lon[0] -= dist;
      lon[1] += dist;
      local_dem = loadDEM(lon, lat);
      }
    else { /* find portion of full DEM in range */
      /* N.B. - use ellipsoid for height when finding polys, so don't
       * have problems with oceans and heights of -3 to -10 km. Then
       * use real elevations for actual calculations.
       */
      local_dem = octTreeToGrid(tree, llh2xyz(deg2rad(stns.info[i].location), 0), MAXDIST*1.25);
      }
    if(terrain_verbose)fprintf(stderr,"### %-25s = ", stns.info[i].name);
    /* run through DEM polys, accumulating dg */
    if(map_debug) { mapfp = fopen(stns.info[i].name, "wt"); if(mapfp==NULL) map_debug=0; }
    for(j=0; j<local_dem.npolys; j++) {
      dist=ellipsoid_dist(deg2rad(local_dem.polys[j].center), deg2rad(stns.info[i].location));
      if (dist > MAXDIST) { continue; } // skip cells too far away....
      if(map_debug) { // if wanted, output cell info to map file for debugging, etc.
        fprintf(mapfp, "%ld %.3f %.7f %.7f %.7f %.7f %.7f %.7f %.7f %.7f %.7f %.7f\n",
          local_dem.polys[j].id, local_dem.vals[j], local_dem.polys[j].center.lon, local_dem.polys[j].center.lat,
          local_dem.polys[j].verts[0].lon, local_dem.polys[j].verts[0].lat,
          local_dem.polys[j].verts[1].lon, local_dem.polys[j].verts[1].lat,
          local_dem.polys[j].verts[2].lon, local_dem.polys[j].verts[2].lat,
          local_dem.polys[j].verts[3].lon, local_dem.polys[j].verts[3].lat);
          }
      // this poly in range, compute grav effect
      for(n=0; n<4; n++) { // setup polygon vertices into P[]
	P[n] = deg2rad(local_dem.polys[j].verts[n]);
	}
      // get zmin, zmax, ellipsoid radius
      if(local_dem.vals[j] < stns.info[i].elev) { zmin=local_dem.vals[j]; zmax=stns.info[i].elev; }
      else { zmax=local_dem.vals[j]; zmin=stns.info[i].elev; }
      re = ellipsoid_radius(deg2rad(local_dem.polys[j].center));
      /* compute order entries
       * ... so divide the max dimension of the prism by the observation
       * distance to get order requirement.
       */
      scale = ellipsoid_dist(P[0], P[2]);
      if(scale>zmax-zmin) scale/=dist; // if radial dimension biggest, use that!
      else scale=(zmax-zmin)/dist;
      for(n=0; n<3; n++) {
	order[n] = (int)scale;
	if(order[n]<1) order[n]=1;
	if(order[n]>12) order[n]=12; // clip to size of arrays in gausslegquad.c
	}
      // compute grav effect of diamond on sphere of Earth radius...
      if(local_dem.mask[j] == 1) { // ocean cell;
      		// replace water-filled cell from DEM elev to 0 with grav-equiv cell of air
      		// by resetting z levels around center of mass....
      		// assume density of seawater is 1035 kg/m^3
        zmin = local_dem.vals[j]/2 + local_dem.vals[j]/2*(2670.0-1035.0)/2670.0; // scale height by drho/2670, then 1/2 height below
        zmax = local_dem.vals[j]/2 - local_dem.vals[j]/2*(2670.0-1035.0)/2670.0; // scale height by drho/2670, then 1/2 height above
	// integrate water cell as shorter air cell centered on ctr-of-mass of water cell
	g = integrateDiamond(P, deg2rad(stns.info[i].location), stns.info[i].elev+re_stn, zmin+re, zmax+re, 2670, order);
	if(fabs(g)>4){ if(terrain_verbose)fprintf(stderr,"\nWARNING: EQUIV WATER POLYGON WITH |TC|>4 mGal: %16.8f %lu\n", g, local_dem.polys[j].id);
	  if(fabs(g)>1e3){ if(terrain_verbose)fprintf(stderr, " TC_DEBUG: dem:%lf re:%lf zmin:%lf zmax:%lf elev:%lf\n",
	    local_dem.vals[j], re, zmin, zmax, stns.info[i].elev);
	    exit(12);
	    }
	  }
	if(isnan(g)) { if(terrain_verbose)fprintf(stderr,"nan result for integration of ocean poly %lu, dist=%.3f\n",local_dem.polys[j].id, dist); g=0; }
	ncells++;
	// integrate MSL to stn elevation as air cell
	if(stns.info[i].elev < 0) { zmin=stns.info[i].elev; zmax=0; }
	else { zmax=stns.info[i].elev; zmin=0; }
	g += integrateDiamond(P, deg2rad(stns.info[i].location), stns.info[i].elev+re_stn, zmin+re, zmax+re, 2670, order);
	if(!isnan(g)) dg += g; // add unless a nan
	else if(terrain_verbose)fprintf(stderr,"nan result for integration of air above water poly %lu, dist=%.3f\n",local_dem.polys[j].id, dist);
	ncells++;
        }
      else { // land cell, fill z difference with air (drho = 2670)
	g = integrateDiamond(P, deg2rad(stns.info[i].location), stns.info[i].elev+re_stn, zmin+re, zmax+re, 2670, order);
	if(!isnan(g)) dg += g; // add unless a nan
	else {
	  if(terrain_verbose)fprintf(stderr,"nan result for integration of air poly %lu, dist=%.3f\n",local_dem.polys[j].id, dist);
	  }
	ncells++;
	if(fabs(g)>1){ if(terrain_verbose)fprintf(stderr,"\nWARNING: SINGLE POLYGON WITH |TC|>1 mGal: %16.8f %lu\n", g, local_dem.polys[j].id);
	  if(fabs(g)>1e3){ if(terrain_verbose) fprintf(stderr, " TC_DEBUG: dem:%lf re:%lf zmin:%lf zmax:%lf elev:%lf\n",
	    local_dem.vals[j], re, zmin, zmax, stns.info[i].elev);
	    }
	  }
	}
      }
    if(map_debug) { fclose(mapfp); }
    if(fabs(dg)>1e4) { fprintf(stderr, "WARNING: |TC|>10 000 ==> RERUN: "); }
    // output updated exploration format
    printf("%-25s %11.6f %11.6f %11.3f %12.3lf %5.3lf %8.3f %8.3f\n",
      stns.info[i].name, stns.info[i].location.lon, stns.info[i].location.lat, stns.info[i].elev,
      stns.info[i].gravity, stns.info[i].stderr, stns.info[i].iztc, dg+stns.info[i].iztc);
    fflush(stdout);
    if(terrain_verbose)fprintf(stderr,"%10.5lf  [%8lu cells from %lu polys]  %6.2f%% done\n", stns.info[i].iztc+dg, ncells, local_dem.npolys, ((i+1)*100.0)/stns.n);
    else fprintf(stderr, ".");
    free(local_dem.polys); /* clean up local_dem for next station */
    free(local_dem.vals);
    free(local_dem.mask);
    }
  if(terrain_verbose)fprintf(stderr,">>> Done.\n");
}
