/* Rotations and great circle operations on a sphere
 *
 * Useful for geographic stuff where need to move things along
 * the surface of a sphere.
 */
#ifndef SPHERICAL_GEOGRAPHIC_TYPES_H
#define SPHERICAL_GEOGRAPHIC_TYPES_H

/* make sure we have PI, RAD2DEG, etc. */
#include "constants.h"
/* make sure we have GeoPoint, GeoPolygon, etc. */
#include "geopolys.h"

/* compute colatitude from geographic latitude */
#define colat(x) (PI/2-(x)) 

typedef struct rotationMatrixStruct {
  double v[3][3];
  } RotMatrix;

/* convert lat/lon to radians and back */
GeoPoint deg2rad(GeoPoint A);
GeoPoint rad2deg(GeoPoint A);

/* Compute distance along the ellipsoid (WGS84) between 2 GeoPoints.
 *
 * Returns distance, in m, between A and B along the WGS84 ellipsoid surface.
 */
double ellipsoid_dist(GeoPoint A, GeoPoint B);
/* Compute radius of the WGS84 ellipsoid at the specified location
 *
 * returns the radius in m
 */
double ellipsoid_radius(GeoPoint A);

/* Compute a point partway along a great circle path
 *
 * Returns a new GeoPoint that is partway between P and Q;
 * alpha between [0,1] is the fraction of the path along PQ.
 */
GeoPoint greatCirclePt(GeoPoint P, GeoPoint Q, double alpha);
/* Compute points partway along a great circle path
 *
 * Returns an array of GeoPoints that are partway between P and Q;
 * alpha between [0,1] is the fraction of the path along PQ.
 * n is number of points in alpha[]
 * Don't forget to free the returned array!
 */
GeoPoint *greatCirclePtAry(GeoPoint P, GeoPoint Q, double *alpha, int n);
/* Compute a point partway along a great circle path
 *
 * Returns a new GeoPoint that is partway between P and Q;
 * new point is along the arc by (lon-P.lon)/(Q.lon-P.lon).
 */
GeoPoint greatCirclePtLon(GeoPoint P, GeoPoint Q, double lon);

/* return a 3x3 rotation matrix to rotate about the Euler pole at
 * lon,lat by the angle omega, in radians
 *
 * returned pointer is to a newly allocated matrix, so don't forget
 * to free after use!
 */
RotMatrix rotation_matrix(GeoPoint Q, double omega);
/* Given 2 rotation matrices R1, R2, compute compound rotation matrix
 * Q = R1*R2
 *
 * Returns new RotMatrix
 */
RotMatrix compound_rotation(RotMatrix R1, RotMatrix R2);
/* Transpose a rotation matrix
 *
 * Returns new RotMatrix
 */
RotMatrix transpose_rotation(RotMatrix R);

/* Rotate a point (x,y,z) by a rotation matrix
 *
 * Does a matrix multiply of R*[x y z]' and
 * returns a new (x, y, z) point
 */
XyzPoint rotate_point(XyzPoint p, RotMatrix R);
/* Undo a 3-matrix compound rotation on a GeoPoint
 *
 * Returns a new GeoPoint that has been unrotated by (R3*R2*R1)^-1 * P
 */
GeoPoint unrotate_point(GeoPoint P, RotMatrix R1, RotMatrix R2, RotMatrix R3);

/* Convert from geographic coordinates (azimuth = longitude,
 * elevation = latitude, radius) to Cartesian coordinates (x, y, z)
 *
 * Angles in RADIANS!
 */
XyzPoint sph2xyz(GeoPoint P, double r);
/* Convert from Cartesian (x, y, z) coordinates to geographic
 * azimuth=longitude, elevation=latitude
 *
 * Angles in RADIANS!
 */
GeoPoint xyz2sph(XyzPoint P);
/* Convert from lon, lat, height to x,y,z in Earth-centered Earth-fixed
 * ref frame, but assuming WGS84 ellipsoid!
 *
 * lat, lon in RADIANS, h in m
 * Eqns from WGS84 reference paper from NGS
 */
XyzPoint llh2xyz(GeoPoint P, double h);

#endif
