/* Rotations and great circle operations on a sphere
 *
 * Useful for geographic stuff where need to move things along
 * the surface of a sphere.
 */
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

#include "spherical.h"
#include "constants.h"

/* convert degrees to radians and back */
GeoPoint deg2rad(GeoPoint A)
{
  GeoPoint B;
  B.lat = A.lat*DEG2RAD;
  B.lon = A.lon*DEG2RAD;
  return(B);
}
GeoPoint rad2deg(GeoPoint A)
{
  GeoPoint B;
  B.lat = A.lat*RAD2DEG;
  B.lon = A.lon*RAD2DEG;
  return(B);
}


/* Compute distance along the ellipsoid (WGS84) between 2 GeoPoints.
 *
 * lat, lon in radians!
 *
 * Returns distance, in m, between P and Q along the WGS84 ellipsoid surface.
 */
double ellipsoid_dist(GeoPoint P, GeoPoint Q)
{
  /* constants.h has WGS84_{a,b,f} for constants of WGS84 ellipsoid */
  double s=0, A=0, B=0, dSigma=0; 
  double sinL=0, cosL=0, sinS=0, cosS=0, sigma=0, sinA=0, cos2A=0, cos2Sm=0;
  double C=0, lambda=0, lold=0, U1=0, U2=0;
  double cosU2=0, cosU1=0, sinU1=0, sinU2=0;
  double u=0;
  int count=0;

  U1 = atan((1-WGS84_f)*tan(P.lat));
  U2 = atan((1-WGS84_f)*tan(Q.lat));

  /* fix points at the south pole:
   * tan(-PI/2) == tan(PI/2) ! <--WRONG!
   * ==> atan(tan(-PI/2)) = PI/2 not -PI/2!
   */
  if(fabs(P.lat+PI/2) < 1e-6) { /* tolerance due to finite precision in data files */
    U1 *= -1;
    }
  if(fabs(Q.lat+PI/2) < 1e-6) {
    U2 *= -1;
    }

  sinU1 = sin(U1); cosU1 = cos(U1);
  sinU2 = sin(U2); cosU2 = cos(U2);
  lambda = Q.lon-P.lon;
  do {
    sinL = sin(lambda); cosL = cos(lambda);
    u=cosU1*sinU2 - sinU1*cosU2*cosL;
    sinS = sqrt( (cosU2*sinL*cosU2*sinL) + u*u );
    if(sinS == 0) return(0.0);	/* coincident points */
    cosS = sinU1*sinU2 + cosU1*cosU2*cosL;
    sigma = atan2(sinS, cosS);
    sinA = cosU1*cosU2*sinL / sinS;
    cos2A = 1 - sinA*sinA;
    cos2Sm = cosS - 2*sinU1*sinU2/cos2A;
    if(isnan(cos2Sm)) { cos2Sm=0; }
    C = WGS84_f/16*cos2A* ( 4+WGS84_f*(4-3*cos2A) );
    lold=lambda;
    lambda = Q.lon-P.lon + (1-C)*WGS84_f*sinA*( sigma+C*sinS*( cos2Sm+C*cosS*(-1+2*cos2Sm*cos2Sm) ) );
    count++;
    } while(fabs(lambda-lold) > 1e-12 && count<100);
  u = cos2A*(WGS84_a*WGS84_a - WGS84_b*WGS84_b) / (WGS84_b*WGS84_b);
  A = 1 + u/16384 * (4096+u*(-768+u*(320-175*u)));
  B = u/1024* (256+u*(-128+u*(74-47*u)));
  dSigma = B*sinS*( cos2Sm+B/4*(cosS*(-1+2*cos2Sm*cos2Sm)-B/6*cos2Sm*(-3+4*sinS*sinS)*(-3+4*cos2Sm*cos2Sm)));
  s = WGS84_b*A*(sigma - dSigma);
  return(s);
}

/* Compute radius of the WGS84 ellipsoid at the specified location
 *
 * lat, lon in radians!
 *
 * returns the radius in m
 */
double ellipsoid_radius(GeoPoint A)
{
  double R;
  R = 1.0/sqrt( sin(A.lat)*sin(A.lat)/(WGS84_b*WGS84_b) + cos(A.lat)*cos(A.lat)/(WGS84_a*WGS84_a) );
  return(R);
}

/* return a 3x3 rotation matrix to rotate about the Euler pole at
 * lon,lat by the angle omega, in radians
 *
 * returned pointer is to a newly allocated matrix, so don't forget
 * to free after use!
 */
RotMatrix rotation_matrix(GeoPoint Q, double omega)
{
  RotMatrix R;
  XyzPoint P;

  P = sph2xyz(Q, 1);

  R.v[0][0] = P.x*P.x*(1 - cos(omega)) + cos(omega);
  R.v[0][1] = P.x*P.y*(1 - cos(omega)) - P.z*sin(omega);
  R.v[0][2] = P.x*P.z*(1 - cos(omega)) + P.y*sin(omega);

  R.v[1][0] = P.y*P.x*(1 - cos(omega)) + P.z*sin(omega);
  R.v[1][1] = P.y*P.y*(1 - cos(omega)) + cos(omega);
  R.v[1][2] = P.y*P.z*(1 - cos(omega)) - P.x*sin(omega);

  R.v[2][0] = P.z*P.x*(1 - cos(omega)) - P.y*sin(omega);
  R.v[2][1] = P.z*P.y*(1 - cos(omega)) + P.x*sin(omega);
  R.v[2][2] = P.z*P.z*(1 - cos(omega)) + cos(omega);

  return(R);
}

/* Given 2 rotation matrices R1, R2, compute compound rotation matrix
 * Q = R1*R2
 *
 * Returns new RotMatrix
 */
RotMatrix compound_rotation(RotMatrix R1, RotMatrix R2)
{
  RotMatrix Q;
  int i, j, k;

  for(i=0; i<3; i++) {
    for(j=0; j<3; j++) {
      Q.v[i][j] = 0;
      for(k=0; k<3; k++) {
	Q.v[i][j] += R1.v[i][k]*R2.v[k][j];
	}
      }
    }
  return(Q);
}

/* Transpose a rotation matrix
 *
 * Returns new RotMatrix
 */
RotMatrix transpose_rotation(RotMatrix R)
{
  RotMatrix Q;
  int i,j;

  for(i=0; i<3; i++) {
    for(j=0; j<3; j++) {
      Q.v[j][i] = R.v[i][j];
      }
    }
  return(Q);
}

/* Convert from geographic coordinates (azimuth = longitude,
 * elevation = latitude, radius) to Cartesian coordinates (x, y, z)
 *
 * lat, lon in radians!
 *
 * Angles in RADIANS!
 */
XyzPoint sph2xyz(GeoPoint P, double r)
{
  XyzPoint Q;
  /* x = r*sin(colat)*cos(az) ==> r*cos(lat)*cos(az) */
  Q.x = r*cos(P.lat)*cos(P.lon);
  /* y = r*sin(colat)*sin(az) */
  Q.y = r*cos(P.lat)*sin(P.lon);
  /* z = r*cos(colat) */
  Q.z = r*cos(PI/2-P.lat);

  return(Q);
}
/* Convert from Cartesian (x, y, z) coordinates to geographic
 * azimuth=longitude, elevation=latitude
 *
 * lat, lon in radians!
 *
 * Angles in RADIANS!
 */
GeoPoint xyz2sph(XyzPoint P)
{
  GeoPoint Q;
  /* colat = arccos(z/sqrt(x^2+y^2+z^2)) */
  Q.lat = PI/2 - (acos(P.z/sqrt(P.x*P.x+P.y*P.y+P.z*P.z)));
  /* az = arcsin(y/sqrt(x^2+y^2)) if x>=0, pi-arcsin() if x<0 */
  Q.lon = ( (P.x<0) ? PI-asin(P.y/sqrt(P.x*P.x+P.y*P.y)) : asin(P.y/sqrt(P.x*P.x+P.y*P.y)) );

  return(Q);
}
/* Convert from lon, lat, height to x,y,z in Earth-centered Earth-fixed
 * ref frame, but assuming WGS84 ellipsoid!
 *
 * lat, lon in RADIANS, h in m
 * Eqns from WGS84 reference paper from NGS
 */
XyzPoint llh2xyz(GeoPoint P, double h)
{
  double N;
  XyzPoint Q;

  N = WGS84_a / sqrt(1-WGS84_e2*sin(P.lon)*sin(P.lon));

  // cos(P.lat) == sin(colat(P.lat))!
  Q.x = (N+h)*cos(P.lat)*cos(P.lon);
  Q.y = (N+h)*cos(P.lat)*sin(P.lon);
  Q.z = (WGS84_b*WGS84_b/(WGS84_a*WGS84_a)*N+h)*sin(P.lat);

  return(Q);
}

/* Rotate a point (x,y,z) by a rotation matrix
 *
 * Does a matrix multiply of R*[x y z]' and
 * returns a new (x, y, z) point
 */
XyzPoint rotate_point(XyzPoint p, RotMatrix R)
{
  XyzPoint q;

  q.x = R.v[0][0]*p.x + R.v[0][1]*p.y + R.v[0][2]*p.z;
  q.y = R.v[1][0]*p.x + R.v[1][1]*p.y + R.v[1][2]*p.z;
  q.z = R.v[2][0]*p.x + R.v[2][1]*p.y + R.v[2][2]*p.z;

  return(q);
}

/* Undo a 3-matrix compound rotation on a GeoPoint
 *
 * lat, lon in radians!
 *
 * Returns a new GeoPoint that has been unrotated by (R3*R2*R1)^-1 * P
 */
GeoPoint unrotate_point(GeoPoint P, RotMatrix R1, RotMatrix R2, RotMatrix R3)
{
  GeoPoint Q;
  RotMatrix R;

  /* construct transposed compound rotation matrix R = R3*R2*R1 */
  R = transpose_rotation( compound_rotation(R3,compound_rotation(R2,R1)) );

  /* unrotate by rotating by transposed compound rotation */
  Q = xyz2sph( rotate_point( sph2xyz(P,1), R) );

  return(Q);
}


/* Compute a point partway along a great circle path
 *
 * Returns a new GeoPoint that is partway between P and Q;
 * alpha between [0,1] is the fraction of the path along PQ.
 * lat, lon in radians!
 *
 */
GeoPoint greatCirclePt(GeoPoint P, GeoPoint Q, double alpha)
{
  RotMatrix R1, R2, R3;
  GeoPoint Q1, Q2, nP;	/* Euler poles, return value */
  GeoPoint Pp, Qp;	/* new pt in rotated scheme, rotated end of arc */
  double omega;

  Q1.lon = 0; Q1.lat = -PI/2;
  Q2.lon = PI/2; Q2.lat = 0;
  R1 = rotation_matrix(Q1, P.lon);
  R2 = rotation_matrix(Q2, P.lat);

  /* rotate Q by matrix multiplication Qp = xyz2sph( R2*R1*sph2xyz(Q) ) */
  Qp = xyz2sph( rotate_point( rotate_point(sph2xyz(Q,1),R1), R2) );

  /* compute angle between equator and Q'; combination of 2 spherical angle
   * calculations for a triangle.  See Zwillinger, D. 1996,
   * Standard Mathematical Tables and Formulae, CRC Press, p. 468.
   */
  omega = acos( tan(Qp.lon)/tan( acos( cos(Qp.lon)*cos(Qp.lat) ) ) );
  if(Qp.lat > 0) omega *= -1;

  /* compute rotation matrix to bring Qp to equator */
  Q1.lat=0; Q1.lon=0;
  R3 = rotation_matrix(Q1, omega);

  /* rotate Qp to equator, check rotation result: Qp.lat=0 */
  Qp = xyz2sph( rotate_point( rotate_point( rotate_point(sph2xyz(Q,1),R1), R2), R3) );
  if (Qp.lat > 1e-13) {fprintf(stderr, "greatCirclePt: rotation matrices might be wrong!\n");}

  /* compute new point in rotated frame; fraction of arc from 0,0 to Qp.lon,0 */
  Pp.lat = 0;
  Pp.lon = Qp.lon*alpha;

  nP = unrotate_point(Pp, R1, R2, R3);

  return(nP);
}

/* Compute points partway along a great circle path
 *
 * Returns an array of GeoPoints that are partway between P and Q;
 * alpha between [0,1] is the fraction of the path along PQ.
 * n is number of points in alpha[]
 * Don't forget to free the returned array!
 * lat, lon in radians!
 *
 */
GeoPoint *greatCirclePtAry(GeoPoint P, GeoPoint Q, double *alpha, int n)
{
  RotMatrix R1, R2, R3;
  GeoPoint Q1, Q2, *nP;	/* Euler poles, return value */
  GeoPoint Pp, Qp;	/* new pt in rotated scheme, rotated end of arc */
  double omega;
  int i;

  Q1.lon = 0; Q1.lat = -PI/2;
  Q2.lon = PI/2; Q2.lat = 0;
  R1 = rotation_matrix(Q1, P.lon);
  R2 = rotation_matrix(Q2, P.lat);

  /* rotate Q by matrix multiplication Qp = xyz2sph( R2*R1*sph2xyz(Q) ) */
  Qp = xyz2sph( rotate_point( rotate_point(sph2xyz(Q,1),R1), R2) );

  /* compute angle between equator and Q'; combination of 2 spherical angle
   * calculations for a triangle.  See Zwillinger, D. 1996,
   * Standard Mathematical Tables and Formulae, CRC Press, p. 468.
   */
  omega = acos( tan(Qp.lon)/tan( acos( cos(Qp.lon)*cos(Qp.lat) ) ) );
  if(Qp.lat > 0) omega *= -1;

  /* compute rotation matrix to bring Qp to equator */
  Q1.lat=0; Q1.lon=0;
  R3 = rotation_matrix(Q1, omega);

  /* rotate Qp to equator, check rotation result: Qp.lat=0 */
  Qp = xyz2sph( rotate_point( rotate_point( rotate_point(sph2xyz(Q,1),R1), R2), R3) );
  if (Qp.lat > 1e-13) {fprintf(stderr, "greatCirclePtAry: rotation matrices might be wrong!");}

  /* allocate mem for returned array of GeoPoints */
  nP = (GeoPoint *)malloc(sizeof(GeoPoint)*n);
  if(nP==NULL) return(nP);

  /* compute new points in rotated frame; fraction of arc from 0,0 to Qp.lon,0 */
  for(i=0; i<n; i++) {
    Pp.lat = 0;
    Pp.lon = Qp.lon*alpha[i];
    nP[i] = unrotate_point(Pp, R1, R2, R3);
    }

  return(nP);
}

/* Compute a point partway along a great circle path
 *
 * Returns a new GeoPoint that is partway between P and Q;
 * new point is along great circle where longitude is specified
 * lat, lon in radians!
 *
 */
GeoPoint greatCirclePtLon(GeoPoint P, GeoPoint Q, double lon)
{
  RotMatrix R1, R2, R3;
  GeoPoint Q1, Q2, nP;	/* Euler poles, return value */
  GeoPoint Pp, Qp;	/* new pt in rotated scheme, rotated end of arc */
  double omega;

  Q1.lon = 0; Q1.lat = -PI/2;
  Q2.lon = PI/2; Q2.lat = 0;
  R1 = rotation_matrix(Q1, P.lon);
  R2 = rotation_matrix(Q2, P.lat);

  /* rotate Q by matrix multiplication Qp = xyz2sph( R2*R1*sph2xyz(Q) ) */
  Qp = xyz2sph( rotate_point( rotate_point(sph2xyz(Q,1),R1), R2) );

  /* compute angle between equator and Q'; combination of 2 spherical angle
   * calculations for a triangle.  See Zwillinger, D. 1996,
   * Standard Mathematical Tables and Formulae, CRC Press, p. 468.
   */
  omega = tan(Qp.lon)/tan( acos( cos(Qp.lon)*cos(Qp.lat) ) );
  if(omega > 1) {
    omega=1; // if omega>1, prevent NaN from acos() call
    }
  omega = acos( omega );
  if(Qp.lat > 0) omega *= -1;

  /* compute rotation matrix to bring Qp to equator */
  Q1.lat=0; Q1.lon=0;
  R3 = rotation_matrix(Q1, omega);

  /* rotate Qp to equator, check rotation result: Qp.lat=0 */
  Qp = xyz2sph( rotate_point( rotate_point( rotate_point(sph2xyz(Q,1),R1), R2), R3) );
  //if (Qp.lat > 1e-11) {fprintf(stderr, "greatCirclePtLon: Qp.lat=%.8e ==> rotation matrices might be wrong!\n", Qp.lat);}


  /* compute new point in rotated frame; fraction of arc from 0,0 to Qp.lon,0
   * fraction found from fraction between unrotated lon difference of P, Q
   */
  Pp.lat = 0;
  Pp.lon = Qp.lon*(lon-P.lon)/(Q.lon-P.lon);

  nP = unrotate_point(Pp, R1, R2, R3);

  return(nP);
}

