/*
 * octtree.h
 *
 * header file for octree.c
 *
 * This file is part of micrograv.
 *
 * micrograv is copyright (c) 2000-2003 by
 * Paul Gettings,
 * Department of Geology & Geophysics,
 * University of Utah.
 *
 * All Rights Reserved.
 * 
 * micrograv is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * micrograv is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */
#ifndef OCTREE_H
#define OCTREE_H

#include "global_grid/geopoly_types.h"

/* flag to indicate no parents */
#define ROOT_NODE	NULL

/* Type Range holds max/min ranges in x, y, z for building the
 * oct-tree.  minResX is the minimum resolution the tree will reach
 * in x before assigning a leaf node.  Similar for minResY or Z.
 * This is used internally, and need not normally be directly touched.
 */
typedef struct RangeStruct {
  double xmax, xmin;
  double ymax, ymin;
  double zmax, zmin;
  double minResX;
  double minResY;
  double minResZ;
  } Range;

/* Type OctTree holds the 8 possible branches below this point in the
 * tree; a full oct-tree is a linked list of these pointers, starting
 * at the root branch.  If this instance is a leaf (no branches down),
 * then Leaf is true.
 */
typedef struct OctTreeStruct {
  void *branches[8];
  char Leaf;	/* 1 if leaf node */
  Range limits; /* min, max x, y, z values of this node */
  } OctTree;


  /* Generate an octree from the supplied GlobalGrid, with nodes
   * subdividing to no more than xres x yres x zres m
   */
OctTree *genOctTree(GlobalGrid data, double xres, double yres, double zres);

  /* Given an oct-tree, such as from genOctTree(), and a location,
   * creates a list of polygons within a range
   */
GlobalGrid octTreeToGrid(OctTree *tree, XyzPoint loc, double range);

  /* When done with the oct-tree, call freeOctTree() to clean up and
   * release memory in the tree branches; this will free the leaves as
   * well, so extract information from the tree first!
   */
void freeTree(OctTree *tree);
  /* When done with a DataList, free the whole list, including the
   * starting pointer. [Calls freeListEntries() and then free(ptr)]
   */

XyzLinkedList *traverseTree(OctTree *tree, XyzPoint loc, double range, XyzLinkedList *list);
double dist(XyzPoint A, XyzPoint B);
int locationCheck(XyzPoint loc, Range limits, double dist);
void resetDepth(void);
unsigned long countLeaves(OctTree *tree);
OctTree *allocateBranch(Range limits);
int insertLeaf(XyzPoly P, OctTree *tree);
void copyPoly(XyzPoly from, XyzPoly *to);

#endif
