/*
 * octree.c
 *
 * Create an octree from a list of DGG polygons. This tree is then used
 * to make a list of polygons within a specified distance of a given
 * (station) location. Build the list by checking limits of nodes
 * (starting at root); if no corner of node is within range, drop node
 * and all children.
 *
 * This file is part of micrograv
 *
 * micrograv is copyright (c) 1998-2011 by
 * Paul Gettings
 *
 * All Rights Reserved.
 * 
 * micrograv is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 * 
 * micrograv is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details, in the file
 * COPYING.  If that file is not present, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA
 * 
 */
#include <stdio.h>
#include <stdlib.h>
#include "octree.h"
#include "spherical.h"

/* tree counters */
int depth=0;
long leaves=0;


/* Convert a tree to a list of elements within range of loc
 * Returns a pointer to the new list
 */
GlobalGrid octTreeToGrid(OctTree *tree, XyzPoint loc, double range)
{
  XyzLinkedList *list, *iter, *prev;
  GlobalGrid out;
  uint64 n=0;

  depth=0;
  list = (XyzLinkedList *)malloc(sizeof(XyzLinkedList));
  list->next = NULL; // clear next pointer, in case memory not empty
  iter = list;

  /* traverse the tree, finding nodes within the range */
  iter = traverseTree(tree, loc, range, iter);

  /* make sure end of list is empty */
  if(iter->next!=NULL) free(iter->next); // free after last block if present
  iter->next = NULL;

  /* now, convert linked list of XyzPolys to GlobalGrid */
  iter = list;
  while(iter!=NULL){
    n++;
    iter=iter->next;
    }
  out.npolys = n;
  out.polys = (GlobalGridPoly *)malloc(sizeof(GlobalGridPoly)*out.npolys);
  out.vals = (float *)malloc(sizeof(float)*out.npolys);
  out.mask = (short int *)malloc(sizeof(short int)*out.npolys);
  iter = list; n=0;
  while(iter!=NULL){
    out.polys[n] = iter->poly.orig;
    out.vals[n] = iter->poly.value;
    out.mask[n] = iter->poly.mask;
    n++;
    iter=iter->next;
    }
  /* free linked list */
  iter = list;
  while(iter != NULL){
    prev=iter;
    iter=iter->next;
    free(prev);
    }

  out.npolys--; // drop last entry which isn't filled
  return(out);
}

/* Traverses a tree, creating a list of elements from each leaf in range
 * Returns the list of elements, which are malloc'd from free memory
 * Returns NULL on error
 */
XyzLinkedList *traverseTree(OctTree *tree, XyzPoint loc, double range, XyzLinkedList *list)
{
  int i;
  XyzPoly *d;
  XyzPoint P;

  depth++;

  if(tree->Leaf) { /* Leaf node, so check if in range */
    leaves++;
    d = (XyzPoly *)tree->branches[0];
    P.x = d->x; P.y=d->y; P.z=d->z;
    if(dist(P, loc) <= range*range) {
      list->poly = *d;
      if((list->next = (void *)malloc(sizeof(XyzLinkedList))) == NULL) {
	fprintf(stderr, "octree.c: cannot allocate memory for next data list entry.\n");
	return(NULL);
	}
      list = (XyzLinkedList *)list->next;
      list->next = NULL; // clear next pointer, in case memory isn't empty
      return(list);
      }
    }
  else { /* check node to see if any corner in range, or location inside node limits */
    if(locationCheck(loc, tree->limits, range)) {
      for(i=0; i<8; i++) { /* check children */
	if(tree->branches[i] == NULL) {
	  continue;
	  }
	else { /* check children that exist */
	  list = traverseTree((OctTree *)tree->branches[i], loc, range, list);
	  }
	}
      }
    }
  depth--;
  return(list);
}

/* distance from A, B */
double dist(XyzPoint A, XyzPoint B)
{
  return((A.x-B.x)*(A.x-B.x) + (A.y-B.y)*(A.y-B.y) + (A.z-B.z)*(A.z-B.z));
}

/* check location is inside range, or at least one corner in range */
int locationCheck(XyzPoint loc, Range limits, double range)
{
  XyzPoint A;
  if(limits.xmin<=loc.x && loc.x<=limits.xmax &&
     limits.ymin<=loc.y && loc.y<=limits.ymax &&
     limits.zmin<=loc.z && loc.z<=limits.zmax) return(1);
  A.x = limits.xmin; A.y = limits.ymin; A.z = limits.zmin; if(dist(A, loc)<=range*range) return(1);
  A.x = limits.xmax; A.y = limits.ymin; A.z = limits.zmin; if(dist(A, loc)<=range*range) return(1);
  A.x = limits.xmin; A.y = limits.ymax; A.z = limits.zmin; if(dist(A, loc)<=range*range) return(1);
  A.x = limits.xmax; A.y = limits.ymax; A.z = limits.zmin; if(dist(A, loc)<=range*range) return(1);
  A.x = limits.xmin; A.y = limits.ymin; A.z = limits.zmax; if(dist(A, loc)<=range*range) return(1);
  A.x = limits.xmax; A.y = limits.ymin; A.z = limits.zmax; if(dist(A, loc)<=range*range) return(1);
  A.x = limits.xmin; A.y = limits.ymax; A.z = limits.zmax; if(dist(A, loc)<=range*range) return(1);
  A.x = limits.xmax; A.y = limits.ymax; A.z = limits.zmax; if(dist(A, loc)<=range*range) return(1);
  return(0);
}

/* Reset the depth level counter for tree traversal, etc.
 */
void resetDepth(void)
{
  depth = 0;
}

/* Traverses a tree, counting the number of leaves
 * Returns the count of leaves
 */
unsigned long countLeaves(OctTree *tree)
{
  int i;

  if(depth == 0) {
    leaves = 0; /* reset the count */
    }
  depth++;

  if(tree->Leaf) {
    leaves++;
    }
  else {
    for(i=0; i<8; i++) {
      if(tree->branches[i] == NULL) {
        continue;
        }
      else {
        countLeaves((OctTree *)tree->branches[i]);
        }
      }
    }
  depth--;
  return(leaves);
}

/* Free an entire OctTree, including all branches and leaves
 */
void freeTree(OctTree *tree)
{
  int i;

  depth++;

  if(tree->Leaf) {
    free(tree->branches[0]);
    if(depth == 1)
      free(tree);
    return;
    }
  else {
    for(i=0; i<8; i++) {
      if(tree->branches[i] == NULL) {
        continue;
        }
      else {
        freeTree((OctTree *)tree->branches[i]);
        free(tree->branches[i]);
        }
      }
    }
  depth--;
  return;
}

/* Allocates a new branch for the oct-tree, setting some parameters
 * Returns a pointer to the new branch node
 */
OctTree *allocateBranch(Range limits)
{
  OctTree *b;
  int i;

  if((b = (OctTree *)malloc(sizeof(OctTree))) == NULL) {
    return(NULL);
    }
  for(i=0; i<8; i++) {
    b->branches[i] = NULL;
    }
  b->Leaf = 0;
  b->limits = limits; /* min, max x,y,z */
  return(b);
}

/* Generates an oct-tree from an XyzGrid of polygons; minimum
 * resolution is xres, yres, zres
 * Returns NULL on error
 */
OctTree *genOctTree(GlobalGrid dem, double xres, double yres, double zres)
{
  Range extents;
  XyzGrid data;
  XyzPoint xyz;
  OctTree *tree;
  uint64 i;
  long ne;
  double elev;

  data.polys = (XyzPoly *)malloc(sizeof(XyzPoly)*dem.npolys);
  data.npolys = dem.npolys;
  for(i=0; i<dem.npolys; i++) { /* convert from lon, lat, h to x,y,z */
    /* N.B. - use assumed height of ellipsoid; that way can find polys
     * in range without worrying about elevations....
     * This comes from the fact that when over oceans, elevations will
     * be -3 to -10 km! So, use elevation of 0 for finding location of
     * grid poly & building octree; will use actual elevation for calcs...
     */
    xyz = llh2xyz(deg2rad(dem.polys[i].center), 0);
    data.polys[i].x = xyz.x;
    data.polys[i].y = xyz.y;
    data.polys[i].z = xyz.z;
    data.polys[i].id = dem.polys[i].id;
    data.polys[i].orig = dem.polys[i];
    data.polys[i].value = dem.vals[i];
    data.polys[i].mask = dem.mask[i];
    }

  /* get range of interest */
  extents.xmax = extents.xmin = data.polys[0].x;
  extents.ymax = extents.ymin = data.polys[0].y;
  extents.zmax = extents.zmin = data.polys[0].z;
  extents.minResX = xres;
  extents.minResY = yres;
  extents.minResZ = zres;
  for(i=1; i<data.npolys; i++) {
    if(data.polys[i].x > extents.xmax) extents.xmax = data.polys[i].x;
    if(data.polys[i].x < extents.xmin) extents.xmin = data.polys[i].x;
    if(data.polys[i].y > extents.ymax) extents.ymax = data.polys[i].y;
    if(data.polys[i].y < extents.ymin) extents.ymin = data.polys[i].y;
    if(data.polys[i].z > extents.zmax) extents.zmax = data.polys[i].z;
    if(data.polys[i].z < extents.zmin) extents.zmin = data.polys[i].z;
    }

  if((tree = allocateBranch(extents)) == NULL) {
    fprintf(stderr, "octree.c: cannot allocate root branch.\n");
    return(NULL);
    }

  /* build the tree */
  ne = 0; depth=0;
  for(i=0; i<data.npolys; i++) {
    ne++;
    if(!insertLeaf(data.polys[i], tree)) {
      fprintf(stderr, "octree.c: error inserting element %ld into tree.\n", ne);
      return(NULL);
      }
    }

  return(tree);
}

/* Inserts a leaf node into the tree, creating branches as necessary.
 * Returns 0 on failure.
 */
int insertLeaf(XyzPoly P, OctTree *tree)
{
  Range r, R;
  int o, rc;
  double midx, midy, midz;

  /* local copy of current tree limits */
  r = tree->limits;

  depth++;
  if((r.xmax - r.xmin)<=r.minResX && (r.ymax-r.ymin)<=r.minResY &&
    (r.zmax-r.zmin)<=r.minResZ) { /* leaf node */
    if(tree->branches[0] != NULL) {	// replace existing leaf
      free(tree->branches[0]);
      }
    if((tree->branches[0] = malloc(sizeof(XyzPoly))) == NULL) {
      depth--;
      return(0);
      }
    copyPoly(P, (XyzPoly *)tree->branches[0]);
    tree->Leaf=1;
    depth--;
    return(1);
    }
  else { /* another branch */
    R.xmax = r.xmax; R.ymax = r.ymax; R.zmax = r.zmax;
    R.xmin = r.xmin; R.ymin = r.ymin; R.zmin = r.zmin;
    R.minResX = r.minResX; R.minResY = r.minResY; R.minResZ = r.minResZ;
    midx = (r.xmax + r.xmin)/2.0;
    midy = (r.ymax + r.ymin)/2.0;
    midz = (r.zmax + r.zmin)/2.0;

    /* find the octant */
    o = 0;
    if(P.x > midx) o += 1;
    if(P.y > midy) o += 2;
    if(P.z > midz) o += 4;

    switch(o) {
      case 0: R.xmax = midx; R.ymax = midy; R.zmax = midz; break;
      case 1: R.xmin = midx; R.ymax = midy; R.zmax = midz; break;
      case 2: R.xmax = midx; R.ymin = midy; R.zmax = midz; break;
      case 3: R.xmin = midx; R.ymin = midy; R.zmax = midz; break;
      case 4: R.xmax = midx; R.ymax = midy; R.zmin = midz; break;
      case 5: R.xmin = midx; R.ymax = midy; R.zmin = midz; break;
      case 6: R.xmax = midx; R.ymin = midy; R.zmin = midz; break;
      case 7: R.xmin = midx; R.ymin = midy; R.zmin = midz; break;
      default:
              depth--;
              return(0);
              break;
      }
    if(tree->branches[o] == NULL) { /* allocate the new branch */
      if((tree->branches[o] = allocateBranch(R)) == NULL) {
	depth--;
	return(0);
	}
      }
    rc = insertLeaf(P, (OctTree *)tree->branches[o]);
    depth--;
    return(rc);
    }
}

/* Copy values from one polygon to another
 * Assumes *to is allocated!
 */
void copyPoly(XyzPoly from, XyzPoly *to)
{
  to->id = from.id;
  to->x = from.x;
  to->y = from.y;
  to->z = from.z;
  to->orig = from.orig;
  to->value = from.value;
  to->mask = from.mask;
  return;
}

