#ifndef GEOGRAPHIC_POLYGON_GRIDS_H
#define GEOGRAPHIC_POLYGON_GRIDS_H

#include "global_grid/geopoly_types.h"

/* Structure to hold a geographic polygon, including
 * id number (num), center/centroid coords, vertex coords
 *
 * used a building block inside GlobalGrid, etc.
 */
typedef struct geographicPolygonStruct {
  unsigned long num;
  unsigned nvert;
  GeoPoint *verts;
  GeoPoint center;
  } GeoPolygon;

/* Store a Discrete Global Grid
 *
 * Holds only polygon information and single radius for
 * conversion to Cartesian coords.  Does not hold any
 * data values for polygons!
 */
typedef struct discreteGlobalGridStruct {
  unsigned int nPolys;
  GeoPolygon *P;
  double R; /* mean radius of Earth in m */
  } DiscreteGlobalGrid;

/* Store a set of values associated with GeoPolygons in a GlobalGrid
 *
 * n is number of entries in pnum[], val[]
 * pnum is array of polygon indices into GlobalGrid.P[]
 * val is array of values for the matching pnum entry.
 *
 * use for observed gravity, etc.
 */
typedef struct discreteGlobalValGridStruct {
  unsigned int n;
  unsigned int *pnum;
  double *val;
  } DiscreteGlobalGridVals;

/* Structure to hold a polygon, including
 * id number (num), center/centroid coords, vertex coords
 *
 * used for spherical to cartesian transformations of DGGs
 */
typedef struct cartesianPolygonStruct {
  unsigned long num;
  unsigned nvert;
  XyzPoint *verts;
  XyzPoint center;
  } Polygon;

/* Store a Discrete Global Grid in Cartesian coords
 *
 * Holds only polygon information and single radius for
 * conversion to Cartesian coords.  Does not hold any
 * data values for polygons!
 */
typedef struct cartesianDGGridStruct {
  unsigned int nPolys;
  Polygon *P;
  } CartesianDGG;

/* Load data values for discrete global grid
 *
 * File should be in dggrid binning format:
 * poly_num value
 * ...	    ...
 *
 * returns a GlobalGridVals struct.
 */
DiscreteGlobalGridVals loadGridValues(char *filename);

/* Load polygons from file into an array
 *
 * file must be in geopolygon format:
 * npolys
 * poly_num nverts clon clat vlon vlat ... ...
 */
DiscreteGlobalGrid loadGridPolygons(char *filename);

/* Copy a GeoPolygon
 *
 * Allocates memory for vertices in GeoPolygon *to!
 */
void copyGeoPolygon(GeoPolygon from, GeoPolygon *to);

/* Rotate a single GeoPolygon to centered at (0,0), with 2 opposite vertices
 * on the equator; also rotates a single GeoPoint with the same compound rotations.
 *
 * returns a new GeoPolygon; new GeoPoint stored in *Op
 */
GeoPolygon rotatePolyAndPt(GeoPolygon P, GeoPoint O, GeoPoint *Op);

/* convert a discrete global grid in geographic coords
 * to cartesian coordinates, e.g. for plotting
 *
 */
CartesianDGG dgg2cartgrid(DiscreteGlobalGrid G);


#endif
