#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "fileio.h"
#include "spherical.h"

/* Load data values for discrete global grid
 *
 * File should be in dggrid binning format:
 * poly_num value
 * ...	    ...
 *
 * returns a GlobalGridVals struct.
 */
DiscreteGlobalGridVals loadGridValues(char *filename)
{
  FILE *fp;
  char *rec;
  unsigned i, np;
  DiscreteGlobalGridVals data;

  if((fp=fopen(filename, "rt")) == NULL) {
    fprintf(stderr, "loadGridValues: can't open file %s\n", filename);
    exit(1);
    }

  i=0;
  while(!feof(fp)) {
    rec = getrecord(fp);
    if(rec == NULL) continue;
    i++;
    }
  rewind(fp);
  np = i;

  data.pnum = (unsigned *)malloc(sizeof(int)*np);
  data.val = (double *)malloc(sizeof(double)*np);
  if(data.pnum==NULL || data.val==NULL) {
    fprintf(stderr, "loadGridValues: can't allocate memory for grid values.\n");
    exit(1);
    }

  data.n = np;

  i=0;
  while(!feof(fp)) {
    rec = getrecord(fp);
    if(rec == NULL) continue;
    sscanf(rec, "%u %lf", &(data.pnum[i]), &(data.val[i]));
    i++;
    }

  fclose(fp);
  return(data);
}


/* Load polygons from file into an array
 *
 * file must be in geopolygon format:
 * npolys
 * poly_num nverts clon clat vlon vlat ... ...
 */
DiscreteGlobalGrid loadGridPolygons(char *filename)
{
  FILE *fp;
  unsigned int i, np, nv;
  unsigned int ind;
  GeoPolygon *polys;
  DiscreteGlobalGrid grid;
  char *rec, tmp[1025], *ptr;
  double lat, lon;

  if((fp=fopen(filename, "rt")) == NULL) {
    fprintf(stderr, "loadPolygons: can't open file %s\n", filename);
    exit(1);
    }
  /* grab the number of polys to read */
  while(!feof(fp)) {
    rec = getrecord(fp);
    if(rec == NULL) continue;
    sscanf(rec, "%ud", &np);
    break;
    }

  /* allocate memory for polys */
  polys = (GeoPolygon *)malloc(sizeof(GeoPolygon)*np);
  if(polys==NULL) {
    fprintf(stderr, "loadPolygons: can't allocate memory for polygon load.\n");
    exit(1);
    }
  grid.nPolys = np;
  grid.P = polys;
  grid.R = 6371008.771; /* average radius of WGS-84 Earth, in m */
  /* read polygons from file */
  while(!feof(fp)) {
    rec = getrecord(fp);
    if(rec == NULL) continue;
    /* read a poly */
    sscanf(rec, "%u %u %lf %lf", &ind, &nv, &lon, &lat);
    if(ind > np) { fprintf(stderr, "loadPolygons: Polygon number > number of polygons!\n"); exit(1); }
    ind--; /* poly number starts at 1, not 0 */
    polys[ind].nvert = nv;
    polys[ind].num = ind+1;
    polys[ind].center.lon = lon*DEG2RAD;
    polys[ind].center.lat = lat*DEG2RAD;
    polys[ind].verts = (GeoPoint *)malloc(sizeof(GeoPoint)*nv);
    if(polys[ind].verts==NULL) {
      fprintf(stderr, "loadPolygons: unable to allocate memory for polygon.\n");
      exit(1);
      }
    /* read vertices from record */
    strncpy(tmp, rec, 1024);
    ptr = strtok(tmp, " ");
    /* skip through poly_n, nverts, center lon, lat */
    ptr = strtok(NULL, " ");
    ptr = strtok(NULL, " ");
    ptr = strtok(NULL, " ");
    ptr = strtok(NULL, " ");
    /* grab tokens for vertex lon, lat */
    for(i=0; i<polys[ind].nvert; i++) {
      if(ptr == NULL) { /* don't run off the end of the string */
        fprintf(stderr, "loadPolygons: error parsing polygon for vertices\n");
        exit(1);
        }
      sscanf(ptr, "%lf", &(polys[ind].verts[i].lon));
      ptr = strtok(NULL, " ");
      sscanf(ptr, "%lf", &(polys[ind].verts[i].lat));
      ptr = strtok(NULL, " ");
      polys[ind].verts[i].lon *= DEG2RAD;
      polys[ind].verts[i].lat *= DEG2RAD;
      }
    }
  fclose(fp);

  return(grid);
}


/* Copy a GeoPolygon
 *
 * Allocates memory for vertices in GeoPolygon *to!
 */
void copyGeoPolygon(GeoPolygon from, GeoPolygon *to)
{
  unsigned int i;

  to->num = from.num;
  to->nvert = from.nvert;
  to->center.lon = from.center.lon;
  to->center.lat = from.center.lat;
  to->verts = (GeoPoint *)malloc(sizeof(GeoPoint)*from.nvert);
  if(to->verts==NULL) {
    fprintf(stderr, "copyGeoPolygon: can't allocate memory for vertices. Die; badly.\n");
    exit(2);
    }
  for(i=0; i<from.nvert; i++) {
    to->verts[i].lon = from.verts[i].lon;
    to->verts[i].lat = from.verts[i].lat;
    }
}


/* Rotate a single GeoPolygon to centered at (0,0), with 2 opposite vertices
 * on the equator; also rotates a single GeoPoint with the same compound rotations.
 *
 * returns a new GeoPolygon; new GeoPoint stored in *Op
 */
GeoPolygon rotatePolyAndPt(GeoPolygon P, GeoPoint O, GeoPoint *Op)
{
  RotMatrix R1, R2;
  GeoPolygon Q;
  GeoPoint A, Q1, Q2;
  double omega;
  unsigned int i;

  /* rotate center of P to 0,0 */
  Q1.lon=0; Q1.lat=-PI/2;
  Q2.lon=PI/2; Q2.lat=0;
  R1 = compound_rotation( rotation_matrix(Q1, P.center.lon), rotation_matrix(Q2, P.center.lat) );

  copyGeoPolygon(P, &Q);

  /* rotate polygon coordinates */
  for(i=0; i<Q.nvert; i++) {
    Q.verts[i] = xyz2sph( rotate_point(sph2xyz(P.verts[i],1),R1) );
    }

  /* now, rotate around 0,0 to bring vertex 1 onto equator
   * compute angle between equator and vertex 1; combination of 2 spherical angle
   * calculations for a triangle.  See Zwillinger, D. 1996,
   * Standard Mathematical Tables and Formulae, CRC Press, p. 468.
   */
  omega = acos( tan(Q.verts[0].lon)/tan( acos( cos(Q.verts[0].lon)*cos(Q.verts[0].lat) ) ) );
  if(Q.verts[0].lat > 0) omega *= -1;
  Q1.lat=0; Q1.lon=0;
  R2 = compound_rotation( rotation_matrix(Q1, omega), R1 );

  /* rotate polygon coordinates */
  for(i=0; i<Q.nvert; i++) {
    Q.verts[i] = xyz2sph( rotate_point(sph2xyz(Q.verts[i],1),R2) );
    }
  if(Q.verts[(int)(Q.nvert/2)].lat > 1e-13) fprintf(stderr, "rotation of GeoPolygon resulted in nvert/2 vertex not on equator!\n");

  /* rotate the observation point by same compound rotation */
  A = xyz2sph( rotate_point(sph2xyz(O,1),R2) );
  Op->lon = A.lon; Op->lat = A.lat;

  return(Q);
}


/* convert a discrete global grid in geographic coords
 * to cartesian coordinates, e.g. for plotting
 *
 */
CartesianDGG dgg2cartgrid(DiscreteGlobalGrid G)
{
  unsigned i, j;
  CartesianDGG C;

  C.P = (Polygon *)malloc(sizeof(Polygon)*G.nPolys);
  if(C.P == NULL) {
    fprintf(stderr, "dgg2cart: can't allocate memory for cartesian polygons.\n");
    exit(1);
    }
  C.nPolys = G.nPolys;

  /* convert each polygon vertex from lon,lat,r to x,y,z */
  for(i=0; i<G.nPolys; i++) {
    C.P[i].verts = (XyzPoint *)malloc(sizeof(XyzPoint)*G.P[i].nvert);
    if(C.P[i].verts == NULL) {
      fprintf(stderr, "dgg2cart: can't allocate memory for cartesian polygon vertices.\n");
      exit(1);
      }
    C.P[i].num = G.P[i].num;
    C.P[i].nvert = G.P[i].nvert;
    C.P[i].center = sph2xyz(G.P[i].center, G.R);
    for(j=0; j<G.P[i].nvert; j++) {
      C.P[i].verts[j] = sph2xyz(G.P[i].verts[j], G.R);
      }
    }
  return(C);
}
