##
# Compute terrain corrections for a gravity station, given a file with
# elevation info out to at least 100 mi from each station.
#

import floatdem
from math import *

def correct(data, demname, logger, maxDist=166.7):
  logger(">>> Computing terrain corrections for %d stations....\n"%len(data))
  logger("->> Using DEM in file '%s.flt' and '%s.hdr'\n"%(demname, demname))
  maxDist = float(maxDist)
  # grab dem data
  dem = floatdem.FloatDEM()
  dem.fromfile(demname)
  logger("--> Loaded DEM; extents are %f N,%f E to %f N,%f E\n"%(dem.north(), dem.west(), dem.south(), dem.east()) )
  logger("--> Check DEM for coverage of stations to %.3f km\n"%maxDist)

  # check that our DEM goes far enough NSEW
  for k in data.keys():
    stn = data[k]
    # compute distance from NW and SE corners of DEM to stations
    # and make sure distance >= maxDist for all stations
    d = sphere_d(dem.north(), dem.west(), stn.lat, stn.lon)
    if d < maxDist:
      logger("!!! DEM does not extend to %.3f km from station '%s'; DEM NW corner is %f km from stn."%(maxDist, stn.station_id, d))
      return False;
    d = sphere_d(dem.south(), dem.east(), stn.lat, stn.lon)
    if d < maxDist:
      logger("!!! DEM does not extend to %.3f km from station '%s'; DEM SE corner is %f km from stn."%(maxDist, stn.station_id, d))
      return False;

  # find the offset in pixels that corresponds to our maxDist in km
  # starting at row, col, step out by a pixel until distance on
  # sphere >= maxDist or we hit edge of DEM
  key = data.keys()[0]	# pick the first station
  stn = data[key]
  (row, col) = dem.LL2rc(stn.lat, stn.lon)
  if row == None:
    logger("!!! Station %s location %f,%f not in the DEM.  Die.\n"%(stn.station_id, stn.lat, stn.lon))
    return False
  maxOff = min( (row, dem.nrows-row, col, dem.ncols-col) )
  for offset in range(1,maxOff):
    delta = sphere_d(dem.row2lat(row), dem.col2lon(col+offset), stn.lat, stn.lon)
    if delta >= maxDist:
      break;	# Found offset for this DEM
  logger("--> DEM pixel offset for %.3f km radius: %d\n"%(maxDist, offset))

  # DEM covers sufficient area, so now to compute corrections
  # Only compute for a subset of (large!) DEM around each station
  logger("->> Computing terrain corrections; this may take a bit...\n")
  for k in data.keys():
    stn = data[k]
    stn.tc = 0.0;
    (row, col) = dem.LL2rc(stn.lat, stn.lon);
    if row == None:
      logger("!!! Station %s location %f,%f not in the DEM.  Terrain correction left at 0.\n"%(stn.station_id, stn.lat, stn.lon))
      continue
    # now cycle through all dem pixels within offset of row,col, including the column at the stn!
    for i in range(-offset,offset+1):
      for j in range(-offset,offset+1):
	if sphere_d(dem.row2lat(row+i), dem.col2lon(col+i), stn.lat, stn.lon) <= maxDist:
	  dg = calc_column(stn, dem.row2lat(row+i), dem.col2lon(col+j), dem.get(row+i, col+j), dem.cellsize) * 1e5
	  # 1e5 converts from m/s/s to mGal
	  stn.tc += dg
    logger("--> %s = %7.3f\n"%(stn.station_id, stn.tc))
  logger("--> Done!\n");
  return True;

# distance on sphere between (lat1,lon1) and (lat2,lon2), in km on
# spherical(!) Earth; lat, lon in degrees!
def sphere_d(lat1,lon1,lat2,lon2):
   p1 = cos(radians(lon1-lon2))
   p2 = cos(radians(lat1-lat2))
   p3 = cos(radians(lat1+lat2))

   return acos(((p1*(p2+p3))+(p2-p3))/2)*6371	# km for Earth, roughly

# compute the terrain correction from the crustal column at (lat,lon,z) compared to (station.lat,station.lon,station.z)
def calc_column(station, lat, lon, h, cellsize):
  RHO = 2670.0;	# kg/m^3; we use 2670 and then scale TC for other densities later...
  lat = radians(lat); lon = radians(lon);	# convert to radians to make eqns easier to read
  lat0 = radians(station.lat); lon0 = radians(station.lon);

  # find a dx, dy value
  (dx, toss) = projectTM(lat0, lon0+radians(cellsize), lat0, lon0)
  (toss, dy) = projectTM(lat0+radians(cellsize), lon0, lat0, lon0)

  # compute dz, dx for this column
  dz = station.elev - h;	# m
  if fabs(dz) <= 0.1:	# difference <1 dm, ignore this whole column
    return 0.0;
  nx = 1;
  if dx/dz > 10:	# aspect ratio bad for simulation, so blockify...
    nx = int(ceil(dx/dz));	# guarantee dx/dz <= 10
  dx = dx/nx;
  dy = dy/nx;
  
  # convert from lat,lon to x,y in m
  #   FloatDEM has a function for this, but we feed it station lat/lon
  #   as the origin of the projection!
  (x, y) = projectTM(lat, lon, lat0, lon0)
  z = dz/2.0;	# block stretches from dz to 0, centered at dz/2

  dg = 0
  # Now, compute terrain effect of column dz x cellsize , in blocks of dx,dx,dz...
  for i in range(nx):
    X = x + (i*dx + dx/2.0)
    for j in range(nx):
      Y = y + (j*dy + dy/2.0)		# get center of block
      dg += calc_block(X, Y, z, dx/2.0, dy/2.0, dz/2.0, RHO)
  return dg

def projectTM(lat, lon, lat0, lon0):
  k0 = 1.0;	# see Map Projections - A Working Manual by Snyder, USGS for terms...
  R = 6371204;	# radius of Earth, m
  B = cos(lat)*sin(lon - lon0)
  x = 0.5*R*k0 * log( (1+B) / (1-B) )
  y = R*k0 * ( atan( tan(lat)/cos(lon-lon0) ) - lat0 )
  return (x, y)

# calculate gravity effect at 0,0,0 of a single block at x,y,z with density rho
# and half-sizes a,b,c (in x,y,z order)
def calc_block(x,y,z,a,b,c, rho):
  ### WARNING - HARD-CODED CONSTANTS
  GAMMA = 6.671e-11;	# universal constant of gravitation

  r  = sqrt(x*x + y*y + z*z);
  if r == 0: return(0.0); # observation point at COM of block so ignore this element
  r2 = r*r;
  r3 = r2*r;
  r5 = r3*r2;
  r7 = r5*r2;

  B00 = 8*GAMMA*rho*a*b*c;
  B02 = B00 * (2*c*c - a*a - b*b) / 6.0;
  B22 = B00 * (a*a - b*b) / 24.0;

  F  = B00*z / r3;
  F += ( B02/(2*r5) ) * ( (5*z*(3*z*z-r2)/r2) - 4*z );
  F += B22*15*z*(x*x-y*y) / r7;

  return F;

