#! /usr/bin/python
#
# Create input station parameter file for reduce.py
#

# nice Tk GUI, so we can edit lines, click around, etc.

import sys, string, time

from Tkinter import *
from tkFileDialog import *
from tkSimpleDialog import *
from tkMessageBox import *

from parmio import *
from parmclass import *
from StatusWindow import *

# Title/Author/Version Info
Title = "Station Parameter Editor"
Win_Title = "Station Editor"
Version = "v1.0 (Jul 1999)"
Author = "Paul Gettings\nUniversity of Utah\nDep't of Geology & Geophysics"

parms = []

Edited = 0

######################
### CALLBACK FNS   ###
######################
def quit_cb(*event):
  global Edited
  if Edited:
    if askyesno("Save parameters", "Save station parameters to disk?", parent=main):
      types = (("Parameter Files", "*.par"), ("All Files", "*"))
      fname = asksaveasfilename(title="Save as", parent=main, initialdir=".", filetypes=types)
      if fname:
	timestamp = time.ctime(time.time())
	comment = askstring("Comment", "Parameter File comment", parent=main, initialvalue="Created by station_info")
	write_parms(parms, fname, comment, timestamp)
  sys.exit(1)

def save_cb(*event):
  global Edited
  types = (("Parameter Files", "*.par"), ("All Files", "*"))
  fname = asksaveasfilename(title="Save as", parent=main, initialdir=".", filetypes=types)
  if fname:
    timestamp = time.ctime(time.time())
    comment = askstring("Comment", "Parameter File comment", parent=main, initialvalue="Created by station_info")
    write_parms(parms, fname, comment, timestamp)
    showinfo("Saved", "Parameters saved to file %s."%fname, parent=main)
    Edited=0

def load_cb(*event):
  load_b.config(state=DISABLED)
  global Edited
  global parms
  types = (("Parameter Files", "*.par"), ("All Files", "*"))
  fname = askopenfilename(parent=main, title="Open", initialdir=".", filetypes=types)
  if fname:
    parms = read_parms(fname)
  showinfo("File Loaded", "Data file %s loaded into memory."%fname)
  load_b.config(state=NORMAL, relief=RAISED)
  Edited=0

def edit_cb(*event):
  edit_b.config(state=DISABLED)
  edit_parms()

def edit_quit_cb(*event):
  global edit
  edit.win.withdraw()
  del edit
  edit_b.config(state=NORMAL, relief=RAISED)

def commit_cb(*event):
  global Edited
  global parms, edit, view
  # commit parameters to database
  # get station id for searching
  if not edit.identry.get():
    showerror("No ID", "You must enter a station ID!", parent=edit.win)
    return
  if not edit.nameentry.get():
    showerror("No Name", "You must enter a station name!", parent=edit.win)
    return
  station_id = edit.identry.get()

  # match station_id to database
  Match = 0
  Success = " NOT"
  for i in range(len(parms)):
    if parms[i].sid == station_id:
      # we have a match, so replace info
      if edit.nameentry.get():
	parms[i].name = edit.nameentry.get()
      if edit.latentry.get():
	parms[i].lat = float(edit.latentry.get())
      if edit.lonentry.get():
	parms[i].lon = float(edit.lonentry.get())
      if edit.Zentry.get():
	parms[i].Z = float(edit.Zentry.get())
      if edit.dZentry.get():
	parms[i].dZ = float(edit.dZentry.get())
      #if edit.repeatentry.get():
      parms[i].repeat = parse_repeats(edit.repeatentry.get())
      Match = 1
      Success = ""
      Edited=1
  if not Match:
    # no match, so ask if want to add
    if askyesno("Create New Station", "No matching Station ID found.  Create a new station?", parent=main):
      new = StationParameters()
      new.sid = edit.identry.get()
      new.name = edit.nameentry.get()
      if edit.latentry.get():
	new.lat = float(edit.latentry.get())
      else:
	new.lat = 0.0
      if edit.lonentry.get():
	new.lon = float(edit.lonentry.get())
      else:
	new.lon = 0.0
      if edit.Zentry.get():
	new.Z = float(edit.Zentry.get())
      else:
	new.Z = 0.0
      if edit.dZentry.get():
	new.dZ = float(edit.dZentry.get())
      else:
	new.dZ = 0.0
      if edit.repeatentry.get():
	new.repeat = parse_repeats(edit.repeatentry.get())
      else:
	new.repeat = ()
      parms.append(new)
      Success = ""
      Edited=1
    else:
      Success = " NOT"
  # refresh view window, if open
  try:
    if view:
      refresh_view_cb(event)
  except NameError:
    pass
  showinfo("Commit Result", "Parameters for station %s%s updated."%(station_id, Success), parent=main)

def new_cb(*event):
  global Edited
  new_b.config(state=DISABLED)
  global parms
  if askyesno("Clear Database", "Clear the database of all stations?", parent=main):
    parms = []
  new_b.config(state=NORMAL)
  Edited = 0

def view_quit_cb(*event):
  global view
  view.win.withdraw()
  del view
  view_b.config(state=NORMAL, relief=RAISED)

def view_cb(*event):
  view_b.config(state=DISABLED)
  global parms
  # create view window - window with lots of columns
  # cycle over database, printing info in ordered columns
  global view
  view = ViewWindow("View Station Parameters")
  view.text = StatusWindow(view.win, 80, 24)
  refresh_view_cb(event)
  view.text.show(columnspan=2)
  view.quit_b = Button(view.win, text="Close Viewer", command=view_quit_cb)
  view.refresh_b = Button(view.win, text="Refresh Viewer", command=refresh_view_cb)
  view.quit_b.grid(column=0)
  view.refresh_b.grid(column=1, row=view.quit_b.grid_info().get("row"))

  view.win.update()

def refresh_view_cb(*event):
  global view
  global parms
  # clear the text window
  view.text.clear()
  # column headings
  view.text.insert(END, "  ID  |    STATION NAME    |REPEAT|  LATITUDE  | LONGITUDE  | ELEVATION|  dZ \n")
  # loop over database
  for i in range(len(parms)):
    # print parameters
    view.text.insert(END, "%6s|%-20s|"%(parms[i].sid, parms[i].name))
    if not parms[i].repeat:
      view.text.insert(END, " None ")
    else:
      view.text.insert(END, "%6s" % parms[i].repeat[0])
    view.text.insert(END, "|%12.7f|%12.7f|%10.3f|%5.1f\n" % (parms[i].lat, parms[i].lon, parms[i].Z, parms[i].dZ))
    for j in range(1,len(parms[i].repeat)):
      view.text.insert(END, "......|....................|%6s|............|............|..........|\n"%parms[i].repeat[j])
  view.text.update()

def search_cb(*event):
  global edit, parms
  # search through current database for a match to station id/name
  # then fill in fields
  sid = edit.identry.get()
  sname = edit.nameentry.get()
  for i in range(len(parms)):
    if parms[i].sid == sid:
      # we have an id match - fill in all fields
      fill_all_fields(i)
      return
  # no ID matches, maybe name?
  for i in range(len(parms)):
    if parms[i].name == sname:
      # we have name match - fill in all but name/repeats
      fill_most_fields(i)
      return
  # no match, so inform user
  showinfo("No Match", "No match found for station ID or name.", parent=main);


def next_cb(*event):
  global edit, parms
  # advance to next station
  sid = edit.identry.get()
  if not sid:
    edit.identry.set(parms[0].sid)
    fill_all_fields(0)
  for i in range(len(parms)):
    if parms[i].sid == sid:
      i = i+1
      if i >= len(parms):
	return
      edit.identry.set(parms[i].sid)
      fill_all_fields(i)

def prev_cb(*event):
  global edit, parms
  # advance to next station
  sid = edit.identry.get()
  for i in range(len(parms)):
    if parms[i].sid == sid:
      i = i-1
      if i < 0:
	return
      fill_all_fields(i)
      edit.identry.set(parms[i].sid)
  
def delete_cb(*event):
  global edit, parms
  # search through current database for a match to station id/name
  # then fill in fields
  sid = edit.identry.get()
  for i in range(len(parms)):
    if parms[i].sid == sid:
      if askyesno("Delete Station", "Delete station with Station ID %s?"%parms[i].sid, parent=edit.win):
	if i+1 >= len(parms):
	  if i-1 < 0: j = 0
	  else: j = i-1
	else:
	  j = i+1
	edit.identry.set(parms[j].sid)
	fill_all_fields(j)
	del parms[i]
	refresh_view_cb()
	return

######################
###   FUNCTIONS    ###
######################
def edit_parms():
  # create Tk editing window
  global edit
  edit = EntryWindow("Edit Parameters")

  # edit window

  edit.identry = TextEntry(edit.win, "Station ID:")
  edit.nameentry = TextEntry(edit.win, "Station Name:")
  edit.nameentry.field.config(width=20)
  edit.np_win = Frame(edit.win)
  edit.prev_b = Button(edit.np_win, text="<--", command=prev_cb)
  edit.next_b = Button(edit.np_win, text="-->", command=next_cb)
  edit.search_b = Button(edit.win, text="Find Match", command=search_cb)
  edit.delete_b = Button(edit.win, text="Delete Station", command=delete_cb)

  edit.ef1 = Frame(edit.win)
  edit.ef2 = Frame(edit.win)
  edit.latentry = TextEntry(edit.ef1, "Latitude:")
  edit.latentry.field.config(width=16)
  edit.lonentry = TextEntry(edit.ef1, "Longitude:")
  edit.lonentry.field.config(width=15)
  edit.Zentry = TextEntry(edit.ef2, "Elevation (m):")
  edit.Zentry.field.config(width=15)
  edit.dZentry = TextEntry(edit.ef2, "Elev. Change (cm):")
  edit.dZentry.field.config(width=11)

  edit.repeatentry = TextEntry(edit.win, "Reoccupation(s)")
  edit.repeatentry.field.config(width=30)

  edit.quit_b = Button(edit.win, text="Stop Editing", command=edit_quit_cb)
  edit.commit_b = Button(edit.win, text="Commit Parameter Changes", command=commit_cb)
  edit.commit_b.config(default=ACTIVE)

  # layout the dialog
  edit.identry.box.grid(column=0, row=0)
  edit.nameentry.box.grid(column=1, row=0)
  edit.search_b.grid(column=0, row=2)
  edit.delete_b.grid(column=1, row=2)
  edit.next_b.grid(column=1, row=0)
  edit.prev_b.grid(column=0, row=0)
  edit.np_win.grid(column=0, row=1)

  edit.latentry.box.grid(sticky="w")
  edit.lonentry.box.grid(sticky="w")
  edit.Zentry.box.grid(sticky="w")
  edit.dZentry.box.grid(sticky="w")
  edit.ef1.grid(column=0, row=3)
  edit.ef2.grid(column=1, row=3)

  edit.repeatentry.box.grid(columnspan=2, row=4)

  edit.quit_b.grid(column=0, row=5)
  edit.commit_b.grid(column=1, row=5)

  edit.win.update()

def parse_repeats(list):
  # convert comma separated list into Python tuple
  fields = []
  if not list:
    return fields
  fields = string.split(list, ',')
  return fields

def unparse_repeats(reps):
  # convert array of repeats into comma sep. string
  if not reps:
    return ""
  string = "%s"%reps[0]
  for i in range(1,len(reps)):
    string = "%s,%s"%(string, reps[i])
  return string

def fill_all_fields(i):
  global edit, parms
  edit.nameentry.set(parms[i].name)
  repeats = unparse_repeats(parms[i].repeat)
  edit.repeatentry.set(repeats)
  edit.latentry.set(parms[i].lat)
  edit.lonentry.set(parms[i].lon)
  edit.Zentry.set(parms[i].Z)
  edit.dZentry.set(parms[i].dZ)

def fill_most_fields(i):
  global edit, parms
  edit.latentry.set(parms[i].lat)
  edit.lonentry.set(parms[i].lon)
  edit.Zentry.set(parms[i].Z)
  edit.dZentry.set(parms[i].dZ)

######################
### MAIN CODE HERE ###
######################
# Create Tk interface
# main window
main = Tk()
main.title(Win_Title)

if len(sys.argv) > 1:
  parms = read_parms(sys.argv[1])
  if parms:
    showinfo("File Loaded", "Data file %s loaded into memory."%sys.argv[1])

title_box = Frame(main, relief="raised", borderwidth=3)
Label(title_box, text=Title).grid()
Label(title_box, text=Version).grid()
Label(title_box, text=Author).grid()

title_box.grid(columnspan=2)

# main window buttons
new_b = Button(main, text="New", command=new_cb)
load_b = Button(main, text="Load", command=load_cb)
edit_b = Button(main, text="Edit", command=edit_cb)
view_b = Button(main, text="View", command=view_cb)
save_b = Button(main, text="Save", command=save_cb)
quit_b = Button(main, text="Quit", command=quit_cb)

edit_b.grid(column=0, row=1)
view_b.grid(column=1, row=1)
new_b.grid(column=0, row=2)
load_b.grid(column=1, row=2)
save_b.grid(column=0, row=3)
quit_b.grid(column=1, row=3)

main.mainloop()
