# platform-independant serial I/O port class
# actually, we have platform-dependant serial classes
# and then a simple wrapper that determines which to use

import sys

from string import upper


# Unix serial routines
from TERMIOS import *
import termios

class SerialPort:
# wrapper for platform specific code 
  def __init__(self):
    self.__Unix = 1
    self.__Win32 = 2
    # test for win32
    if upper(sys.platform) == 'WIN32':
      self.platform = self.__Win32
      self.__connection = Win32SerialPort()
    # otherwise, assume unix-like, with POSIX termio capability
    else:
      print "UNIX TERMIO PORT"
      print self
      self.platform = self.__Unix
      self.__connection = UnixSerialPort()
      print self
      print self.__connection

  def __del__(self):
    self.__connection.__del__()

  def open(self, port, baud=9600, parity="No", dataLen=8, stopBits=1):
    self.__connection.open(port, baud, parity, dataLen, stopBits)

  def close(self):
    self.__connection.close()

  def read(self, n=1):
    return self.__connection.read(n)

  def write(self, string): 
    return self.__connection.write(string)

# Class taken from Kevin Dahlhausen
# code mailed to a Python mailing list
class Win32SerialPort:
	# Win32 serial routines
	def __init__(self):
		import serial
		self.thePort = serial.Port()

	def __del__(self):
		self.close()

	def open(self, portNum=1, baud=19200, parity="No", dataLen=8, stopBits=1):
		""" Opens the port for communications.

			Keyword arguments:

			w2portNum -- comm port to use (1-9)
			baud -- baud rate: 110, 330, 1200, 2400, 4800, 9600, 19200, 
						       38400, 57600, 115200
			parity -- parity.  Must be one of:
					  "No", "Odd", "Even", "Mark", or "Space"
			dataLen -- data length (5-8)
			stopBits -- # of stop bits (1-2)
		"""
		cfg = serial.PortDict()
		ports = ( serial.COM1, serial.COM2, serial.COM3, 
				  serial.COM4, serial.COM5, serial.COM6, 
				  serial.COM7, serial.COM8, serial.COM9)
		cfg['port']=ports[portNum-1]

		bauds ={ 110:serial.Baud110,
					300:serial.Baud300,
					1200:serial.Baud1200,
					2400:serial.Baud2400,
					4800:serial.Baud4800,
					9600:serial.Baud9600,
					19200:serial.Baud19200,
					38400:serial.Baud38400,
					57600:serial.Baud57600,
					115200:serial.Baud115200 }
		cfg['baud']=bauds[baud]

		parities = {	"No":serial.NoParity,
        				"Odd":serial.OddParity, 
						"Even":serial.EvenParity,
        				"Mark":serial.MarkParity,
				        "Space":serial.SpaceParity}	
		cfg['parity']=parities[parity]

		wls = {	5:serial.WordLength5,
				6:serial.WordLength6,
				7:serial.WordLength7,
				8:serial.WordLength8 }
		cfg['dataBits']=wls[dataLen]

		sbits=( serial.OneStopBit, serial.TwoStopBits)
		cfg['stopBits']=sbits[stopBits-1]

		self.thePort.open(cfg)

	def openFromDict(self, dict):
		""" Opens the port for communications using a dictionary for the opts.

			Keyword arguments:

			dict -- dictionary containing the following STRINGS:
				PortNumber 
				BaudRate 
				Parity
				DataLength
				StopBits
		"""
		try:	
			print str(dict)
			cfg = serial.PortDict()
			cfg['port']=serial.ports[dict["PortNumber"]]
			cfg['baud']=serial.bauds[dict["BaudRate"]]
			cfg['parity']=serial.parities[dict["Parity"]]
			cfg['dataBits']=serial.wordLengths[dict["DataLength"]]
			cfg['stopBits']=serial.stopBits[dict["StopBits"]]
			self.thePort.open(cfg)	
		except KeyError, ke:
			print "ofd:Ke "+str(ke)
			dict["PortNumber"]="1"
			dict["BaudRate"]="19200"
			dict["Parity"]="No"
			dict["DataLength"]="8"
			dict["StopBits"]="1"
			for k in dict.keys():
				print "k=%s = %s"%(k, dict[k])
			self.openFromDict(dict)

	def close(self):
		#print "closing Win32SerialPort"
		self.thePort.close()
		

	def read(self, numToRead=1):
		"""Returns next char from port. (V) 

		   Keyword arguments:

		   numToRead -- # of characters to read
		"""
		s = self.thePort.read(numToRead)
		if (s==''):
			return None
		else:
			t = ord(s)
			#print "read: %4x %4d %4c"%(t, t, t)
			return ord(s)

	def write(self, c):
		"""Writes character to the port. (PV)

			Keyword arguments:

			c - character to write
		"""
		m = '%c' % (c)
		#print "wrote: %4x %4d %4c"%(c, c, c)
		self.thePort.write(m,1)

# convenience constants for tcgetattr/tcsetattr lists
iflags = 0
oflags = 1
cflags = 2
lflags = 3
ispeed = 4
ospeed = 5
ctrl_chars = 6

class UnixSerialPort:
  def __init__(self):
    self.port = None

  def __del__(self):
    self.close()

  def open(self, portName="/dev/ttyS0", baud=9600, parity="No", dataLen=8, stopBits=1):
    # Opens the port for communications.
    # Keyword arguments:
    # portName -- name of special file to open
    # baud -- baud rate: 110, 330, 1200, 2400, 4800, 9600, 19200, 
    #   38400, 57600, 115200
    # parity -- parity.  Must be one of:
    #   "No", "Odd", "Even"
    # dataLen -- data length (5-8)
    # stopBits -- # of stop bits (1-2)
    #try:
    self.port = open(portName, "r+")
#    except:
#      self.port = None
#      return None

    self.portfd = self.port.fileno()
    self.oldPortAttr = termios.tcgetattr(self.portfd)
    new = termios.tcgetattr(self.portfd)

    BR = {110:B110, 300:B300, 1200:B1200, 2400:B2400, 4800:B4800,
      9600:B9600, 19200:B19200, 38400:B38400, 57600:B57600, 115200:B115200}

    # setup the port
    new[ispeed] = BR[baud]
    new[ospeed] = BR[baud]

    #   test for parity
    parity = upper(parity)
    if parity == 'NO':
      new[cflags] = new[cflags] & ~PARENB
    elif parity == 'ODD':
      new[cflags] = new[cflags] & PARENB
      new[cflags] = new[cflags] & PARODD
    elif parity == 'EVEN':
      new[cflags] = new[cflags] & PARENB
      new[cflags] = new[cflags] & ~PARODD
    else:
      self.port.close()
      return None

    #    set data length
    CS = {5:CS5, 6:CS6, 7:CS7, 8:CS8}
    try:
      new[cflags] = new[cflags] & CS[dataLen]
    except:
      self.port.close()
      return None

    #    set stop bits
    if stopBits == 2:
      new[cflags] = new[cflags] & CSTOPB
    else:
      new[cflags] = new[cflags] & ~CSTOPB

    #    write settings to port
    try:
      termios.tcsetattr(self.portfd, TCSANOW, new)
    except:
      self.port.close()
      return None

    return 1

  def close(self):
    if not self.port:
      return
    self.port.close()

  def read(self, numToRead=1):
    # Returns characters from port.
    # numToRead -- # of characters to read
    s = self.port.read(numToRead)
    return s

  def write(self, s):
    # write a string to port
    self.port.write(s)
    # wait for string to be written
    termios.tcdrain(self.portfd)

