"""
	SYNOPSIS:

	Provides the classes used for Python communocations.

	EXPORTS:

		Port -- base class for communications channel
		Win32SerialPort -- Communications through a serial port under windows
		SocketPort -- Communications through TCP/IP
		DebugPort -- Prints data

"""

__author__  = "Kevin Dahlhausen  ap096@po.cwru.edu"
__version__ = "$Revision: 1.1.1.1 $"
__date__  = "$Date: 2000/04/11 21:41:28 $"	


class Port:
	""" Abstract base class for all Port classes. """

	def open(self):
		""" Opens the port, allowing it to be used. (V)"""
		Pass

	def close(self):
		""" Closes the port. (V)"""
		Pass

	def	read(self):
		"""Returns next char from port. (PV) """
		None

	def write(self, c):
		"""Writes character to the port. (PV)

			Keyword arguments:

			c - character to write
		"""
		None


import serial
class Win32SerialPort(Port):
	""" Implements communications through RS-232 port under 32 bit windows.""" 

	def __init__(self):
		self.thePort = serial.Port()

	def __del__(self):
		self.close()

	def open(self, portNum=1, baud=19200, parity="No", dataLen=8, stopBits=1):
		""" Opens the port for communications.

			Keyword arguments:

			w2portNum -- comm port to use (1-9)
			baud -- baud rate: 110, 330, 1200, 2400, 4800, 9600, 19200, 
						       38400, 57600, 115200
			parity -- parity.  Must be one of:
					  "No", "Odd", "Even", "Mark", or "Space"
			dataLen -- data length (5-8)
			stopBits -- # of stop bits (1-2)
		"""
		cfg = serial.PortDict()
		ports = ( serial.COM1, serial.COM2, serial.COM3, 
				  serial.COM4, serial.COM5, serial.COM6, 
				  serial.COM7, serial.COM8, serial.COM9)
		cfg['port']=ports[portNum-1]

		bauds ={ 110:serial.Baud110,
					300:serial.Baud300,
					1200:serial.Baud1200,
					2400:serial.Baud2400,
					4800:serial.Baud4800,
					9600:serial.Baud9600,
					19200:serial.Baud19200,
					38400:serial.Baud38400,
					57600:serial.Baud57600,
					115200:serial.Baud115200 }
		cfg['baud']=bauds[baud]

		parities = {	"No":serial.NoParity,
        				"Odd":serial.OddParity, 
						"Even":serial.EvenParity,
        				"Mark":serial.MarkParity,
				        "Space":serial.SpaceParity}	
		cfg['parity']=parities[parity]

		wls = {	5:serial.WordLength5,
				6:serial.WordLength6,
				7:serial.WordLength7,
				8:serial.WordLength8 }
		cfg['dataBits']=wls[dataLen]

		sbits=( serial.OneStopBit, serial.TwoStopBits)
		cfg['stopBits']=sbits[stopBits-1]

		self.thePort.open(cfg)

	def openFromDict(self, dict):
		""" Opens the port for communications using a dictionary for the opts.

			Keyword arguments:

			dict -- dictionary containing the following STRINGS:
				PortNumber 
				BaudRate 
				Parity
				DataLength
				StopBits
		"""
		try:	
			print str(dict)
			cfg = serial.PortDict()
			cfg['port']=serial.ports[dict["PortNumber"]]
			cfg['baud']=serial.bauds[dict["BaudRate"]]
			cfg['parity']=serial.parities[dict["Parity"]]
			cfg['dataBits']=serial.wordLengths[dict["DataLength"]]
			cfg['stopBits']=serial.stopBits[dict["StopBits"]]
			self.thePort.open(cfg)	
		except KeyError, ke:
			print "ofd:Ke "+str(ke)
			dict["PortNumber"]="1"
			dict["BaudRate"]="19200"
			dict["Parity"]="No"
			dict["DataLength"]="8"
			dict["StopBits"]="1"
			for k in dict.keys():
				print "k=%s = %s"%(k, dict[k])
			self.openFromDict(dict)

	def close(self):
		#print "closing Win32SerialPort"
		self.thePort.close()
		

	def	read(self, numToRead=1):
		"""Returns next char from port. (V) 

		   Keyword arguments:

		   numToRead -- # of characters to read
		"""
		s = self.thePort.read(numToRead)
		if (s==''):
			return None
		else:
			t = ord(s)
			#print "read: %4x %4d %4c"%(t, t, t)
			return ord(s)

	def write(self, c):
		"""Writes character to the port. (PV)

			Keyword arguments:

			c - character to write
		"""
		m = '%c' % (c)
		#print "wrote: %4x %4d %4c"%(c, c, c)
		self.thePort.write(m,1)

SerialPort = Win32SerialPort


import socket
class SocketPort(Port): 
	def __init__(self):
		self.socket = None

	def __del__(self):
		if (self.socket != None):
			self.socket.close()

	def open(self, host = "127.0.0.1", port = 50007):
		"""Connects to the given host and port.

		   Keyword arguments:

			host -- IP address of host with which to connect.
				    (Can use name?)
			port -- TCP/IP port number to use.
		"""
		self.socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
		self.socket.connect(host, port)

	def close(self):
		self.socket.close()
		
	def	read(self, numToRead=1):
		c = ord(self.socket.recv(1))
		print "sockread:%d" % c
		return c
		
	def write(self, c):
		self.socket.send(chr(c))



class DebugPort:
	""" Prints debug data """

	def open(self):
		""" Opens the port, allowing it to be used. (V)"""
		print "<DebugPort.open>"

	def close(self):
		""" Closes the port. (V)"""
		print  "<DebugPort.close>"
		Pass

	def	read(self):
		"""Returns next char from port. (PV) """
		print "<DebugPort.read>"
		return 32
		
		

	def write(self, c):
		"""Writes character to the port. (PV)

			Keyword arguments:

			c - character to write
		"""
		print "<DebugPort.Write> %x %d %c'"%(c,c,c)

