#! /usr/bin/python

# Gravity reduction program in Python
# Uses raw output of CG-3M, parses it, and computes corrections.
# Then creates drift function, complete with interactive mouse input of
# tares!
# Mouse input and graphical functions require Tk interface

# Includes full GUI, but can be used in "Batch Mode" with control file
# Control file is long, and annoying, so GUI is most likely mode
# Batch Mode useful for re-running reduction later with same options

# line width:########################################################
VERSION  =  "##            Version 3.7.0 (Oct 2013)                ##";

# known meter types, as all uppercase
METER_TYPES = {'CG-3':1, 'ALIOD':1, 'CG-5':1}

# system modules
import sys, string
import Tkinter
import tkFileDialog, tkMessageBox, tkSimpleDialog
from math import floor
import time
from types import *
import traceback

# custom modules
import fileop
import cg3file, cg5file, aliodfile
import atmosphere, etide, drift, deltaz, tilt
from grav_util import Truth
import grav_util
import grav_gui
from grav_data import *
import embeddedViewer, StatusWindow
import logger
import qc

def debug_status(write):
  return
  write("*** CURRENT STATUS OF CODE ***\n")
  write("Known Issues:\n")
  write("  Tilt zero error fixes not correct\n")
  write("******************************\n")

def main():
  global VERSION
  global write, log
  BatchMode = 0
  ##Parse command-line args
  for i in range(len(sys.argv)):
    if sys.argv[i] == '-F':
      # batch mode - get input from control file
      BatchMode = 1
      try:
	controlfilename = sys.argv[i+1]
      except:
	sys.stderr.write("Must supply control file name; falling back to interactive mode\n")
	BatchMode=0

  options = {}
  if BatchMode:
    # create logger instance 
    log = logger.Logger("reduce.log")
    # set output function
    write = log.scrn
    options = fileop.grokCmdFile(controlfilename)
    if not options:
      # scream and die
      print "CANNOT PARSE COMMAND FILE %s. ABORT."%(controlfilename)
      sys.exit(0)

  else:
    ##Convenience variables
    data_filetypes = (("Data Files","*.dat"),
      ("Raw Data Files","*.raw"), ("Text Data Files","*.txt"),
      ("All Files","*"))
    out_filetypes  = (("Output Files","*.out"), ("All Files","*"))
    par_filetypes  = (("Parameter Files","*.par"), ("All Files","*"))
    atm_filetypes  = (("Atmospheric Data Files","*.atm"), ("All Files","*"))
    meter_filetypes = (("Meter Info Files","*.inf"), ("All Files","*"))

    ##Fire up the GUI
    root = Tkinter.Tk()
    root.title("MicroGravity")
    # create status window
    status_win = StatusWindow.StatusWindow(root, 150,30);
    status_win.show()

    # create logger instance 
    log = logger.Logger("reduce.log", window=status_win)

    # set output function
    write = log.win

    # get default control options
    options = fileop.defaultOptions()

  # timestamp the log files
  write("Started %s\n"%time.asctime(time.localtime(time.time())))
  write("########################################################\n")
  write("##      High-Precision Gravity Reduction Program      ##\n")
  write("%s\n"%VERSION)
  write("##                   Paul Gettings                    ##\n")
  write("##         Department of Geology & Geophysics         ##\n")
  write("##                University of Utah                  ##\n")
  write("##        http://thermal.gg.utah.edu/~gettings        ##\n")
  write("########################################################\n")
  write("\n*** READ RAW DATA ***\n\n")

  if not BatchMode:
    write("#-----------------------\n")
    write("# Select the type of gravity meter used to collect this data.\n")
    write("# Gravity meter type sets the expected data file format!\n")
    write("#-----------------------\n")
    root.update()
    meter_type = tkSimpleDialog.askstring("Meter Type",
      "CG-3 ==> CG-3(M)\nCG-5 ==> CG-5\naliod ==> Aliod 10x/100x",
      parent=root, initialvalue="CG-5")
    if not meter_type:
      write( "!!! must enter a meter type!\n")
      tkMessageBox.showerror("No Meter Type", "Must enter a meter type!")
      sys.exit()
  else:
    meter_type = options["meter_type"]

  # check meter type is one we know
  if METER_TYPES.has_key(meter_type.upper()):
    options["meter_type"] = meter_type.upper()
    write(">>> Data file format from meter type '%s'.\n"%options["meter_type"])
  else:
    write("!!! Meter type '%s' not recognized; dying!\n")%meter_type
    sys.exit()

  if options["meter_type"] == "ALIOD":
    if not BatchMode:
      write("#-----------------------\n")
      write("# Select the meter information file. This file\n")
      write("# MUST have the calibration table.\n")
      write("#-----------------------\n")
      root.update()
      meter_file = tkFileDialog.askopenfilename(parent=root, title="Aliod Meter Info File", initialdir=".", filetypes=meter_filetypes)

    else:
      meter_file = options["meter_file"]

    if not meter_file:
      write( "!!! must enter a meter information filename for Aliod data!\n")
      if not BatchMode:
	tkMessageBox.showerror("No Meter File Name", "Must enter a meter information filename!")
      sys.exit()

  if not BatchMode:
    write("#-----------------------\n")
    write("# Select the raw gravity data file.  Note that the any garbage at\n")
    write("# the beginning on the file should be removed before processing\n")
    write("#-----------------------\n")

  if not BatchMode:
    root.update()
  #
  # Read Raw Data
  if BatchMode:
    raw_file = options["raw_file"]
  else:
    raw_file = tkFileDialog.askopenfilename(parent=root, title="Gravity Data File", initialdir=".", filetypes=data_filetypes)

  if not raw_file:
    write( "!!! must enter a data filename!\n")
    if not BatchMode:
      tkMessageBox.showerror("No File Name", "Must enter a data filename!")
    sys.exit()

  write(">>> Reading station data file\n")
  try:
    if options["meter_type"] == "CG-3":
      raw_data = cg3file.get_cg3_data(raw_file)
    elif options["meter_type"] == "CG-5":
      raw_data = cg5file.get_cg5_data(raw_file)
    elif options["meter_type"]== "ALIOD":
      raw_data = aliodfile.get_aliod_data(raw_file, meter_file, 1)
    else:
      write("!!! Unknown meter type '%s' for data read; die.\n"%options["meter_type"])
  except Exception, e:
    write("!!! error processing data file %s\nException: %s\n"%(raw_file, e))
    if not BatchMode:
      tkMessageBox.showerror("Data File Error", "Error processing data file %s!"%raw_file)
    sys.exit()
  write("--> Processed gravity data file %s with %d data records\n"%(raw_file, len(raw_data)))

  options["raw_file"] = raw_file
  if options["meter_type"] == "ALIOD":
    options["meter_file"] = meter_file

  if not BatchMode:
    write("#-----------------------\n")
    write("# Select the parameter file.\n");
    write("#-----------------------\n")

  #
  # Get Station Parameters - lat/lon/Z, repeats, names
  if BatchMode:
    parm_file = options["parm_file"]
  else:
    parm_file = tkFileDialog.askopenfilename(parent=root, title="Station Parameter File",
      initialdir=".", filetypes=par_filetypes)
  if not parm_file:
    write("!!! must give station parameter file!\n")
    if not BatchMode:
      tkMessageBox.showerror("No File Name", "Must give parameter file name!")
    sys.exit()
  write(">>> Reading station parameter file\n")
  try:
    parms = fileop.get_sta_parms(parm_file)
    repeats = fileop.get_sta_repeats(parm_file)
    names = fileop.get_sta_names(parm_file)
    skips = fileop.get_sta_skips(parm_file)
  except:
    write("!!! cannot process station parameter file %s.\n" % parm_file)
    if not BatchMode:
      tkMessageBox.showerror("Parameter File Error", "Error processing parmater file!")
    sys.exit()
  #
  # Remove "skipped" stations
  write("->> Removing stations in skip list\n")
  list = []
  del_stations = {}
  for i in range(len(raw_data)):
    if skips.has_key(raw_data[i].station_id):
      list.append(i)
      del_stations[raw_data[i].station_id] = 1
  list.reverse()
  for j in list:
    del raw_data[j]
  write("--> Removed %d of %d stations in skip list (%d data records)\n"%(len(del_stations), len(skips), len(list)))
  #
  # Check to make sure that the parameter file has an entry for every
  # station in data file
  write("->> Checking for missing stations in parameter file\n");
  for i in range(len(raw_data)):
    if not parms.has_key(raw_data[i].station_id):
      write("!!! cannot find parameter file entry for station id [ %s ]\n"%(raw_data[i].station_id))
      write("!!! Abort.\n")
      if not BatchMode:
	tkMessageBox.showerror("Unknown Station", "No parameter information for station id %s!"%raw_data[i].station_id)
      sys.exit()
  write("->> Removing parameter entries with no matching station data\n")
  parms = grav_util.remove_extra_parms(parms, raw_data)
  #
  # Get the number of unique station ids and store
  nuniqids = len(grav_util.get_uniq_ids(raw_data))
  write("->> Raw data has %d unique station ids\n"%nuniqids)
  #
  # Check our repeat information against the data; rebuild the repeat
  # information if we have subset of the full data
  # create list of unique station ids
  write("->> Checking station repeat information for missing stations\n");
  (repeats, changes) = grav_util.check_repeats(raw_data, repeats)
  write("--> station repeat information has %d changes\n"%changes)
  del changes

  write("--> Processed station parameter file %s\n"%(parm_file))

  options["parm_file"] = parm_file

  if not BatchMode:
    grav_gui.parameterView(parms, names, repeats)

  #
  # Compute relative dates for each data point
  #start_jday = grav_util.calc_relative_date(raw_data)

  #
  # Get pre-existing tare database
  # parse from previous output run
  if not BatchMode:
    write("#-----------------------\n")
    write("# Select tare database file\n")
    write("# Enter the filename with existing tare data.\n")
    write("# This can be the output file from a previous reduction, or\n")
    write("# any text file with the same format.\n")
    write("#-----------------------\n")

  if BatchMode:
    tare_file = options["tare_file"]
  else:
    tare_file = tkFileDialog.askopenfilename(parent=root, title="Tare Data File",
      initialdir=".", filetypes=out_filetypes)
  if not tare_file:
    write("!!! Assuming no previous tare data.\n")
    tares = {}
    tare_file = ""
  else:
    write(">>> Reading tare data from file\n")
    tares = fileop.get_tares(tare_file)
    if not tares:
      write("!!! No tare data found in %s.  Assuming no previous tare data.\n"%tare_file)
      tares = {}
    else:
      k = tares.keys()
      k.sort
      for i in k:
	write("%s (%12.2f) %6.3f\n"%(grav_util.datestr(i), i, tares[i]))
    write("--> done\n")

  options["tare_file"] = tare_file

  #
  # Get correction options
  if not BatchMode:
    corrections = grav_gui.getCorrections()
    options2 = corrections.run()

    if not options2:
      write("!!! Correction options dialog cancelled. Abort.\n")
      sys.exit()

    # overwrite default options with chosen entries
    for i in options2.keys():
      options[i] = options2[i]

  #
  # view the data before any corrections
  write("\n*** VIEW UNCORRECTED DATA ***\n\n")
  if Truth(options["raw_view"]):
    write(">>> starting data viewer...\n")
    # fire it up
    embeddedViewer.main(raw_data)
  else:
    write(">>> skipping view data\n")

  #
  # Reset lat/lon/z from parameter file
  for i in range(len(raw_data)):
    raw_data[i].lon = parms[raw_data[i].station_id].lon
    raw_data[i].lat = parms[raw_data[i].station_id].lat
    raw_data[i].elevation = parms[raw_data[i].station_id].elevation

  write("\n*** CORRECTIONS TO RAW DATA ***\n\n")
  write(">>> Pre-average corrections\n")

  #
  # ETC removal
  write("->> Remove Meter's Earth Tide Correction?\n")
  if Truth(options["longman"]):
    write("--> removing ETC applied by meter...")
    for i in range(len(raw_data)):
      raw_data[i].gravity = raw_data[i].gravity - raw_data[i].etc
    write("done.\n")
  else:
    write("--> leaving ETC applied by meter.\n")
  #
  # Drift removal
  write("->> Remove Meter Drift Correction?\n")
  if options["meter_type"] != "ALIOD":
    if Truth(options["drift"]):
      write("--> removing drift correction applied by meter to ALL readings...")
      for i in range(len(raw_data)):
	dg = raw_data[i].meterInfo.DriftCo * (raw_data[i].jul_day - raw_data[i].meterInfo.DriftStart)
	raw_data[i].gravity = raw_data[i].gravity + dg
      write("done.\n")
    else:
      write("--> leaving drift correction applied by meter.\n")
  else:
    write("--> ignoring option for Aliod meter.\n")
  #
  # Atmospheric Correction - correct gravity value for changes in atmospheric
  # pressure relative to some base_pressure
  write("->> Atmospheric Pressure Correction?\n")
  if Truth(options["atmosphere"]):
    if BatchMode:
      atm_file = options["atm_file"]
    else:
      atm_file = tkFileDialog.askopenfilename(parent=root, title="Atmospheric Pressure File",
	initialdir=".", filetypes=atm_filetypes)
    write("--> performing atmospheric corrections...")
    base_pressure = float(options["base_pressure"])
    rc = atmosphere.correction(atm_file, raw_data, base_pressure)
    if rc == None:
      write("!!! ERROR PROCESSING ATMOSPHERIC PRESSURE FILE %s!\n", atm_file);
      write("!!! No atmospheric corrections\n")
      atm_file = "NONE"
      base_pressure = 0.0
      for i in range(len(raw_data)):
	raw_data[i].atmospheric_correction = 0.0
    write("done.\n")
  else:
    atm_file = "NONE"
    write("--> no atmospheric corrections\n")
    base_pressure = 0.0
    for i in range(len(raw_data)):
      raw_data[i].atmospheric_correction = 0.0

  options["atm_file"] = atm_file
  options["base_pressure"] = base_pressure

  #
  # Reset time of all data to GMT
  # use point's GMT offset, then set offset to 0
  # override data file settings, if desired
  write(">>> Offset data times to GMT, using GMT offset...\n")
  if options.has_key("gmt_offset"):
    if options["gmt_offset"]:
      try:
	options["gmt_offset"] = float(options["gmt_offset"])
      except:
	options["gmt_offset"] = 0.0
      write("--> Reset GMT offset to %.1f hrs..."%options["gmt_offset"])
      for i in range(len(raw_data)):
	raw_data[i].GMT_Diff = options["gmt_offset"]
      write("done.\n")
  write("--> Offset data times to GMT...")
  for i in range(len(raw_data)):
    raw_data[i].jul_day -= raw_data[i].GMT_Diff/24.0;
    raw_data[i].GMT_Diff = 0;
  write("done.\n")

  #
  # Earth Tide Correction
  # assume we removed the ETC of the meter, so we need full ETC correction,
  # based on lat/lon/GMT_Diff of each station
  write("->> Tamura Earth Tide Correction?\n")
  if Truth(options["tamura"]):
    write("--> performing earth tide correction...")
    etide.correction(raw_data)
    ETC_flag = 1
    write("done.\n")
    #write("--> list of calculated ETC values:\n")
    #for i in range(len(raw_data)):
    # write("    %-20s %7.3f %7.3f\n"%(raw_data[i].station_id, raw_data[i].etc_correction, raw_data[i].etc))
  else:
    for i in range(len(raw_data)):
      raw_data[i].etc_correction = 0.0
    write("--> no earth tide corrections\n")
    ETC_flag = 0
  #
  # Tilt meter correction
  # correct for incorrectly set zero-levels for the tilt meters on the
  # instrument
  write("->> Correct Tilt Correction?\n")
  if Truth(options["tilt"]):
    Tip_flag = 1
    try:
      xe = float(options["xe"])
    except:
      xe = 0.0
    try:
      ye = float(options["ye"])
    except:
      ye = 0.0
    write("--> correcting meter tip/tilt corrections with xe=%f, ye=%f..."%(xe,
      ye))
    tilt.correct(raw_data, xe, ye)
    write("done.\n")
  else:
    write("--> not correcting meter tip/tilt corrections.\n")
    Tip_flag = 0
  #
  # correct the readings before averaging
  for i in range(len(raw_data)):
    raw_data[i].reading = raw_data[i].gravity
    raw_data[i].read_sig = raw_data[i].sigma
    raw_data[i].gravity = raw_data[i].gravity + raw_data[i].etc_correction + raw_data[i].atmospheric_correction

    ##ALGORITHM NOTE##
    # normally, we would update raw_data[i].sigma with the propogation of
    # the error of ETC and atmospheric corrections.  But, we know that
    # error propogates like sigma' = sqrt(sigma^2 + dETC^2 + dATM^2).
    # Now, dETC is of the order of 0.1 ugal = 0.0001 mgal, as is dATM
    # ==> sigma' ~ sqrt( (0.010)^2 + (0.0001)^2 + (0.0001)^2 ) =
    #              sqrt( 1e-4 + 2e-8 ) = sqrt( 1e-4 ) = 0.010
    # Hence, sigma' is vastly dominated by sigma, so we punt dETC and
    # dATM

  #
  # Quality Check the data
  (raw_data, options) = qc.check(raw_data, parms, names, options, write);

  #
  # view the data after corrections
  write("\n*** VIEW PRE_AVERAGE DATA ***\n\n")
  if Truth(options["processed_view"]):
    write(">>> starting data viewer...\n")
    # fire it up
    embeddedViewer.main(raw_data)
  else:
    write(">>> skipping view data\n")

  #
  # Average/extrapolate the data
  write("\n*** REDUCING CORRECTED GRAVITY TIME SERIES ***\n\n")
  if Truth(options["thiele"]):
    #
    # Thiele extrapolation
    write(">>> Using Thiele extrapolation algorithm\n");
    write("->> assuming exponential convergence.\n");
    if options.has_key("thiele_tolerance"):
      options["thiele_tolerance"] = float(options["thiele_tolerance"])
      write("->> setting equality tolerance to %.6e\n"%options["thiele_tolerance"])
    (data, skipped, dropped) = grav_util.thiele_extrapolate(raw_data, parms, names, options)
    for i in data.keys():
      data[i].sigma = data[i].raw_sigma

  else:
    #
    # Average the data w/w.avg
    write(">>> Using weighted averages\n");
    try:
      skip_time = float(options["skip"])
      write("--> skipping first %.3f minutes\n"%skip_time)
    except:
      write("!!! Invalid skip time - assuming skip of 0.0 minutes!\n")
      skip_time = 0.0
    #
    # average into GravityValue array for further processing
    write("->> averaging readings with identical station IDs")
    #   determine how many records to average and fit
    if not options.has_key("max_recs"):
      options["max_recs"] = -1
    max_recs = int(options["max_recs"])
    if Truth(options["grav_samples"]):
      write("; raw data are samples...")
    else:
      write("; raw data are averages...")
    (data, skipped, dropped) = grav_util.average_data(raw_data, parms, names, skip_time, max_recs, options["grav_samples"])
    for i in data.keys():
      data[i].sigma = data[i].raw_sigma

  write("done.\n")
  write("->> %d unique stations after time-series analysis.\n" % len(data))
  write("--> %d records skipped due to bad data.\n"%skipped)
  dropped += nuniqids - len(data)	# add stations completely removed by QC
  write("--> %d stations dropped due to insufficient good data.\n"%dropped)

  #
  # Filter averaged stations for insane values:
  #	i.e. time of 0.0
  dropped = 0
  for i in data.keys():
    if data[i].time == 0.0:
      # time of 0.0 puts us at ~2000BC, which is insane
      # indicates some station with bad data records, so we punt it
      del data[i]
      dropped = dropped+1
  write("--> %d stations dropped due to bad date (JD = 0)\n"%dropped)

  write(">>> Check repeat information for changes due to dropped stations...\n")
  (repeats, changes) = grav_util.check_proc_repeats(data, repeats)
  write("--> station repeat information has %d changes\n"%changes)
  del changes

  #
  # Post-TimeSeries Corrections:
  # 1) Ground Subsidence
  # 2) Instrument Drift
  write("\n*** CORRECTIONS TO OCCUPATION DATA ***\n\n")
  write(">>> Post-time-series Corrections\n")

  #
  # Ground Subsidence - correct gravity for elevation changes
  # assume dz file has delta_z in cm!
  # compute the correction
  write("->> Elevation Change Correction?\n")
  if Truth(options["dz"]):
    write("--> performing elevation change correction...")
    deltaz.correction(parms, data)
    write("done.\n")
  else:
    write("--> no elevation change corrections\n")
    for i in data.keys():
      data[i].dz = 0.0
      data[i].dz_correction = 0.0
  #
  # compute gravity with all corrections but Instrument Drift, to make
  # drift correction better.
  for i in data.keys():
    data[i].nodrift_gravity = data[i].raw_gravity - data[i].dz_correction
  #
  # Convert to partial days from first repeated station, then update start_jday
  index = repeats.keys()[0]
  min_time = data[index].time
  for i in repeats.keys():
    if min_time > data[i].time:
      min_time = data[i].time
      index = i
    for j in range(len(repeats[i])):
      if min_time > data[repeats[i][j]].time:
	min_time = data[repeats[i][j]].time
	index = repeats[i][j]
  minTime = data[index].time
  start_jday = minTime
  for k in data.keys():
    data[k].time -= start_jday
    data[k].start_time = start_jday

  #
  # Instrument Drift (also applies tare information)
  # compute drift function from reocupation file
  # file has list of station IDs that are reoccupations
  # of earlier stations
  write("->> Instrument Drift Correction?\n")
  if not options.has_key("order"):
    options["order"] = 0
  order = int(options["order"]); C = []
  if Truth(options["instrument_drift"]):
    write(">>> Performing drift correction.\n")

    if Truth(options["weighted_drift"]):
      write("->> Drift functions will be found with weighted fit.\n")
    else:
      write("->> Drift functions will be found with UNweighted fit.\n")
    write("--> Please wait; this may take a few minutes.\n");
    write("--> Start time for drift time axis: %s == %f\n"%(grav_util.datestr(start_jday), start_jday));
    write("--> Start time set from earliest station w/repeat in processed stations list!\n")

    if BatchMode:
      gui = 0	# no windows in batch mode
      if Truth(options["batch_drift_plot"]):
        gui = 2		# plot drift fn, etc. even in batch mode; requires X!
    else:
      gui = 1	# interactive drift plots
    (order, C) = drift.correction(repeats, data, tares, options, gui, write)
    write("--> Instrument drift correction done.\n");
    if len(C) > 0:
      write("--> Constants of drift function:\n");
      for i in range(len(C)):
	write("        C[%d] == %18.6f\n"%(i, C[i]))
    else:
      write("--> No constants ==> no corrections\n")
  else:
    write("--> No instrument drift correction computed or applied.\n")
    for i in data.keys():
      data[i].drift_correction = 0.0
  options["order"] = order	# update options to max order used

  # 
  # Final Gravity
  # reference back to the "base"
  # ask for base station's SID, then find the value and compute dg's from
  # that.  Write the result in a pretty format to disk.
  # Also write tare matrix!
  write("\n*** COMPUTE FINAL GRAVITY ***\n\n")
  for i in data.keys():
    g0 = data[i].nodrift_gravity
    dg = data[i].drift_correction
    data[i].gravity = g0 - dg
    # probably need to have some way to add in correction error to sigma
    data[i].sigma = data[i].raw_sigma
  #
  # Get gravity reference value
  if int(options["ref_type"]) == 2:
    write("->> Referencing gravity to average of station %s and its repeats\n"%options["ref_val"])
    # reference to a station
    base_id = options["ref_id"]
    if not base_id: # no reference id, maybe in reference value option
      base_id = options["ref_val"]
      options["ref_id"] = options["ref_val"]
      del options["ref_val"]	# drop ref_val for option output
    if data.has_key(base_id):
      gref = data[base_id].gravity
      if repeats.has_key(base_id):
	gref = grav_util.gref_average(base_id, data, repeats)
    else:
      write('!!! cannot find Station ID in IDs of data.  Reference gravity set to 0.0.\n')
      gref = 0.0
  elif int(options["ref_type"]) == 1:
    # absolute value offset
    write("->> Using absolute reading offset as reference\n")
    try:
      gref = float(options["ref_val"])
      write("--> reference gravity set to %f\n"%gref)
    except:
      write("!!! Invalid value %s - setting reference gravity to 0.0\n"%options["ref_val"])
      gref = 0.0
    base_id = "Absolute Reference"
  elif int(options["ref_type"]) == 3:
    # compute reference value to make station ID & reps a known value
    write("->> Computing reference to set average of station %s and repeats to %s\n"%(options["ref_id"], options["ref_val"]))
    base_id = options["ref_id"]
    try:
      if data.has_key(base_id):
	stn = data[base_id].gravity
	if repeats.has_key(base_id):
	  stn = grav_util.gref_average(base_id, data, repeats)
	grav = float(options["ref_val"])
	gref = stn - grav	# when subtracted from station, result is grav...
      else:
	write('!!! cannot find Station ID in IDs of data.  Reference gravity set to 0.0.\n')
	gref = 0.0; grav=0.0;
      write("--> reference gravity set to %f\n"%gref)
    except:
      write("!!! Invalid value %s - setting reference gravity to 0.0\n"%options["ref_val"])
      gref = 0.0
    base_id = "%.3f ==> set %s to %.3f"%(gref, options["ref_id"], grav)
  else:
    write("!!! Unknown reference type; setting reference ")
    write("gravity to 0.0\n")
    base_id = "None"
    gref = 0.0
  #
  # compute relative gravity
  for i in data.keys():
    data[i].deltag = data[i].gravity - gref
  write(">>> Final gravity computed relative to %s\n"%base_id)

  write("\n*** OUTPUT REDUCED DATA ***\n\n")
  write("#-----------------------\n")
  write("# Select or enter the filename of the output file.  Cancelling the dialog cancels all output.\n")
  write("#-----------------------\n")
  #
  # Output the reduced data, tare and drift functions
  #
  if string.upper(atm_file) == "NONE":
    atm = 0
  else:
    atm = 1
  settings = fileop.Settings(options, base_id, gref, C, tares, start_jday)

  if BatchMode:
    if options["out_name"] and string.upper(options["out_name"]) != "NONE":
      out_name = options["out_name"]
      if out_name != options["raw_file"]:
	if not fileop.output_data(options["out_name"], data, names, settings):
	  write("!!! Could not open %s for write.  Data not written.\n"%out_name)
	else:
	  write("--> Final gravity written to file %s\n" % options["out_name"])
      else:
	write("!!! Cowardly refusing to overwrite raw data file!\n");
    else:
      write("!!! Final data not outputed\n")
      out_name = options["out_name"]

  else:
    while 1:
      out_name = tkFileDialog.asksaveasfilename(parent=root, title="Final Output File",
	filetypes=out_filetypes)
      if not out_name:
	write("!!! Final data not outputed\n")
	tkMessageBox.showwarning("No Output", "Filename entry cancelled; data not output.")
	out_name = "None"
	break
      if out_name != options["raw_file"]:
	if not fileop.output_data(out_name, data, names, settings):
	  write("!!! Could not open %s for write.  Try again.\n"%out_name)
	else:
	  write("--> Final gravity written to file %s\n" % out_name)
	  tkMessageBox.showinfo("Data Written", "All data output to file %s"%out_name)
	  break
      else:
	write("!!! Cowardly refusing to overwrite raw data file!\n");
	tkMessageBox.showwarning("No Output", "Refusing to overwrite raw data file; data not output.")

  options["out_name"] = out_name


  if not options["tare_file"]:
    options["tare_file"] = options["out_name"]

  control_file = "reduce.cmd"
  if not fileop.writeCmdFile(control_file, options):
    write("!!! Unable to write control file %s.\n"%control_file)
  else:
    write("->> Control file written to %s.\n"%control_file)
  write("Completed %s\n"%time.asctime(time.localtime(time.time())))
  sys.exit()

############
# END MAIN #
############

# RUN IT
try:
  main()
except SystemExit:
  sys.exit(1)
except:
  print "Unhandled exception!  Aborting program."
  print "Results not saved; output files suspect!"
  print "Fix problem and try again."
  print "Program execution traceback follows; submit with bug report."
  traceback.print_exc();
  sys.exit(0);
