#!/usr/bin/python

## Use the gravity and water depth data to compute effective
## density change.
##
## Assume Bouguer slab is valid (lateral extent >20x depth)
##
## Gravity and water depth data in following formats:
##  GRAVITY - plottable output of generate_table script; files in plot/
##	YYYY/M/D   StationID  dG  Sigma
##  WELL DATA - simple 2 column format
##	YYYY/M/D  Depth
##
## Compute porosity as a function of time for each well data point.
##
## Porosity computed from: dG(uGal) = 40 * dp (g/cc) * dh (m)
##   where dp is effective density change
##         dh is change in water level, in meters
##         dG is gravity change in microGal
## Since the density change is equivalent to porosity change, can use
## it as a proxy measurement.


# GCONST is the constant term in the Bouguer relation, above
GCONST = 40 * 0.3048   # depths in feet, not meters; hence the 0.3048

import sys, string
import grav_util

class GravityData:
  def __init__(self, g, s):
    self.G = g;
    self.S = s;

def usage():
  print "usage: %s gravity_file well_file"%sys.argv[0]

def main():
  if len(sys.argv) < 3:
    usage();
    sys.exit(0);

  grav = {}
  # open the gravity file and read data
  lines = grav_util.get_file(sys.argv[1]);
  for line in lines:
    fields = string.split(line);
    jd = grav_util.str2jd(fields[0], "12:00:00");
    dg = float(fields[2]);
    sig= float(fields[3]);
    grav[jd] = GravityData(dg, sig);

  # sort the gravity data dates
  keys = grav.keys();
  keys.sort(grav_util.num_sort);
  JDmax = max(keys);
  JDmin = min(keys);

  well = {}
  # read the well data
  lines = grav_util.get_file(sys.argv[2]);
  for line in lines:
    fields = string.split(line);
    jd = grav_util.str2jd(fields[0], "12:00:00");
    depth = float(fields[1]);
    well[jd] = depth;

  wkeys = well.keys();
  m = min(wkeys);
  o = well[m]
  for i in wkeys:
    well[i] = well[i] - o;

  print "#___DATE___ __TIME__ _dDEPTH_ ___dG___ Eff._Density"
  print "#                     (feet)   (uGal)    (g/cm^3)"
  wkeys.sort(grav_util.num_sort);
  for jd in wkeys:
    # find nearest gravity data
    if jd>JDmax or jd<JDmin:
      # outside gravity surveys, reject
      continue;

    m = JDmin; M = JDmax;
    for k in keys:
      if k<=jd:
        m = k;
      if k> jd:
        M = k;
    # interpolate gravity change at well data time
    if m == M:
      dG = grav[m].G
    else:
      dG = grav[m].G + (grav[M].G - grav[m].G)/(M - m) * (jd - m);
    # compute density
    if well[jd] != 0:
      dp = dG / ( GCONST * well[jd] )
    else:
      dp = 0.0

    # print it out
    print " %s %8.3f %8.3f %12.8f"%(grav_util.datestr(jd), well[jd], dG, dp);

# run it
main()
