# Marquardt-Levenberg non-linear least squares inversion
# Iterative technique, combining a steepest descent and chi-square
# minimization.  Algorithm taken from Numerical Recipes in C, 2nd Ed.

from scipy import *
from scipy.linalg import *
from math import *

# Requires NUMERICAL PYTHON extensions, for matrix stuff

# The following are examples of the fit function and derivative
# not tested!  Illustrative examples only!
def example_f(x, a):
  z = a[0] + a[1]*float(x) + a[2]*(float(x)**2)
  return z

def example_df(x, a):
  # for example_f, above, we compute df/da[i] for all i

  # create a copy of a, so we have enough entries in the result list
  z = a[:]

  # df/da[0]
  z[0] = 1.0
  # df/da[1]
  z[1] = float(x)
  # df/da[2]
  z[2] = float(x)**2.0
  return z

#############################################################################
# Marquardt inversion function
# input is:
# x - array of measured x values
# y - array of measured y values
# s - array of sigma values for y, used as weighting factors
# a - initial guessed values for fit parameters - try to make them close to
#	the real answer (converges much faster that way)
# ia - list that determines what parameters are fit:
#	1 ==> fit, 0 ==> fixed
#	MUST BE INTEGER LIST (not tuple)!
#	order is same as a[]
# f - object that is function to fit, taking x point, and "a" vector
# df - object that computes df/da for all a, and returns vector of
#	df/da
# eps - convergence limit; stop iterations when change in chi^2 less
#	than eps; defaults to 1e-5
# debug - If 1, prints status every iteration
#
# returns a tuple of:
# a - updated function parameters
# cov - covariance matrix, as MxM matrix (M is # of parameters to fit)
# C2 - final chi-squared value for fit
# ni - number of iterations in the fit
# if ni >= MAX_ITER, do not trust results!
#############################################################################
def fit(x, y, s, a, ia, f, df, eps=1e-5, debug=0):
  # failsafe, we exceed this, it isn't converging
  MAX_ITER = 1500

  # L is lambda, the scale of the step size
  L = 0.1

  # compute initial chi^2
  C2 = chi2(x, y, s, a, f)

  # compute size of linear system - i.e. number of parameters to fit
  M = ia.count(1)

  # setup counters, etc.
  ni = 1
  dC2 = eps+1

  # Status printing for debugging output
  if debug:
    print "%4d: %14f %s"%(ni-1, C2, a)

  # start iterations
  while abs(dC2) > eps or dC2 > 0:
    # construct linear system of equations
    beta = zeros((M,), Float)
    gamma = zeros((M, M), Float)

    # loop over data
    for i in range(len(y)):
      # compute df/da[] for this point
      dfda = df(x[i], a)
      # compute 1/s**2
      S = 1.0 / (s[i]**2)
      # compute y[i] - f()
      dy = y[i] - f(x[i], a)
      # compute addition to gamma, beta
      j = 0;
      for l in range(len(a)):
	if ia[l]:
	  wt = dfda[l]*S
	  k = 0
	  for m in range(len(a)):
	    if ia[m]:
	      gamma[j,k] = gamma[j,k] + wt*dfda[m]
	      k = k+1
	    beta[j] = beta[j] + dy*wt
	  j = j+1

    # now we have gamma, and need to compute alpha
    alpha = zeros((M, M), Float)
    for j in range(M):
      for k in range(M):
	if j == k:
	  alpha[j,k] = gamma[j,k] * (1 + L)
	else:
	  alpha[j,k] = gamma[j,k]

    # now compute update to a
    da = solve_da(alpha, beta)

    # Check to see if we improved chisq
    atry = a[:]
    j = 0
    for i in range(len(a)):
      if ia[i]:
	atry[i] = atry[i] + da[j]
	j = j+1

    newC2 = chi2(x, y, s, atry, f)

    dC2 = newC2 - C2

    if dC2 < 0:
      # success, keep this solution
      L = L*0.1
      C2 = newC2
      for i in range(len(a)):
	a[i] = atry[i]
    else:
      # not an improvement, so try again with bigger L
      L = L*10.0
    ni = ni+1
    # Status printing for debugging output
    if debug:
      print "%4d: %14f %s"%(ni-1, C2, a)

    if ni > MAX_ITER:
      dC2 = 0.0
    # end of while() loop
  # iterations converged, so we return the new a, and the covariance
  # matrix
  # covariance matrix is inverse of alpha with L=0, which is inverse of
  # gamma
  cov = pinv2(gamma)

  return (a, cov, C2, ni)

# functions used in inversion
def chi2(x, y, s, a, f):
  chisq = 0.0
  for i in range(len(y)):
    chisq = chisq + ( (y[i] - f(x[i], a))/s[i] )**2
  return chisq

def solve_da(alpha, beta):
  # NumPy provides a generalized inverse function, so we don't have to
  # compute from SVD of alpha
  I = pinv2(alpha)
  da = dot(I, beta)
  return da
