###
# Read a DEM file from disk, in GridFloat format, and store in
# an internal object of type FloatDEM.  The obj then has the
# DEM data in an array (set/get with obj.set/get or setLL/getLL).
#
# DEM data assumed in 2 parts; binary file of 4-byte floats and header
# of corner lat/lon, cell size, etc. in text format.  This format is
# specified as "GridFloat" by USGS Seamless Data server for Nat'l Elevation
# Data; e.g. 1 arcsec (30m), 1/9 arcsec, etc. coverage.
#
# Download chunks and then read using this code or others.
#
# Note that fromfile(filename) assumes the 2 files are filename.flt and
# filename.hdr (lowercase!).  If this is wrong, call get_header() and
# get_data with correct names....
#
# Part of micrograv.  Copyright 2009 Paul Gettings
#
# Note that we use struct to unpack the floats from on-disk order to
# local-machine order and then store into an array of floats.
#

import array, struct, sys
from math import *

class FloatDEM:
  def __init__(self):
    self.cellsize=0;
    self.lat = -91;
    self.lon = -181;
    self.nrows = 0;
    self.ncols = 0;
    self.byteorder = 0; # 0 is unknown, 1 is lsb, 2 is msb
    self.nodata = None	# no data value for testing!
    self.data = None	# array of DEM values

  # read the header info from filename.hdr and the data from
  # filename.flt
  def fromfile(self, filename):
    dfile = "%s.flt"%filename
    hfile = "%s.hdr"%filename

    # get the header info
    self.get_header(hfile)
    # fill data array
    self.get_data(dfile)

  def get_header(self, fname):
    file = open(fname, mode="rt")
    lines = file.readlines(); file.close()
    for line in lines:
      line = line.strip()
      if not line: continue
      if line[0] == "#": continue
      (token, value) = line.split(None, 1);
      token = token.upper()
      if token == "NCOLS":
        self.ncols = int(value)
      elif token == "NROWS":
        self.nrows = int(value)
      elif token == "CELLSIZE":
        self.cellsize = float(value)
      elif token == "XLLCORNER":
        self.lon = float(value)
      elif token == "YLLCORNER":
        self.lat = float(value)
      elif token == "NODATA_VALUE":
        self.nodata = float(value)
      elif token == "BYTEORDER":
        if value.upper() == "LSBFIRST":
          self.byteorder = 1
        elif value.upper() == "MSBFIRST":
          self.byteorder = 2
        else:
          self.byteorder = 0
      else:
        sys.stderr.write("FloatDEM.get_header: unknown token %s\n"%token)
    # convert from lower left (SW) corner to upper left (NW)
    self.lat += (self.nrows*self.cellsize);

    # allocate our array of values
    self.data = array.array('f');	# min of 4-byte floats

  def get_data(self, fname):
    file = open(fname, "rb")	# MUST BE BINARY OPEN!!!!
    if self.byteorder == 1:	# LSB
      fmt="<"
    elif self.byteorder == 2:	# MSB
      fmt=">"
    else:			# guess native and hope...
      fmt="="
    chunkSize = self.ncols
    count = 0;
    fmt += "%df"%chunkSize	# construct format of byte-order f
    for i in xrange(self.nrows*self.ncols/chunkSize):
      inc = file.read(chunkSize*4)	# GridFloat GUARANTEES 4-BYTE FLOATS!!!!
      count += len(inc)
      if len(inc) != chunkSize*4:	# short read prob'ly means error!!!
        sys.stderr.write("FloatDEM.get_data: Unexpected EOF from data file; trying for %ld x %ld elements total; got %d of %d bytes.\n"%(self.nrows, self.ncols, count, self.nrows*self.ncols*4))
        sys.exit(1);
      # append this row to array
      self.data.extend(struct.unpack(fmt, inc));
    # done; self updated with data
    file.close()

  def get(self, row, col):
    if row<0 or col<0: return None;
    if row>= self.nrows: return None;
    if col>= self.ncols: return None;
    return self.data[row*self.ncols + col]

  def set(self, row, col, val):
    if row<0 or col<0: return;
    if row>= self.nrows: return;
    if col>= self.ncols: return;
    self.data[row*self.ncols + col] = float(val);

  # convert from lat/lon to row, col indexes
  def LL2rc(self, lat, lon):
    if lat < (self.lat - self.nrows*self.cellsize): return (None, None);
    if lon > (self.lon + self.ncols*self.cellsize): return (None, None);
    if lat > self.lat: return (None, None);
    if lon < self.lon: return (None, None);

    row = int(fabs(self.lat-lat)/self.cellsize);
    col = int(fabs(self.lon-lon)/self.cellsize);
    return (row, col)

  # Get and set values by lat, lon
  def getLL(self, lat, lon):
    (row, col) = self.LL2rc(lat, lon)
    if row == None:
      return None
    else:
      return(self.data[row*self.ncols + col]);

  def setLL(self, lat, lon, val):
    (row, col) = self.LL2rc(lat, lon)
    if row != None:
      self.data[row*self.ncols + col] = float(val)

  # convert from row or column to lat or lon
  def row2lat(self, row):
    if row<0: return None;
    if row>= self.nrows: return None;
    return self.lat-row*self.cellsize	# origin N, rows move S!

  def col2lon(self, col):
    if col<0: return None;
    if col>= self.ncols: return None;
    return self.lon+col*self.cellsize	# origin W, columns move E

  # Extents of DEM area
  def north(self):
    return self.lat
  def south(self):
    return self.lat - self.nrows*self.cellsize;
  def east(self):
    return self.lon + self.ncols*self.cellsize;
  def west(self):
    return self.lon

  # Compute a Transverse Mercator projection, centered at lat0,lon0,
  # for the location lat,lon; returns offsets in m from lat0,lon0
  # ALL INPUTS IN DEGREES!!!
  def projectTM(self, lat, lon, lat0, lon0):
    lat = radians(lat); lon = radians(lon);	# convert to radians
    lat0 = radians(lat0); lon0 = radians(lon0);
    k0 = 1.0;	# see Map Projections - A Working Manual by Snyder, USGS for terms...
    R = 6371204;	# radius of Earth, m
    B = cos(lat)*sin(lon - lon0)
    x = 0.5*R*k0 * log( (1+B) / (1-B) )
    y = R*k0 * ( atan( tan(lat)/cos(lon-lon0) ) - lat0 )
    return (x, y)

  # Convert a row,col index into meters from NW corner of
  # the DEM, using a Transverse Mercator projection; see LL2xy()
  def rc2xy(self, row, col):
    lat = self.row2lat(row); lon = self.col2lon(col)
    return self.LL2xy(lat, lon)

  # project a lat/lon to an X,Y offset in meters from NW corner of DEM
  # Uses the Transverse Mercator projection, with central meridian at
  # lon0 and central "equator" at lat0
  def LL2xy(self, lat, lon):
    # convert from lat,lon to x,y in m
    return self.projectTM(lat, lon, self.south(), self.west())

