##
## File operations for reduce.py, other programs that
## are not specific to a meter type
##

import sys, string, re, time
import grav_util
from grav_util import Truth
from grav_data import *

def get_sta_parms(name):
  # open file, read it in, and return station parameters
  file = open(name, "rt")

  lines = file.readlines()
  file.close()
  parms = {}
  for i in range(len(lines)):
    line = string.strip(lines[i])
    if string.upper(line[0:12]) == '[PARAMETERS]':
      i = i+1 # prevent infinite loop :)
      for j in range(i, len(lines)):
	line = string.strip(lines[j])
	if not line:
	  # blank line - ignore
	  continue
	if line[0] == "[":
	  # "[" implies a new section, so end this one and return
	  break
	if line[0] == "#":
	  # comment line, ignore
	  continue
	fields = string.split(line)
	incoming = Incoming()
	incoming.station_id = fields[0]
	incoming.lat = to_float(fields[1])
	incoming.lon = to_float(fields[2])
	incoming.elevation = to_float(fields[3])
	incoming.Z = to_float(fields[4])
	parms[fields[0]] = incoming
  return parms

def get_sta_repeats(name):
  # open file, read it in, and return station repeats
  file = open(name, "rt")

  lines = file.readlines()
  file.close()
  repeats  = {}
  for i in range(len(lines)):
    if string.upper(string.strip(lines[i][0:9])) == '[REPEATS]':
      i = i+1 # prevent infinite loop :)
      for j in range(i, len(lines)):
	line = string.strip(lines[j])
	if not line:
	  # blank line - ignore
	  continue
	if line[0] == "[":
	  # "[" implies a new section, so end this one and return
	  break
	if line[0] == "#":
	  # comment line, ignore
	  continue
	fields = string.split(line)
	if not repeats.has_key(fields[0]):
	  repeats[fields[0]] = [fields[1]]
	else:
	  repeats[fields[0]].append(fields[1])
  return repeats

def get_sta_names(name):
  # open file, read it in, and return station names
  file = open(name, "rt")

  lines = file.readlines()
  file.close()
  names  = {}
  for i in range(len(lines)):
    line = string.strip(lines[i])
    if string.upper(line[0:7]) == '[NAMES]':
      i = i+1 # prevent infinite loop :)
      for j in range(i, len(lines)):
	line = string.strip(lines[j])
	if not line:
	  # blank line - ignore
	  continue
	if line[0] == "[":
	  # "[" implies a new section, so end this one and return
	  break
	if line[0] == "#":
	  # comment line, ignore
	  continue
	fields = string.split(line, None, 1)
	names[fields[0]] = fields[1]
  return names

def get_sta_skips(name):
  # open file, read it in, and return station names
  file = open(name, "rt")

  lines = file.readlines()
  file.close()
  skips = {}
  for i in range(len(lines)):
    line = string.strip(lines[i])
    if string.upper(line[0:7]) == '[SKIPS]':
      i = i+1 # prevent infinite loop :)
      for j in range(i, len(lines)):
	line = string.strip(lines[j])
	if not line:
	  # blank line - ignore
	  continue
	if line[0] == "[":
	  # "[" implies a new section, so end this one and return
	  break
	if line[0] == "#":
	  # comment line, ignore
	  continue
	sid = string.strip(line)
	skips[sid] = 1
  return skips

class Settings:
  def __init__(self, options, base, gref, coeff, tares, start):
    self.options = options	# options hash!!!
    self.baseID = base
    self.gref = gref
    self.C = coeff
    self.Tare = tares
    self.start_jday = start

  Messages = {
    "atmosphere": "#Corrected for atmospheric pressure",
    "drift": "#Removed meter-applied drift correction" ,
    "dz": "#Applied elevation correction change",
    "gmt_offset": "#Use supplied GMT offset rather than data file offset",
    "grav_samples": "#Raw data are samples, not averages",
    "instrument_drift": "#Compute instrument drift function",
    "weighted_drift": "#Instrument drift functions use weighted fit",
    "longman": "#Removed meter-applied ETC correction",
    "meter_file": "#Meter-information file: ",
    "meter_type": "#Data comes from meter of type: ",
    "order": "#Max order of polynomial instrument drift fns: ",
    "start_order": "#Min order of polynomial instrument drift fns: ",
    "raw_file": "#Raw data file: ",
    "sigma_threshhold": "#QC s.d. threshhold for flagging: ",
    "skip": "#Time skipped at start for weighted averages: ",
    "slope_threshhold": "#QC slope threshhold for detrending time-series: ",
    "tamura": "#Applied Tamura ETC",
    "tare_file": "#Tares read from file: ",
    "temp_correct": "#QC estimated correct temps and corrected gravity values",
    "temp_remove": "#QC removed temps outside threshhold",
    "temp_threshhold": "#QC temp threshhold for correction: ",
    "temp_threshhold_drop": "#QC temp threshhold for removal: ",
    "thiele": "#Used Thiele extrapolation instead of weighted averages",
    "thiele_filt_radius": "#Thiele filtering radius, in data pts: ",
    "thiele_tolerance": "#Equality tolerance for Thiele extrap.: ",
    "tilt": "#Corrected instrument tilt corrections",
    "xe": "#X tilt error: ",
    "ye": "#Y tilt error: "
  }

  AdditionalWriteOptions = {
    "meter_file":1,
    "sigma_threshhold":1,
    "skip":1,
    "slope_threshhold":1,
    "tare_file":1,
    "temp_correct":1,
    "thiele":1,
    "thiele_filt_radius":1,
    "thiele_tolerance":1,
    "temp_remove":1,
    "temp_threshhold":1,
    "temp_threshhold_drop":1,
    "xe":1,
    "ye":1,
    "start_order":1,
    "order":1
    }

def output_data(out_name, data, names, settings):
  try:
    file = open(out_name, "wt")
  except:
    return 0

  for i in data.keys():
    data[i].name = "UNKNOWN"
    if names.has_key(data[i].station_id):
      data[i].name = names[data[i].station_id]

  file.write("#PROCESSED GRAVITY DATA\n")
  file.write("#output at %s\n"%time.asctime(time.localtime(time.time())))
  file.write("#Processing options:\n");

  for k in settings.options.keys():
    if Truth(settings.options[k]):
      if Settings.Messages.has_key(k):
        file.write("#%s"%Settings.Messages[k]);
        if Settings.AdditionalWriteOptions.has_key(k):
          file.write("%s"%settings.options[k])
	file.write("\n");

  file.write("#Reference Base Station: %s\n"%settings.baseID)
#  if settings.baseID == "Absolute Reference":
#    settings.gref = settings.gref*-1.0
  file.write("#Reference Gravity: %.3f\n" % settings.gref)

  file.write("#     STATION NAME             STATION ID    GRAVITY SIGMA    DATE/TIME(GMT)         LONG       LAT      HEIGHT DELTA Z   DZ C    DRIFT RAW GRAV  R SIG\n")
#SPACING HELP 1234567890123456789012345 123456789012345 1234567890 12345 12345678901234567890 12345678901 12345678901 1234567 1234567 1234567 1234567 12345678 123456
  keys = data.keys()
  keys.sort(grav_util.num_sort)
  for i in keys:
    if data[i].sigma < 0.001:
      sigma = 0.001
    else:
      sigma = data[i].sigma
  # write data: Name SID  f_grav fsig Date lon   lat  height  dz  dz_c  drift raw_g raw_s
    file.write("%25s %15s %10.3f %5.3f %20s %11.6f %11.6f %7.3f %7.3f %7.3f %7.3f %8.3f %6.4f\n" %
	(data[i].name, data[i].station_id, data[i].deltag, sigma,
	grav_util.datestr(data[i].time + settings.start_jday),
	data[i].lon, data[i].lat, data[i].elevation, data[i].dz, data[i].dz_correction,
	data[i].drift_correction, data[i].raw_gravity, data[i].raw_sigma))
  file.write("\n#End of processed data\n")
  # write tare data
  write_tares(settings.Tare, file)
  file.close()
  return 1

def write_tares(tares, file):
  file.write("\n#TARE DATA\n")
  k = tares.keys()
  k.sort()
  for i in k:
    file.write("#%s (%14.5f)  %8.3f\n" % (grav_util.datestr(i),
		i, tares[i]))
  file.write("#End tare data\n");

def get_tares(name):
  try:
    file = open(name, "rt")
  except:
    return None

  T = {}

  regex=re.compile("^.* \((.*?)\)")

  while 1:
    line = file.readline()
    if not line: break
    if string.upper(line[0:5]) == "#TARE":
      # start tare data block
      while 1:
	line = file.readline()
	if not line: break
	if string.upper(line[0:14]) == "#END TARE DATA":
	  break
	# regexp match
	try:
	  (toss, jdstr, tstr) = regex.split(line)
	except:
	  continue
	T[to_float(jdstr)] = to_float(tstr)
      break
  file.close()
  return T


def grokCmdFile(name):
   # read and parse command file
   options = defaultOptions()

   comment = re.compile("^[ \t]*#")

   try:
     file = open(name, "rt")
   except:
     return {}
   if not file:
     return {}

   lines = file.readlines()
   file.close()
   for line in lines:
     # strip newline, leading whitespace
     line = string.strip(line);
     if comment.search(line):
       # comment line, toss
       continue
     # split and parse
     try:
       (key, val) = string.split(line, None, 1)
     except ValueError:
       F = string.split(line)
       key = F[0]
       val = ""
     key = key.lower()	# make sure lower case
     if key == "exp_guess" and val != "":
       # parse into tuple
       val = parse_num_tuple(val)

     if key == "detrend" and val != "":
       val = parse_comma_dict(val)

     if key == "detrend_skip" and val != "":
       val = parse_comma_dict(val)

     if key == "gmt_offset":
       if val.lower()=="none": continue	# skip if None!
       val = to_float(val)

     if key == "thiele_filt_radius":
       val = int(to_float(val))

     if key == "temp_threshhold":
       val = to_float(val)

     options[key] = val
   return options

def writeCmdFile(name, options):
  # write out cmd file from options
  # open file
  out = open(name, "wt")
  if not out:
    return 0

  # dump the option database, which has it all
  k = options.keys()
  k.sort()
  for i in k:
    if i == "detrend":
      out.write("%-20s\t"%i);
      for j in options[i].keys():
        out.write("%s,"%j)
      out.write("\n");
    elif i == "detrend_skip":
      out.write("%-20s\t"%i);
      for j in options[i].keys():
        out.write("%s,"%j)
      out.write("\n");
    else:
      if options[i] == "":
        options[i] = "0"
      out.write("%-20s\t%s\n"%(i, options[i]))

  out.close()
  return 1


def parse_num_tuple(val):
  cut = re.compile("\s*[[(]?([0-9.]*),\s*([0-9.]*),\s*([0-9.]*)[])]?")
  F = cut.search(val).groups()
  a = F[0]; b = F[1]; c = F[2]
  a = to_float(a); b = to_float(b); c = to_float(c)
  val = [a, b, c]
  return val

def parse_comma_dict(val):
  D={}
  F = string.split(val, ",")
  for f in F:
    if f == "": continue
    D[f] = 1
  return D

def to_float(string):
  # wrapper around float()
  if not string or string == "None":
    string = "0"
  try:
    V = float(string)
  except ValueError:
    raise ValueError, "Invalid value %s"%string
  return V

def defaultOptions():
  # return a dict with all the options set to default values
  o = {};
  o["atm_file"]=            	"NONE"
  o["atmosphere"]=          	0
  o["base_pressure"]=       	0.0
  o["batch_drift_plot"]=	"yes"
  o["drift"]=               	0
  o["dz"]=                  	1
  #o["gmt_offset"]=		-6 # if present(!), reset ALL records!
  o["grav_samples"]=		0 # if 1, gravity vals are samples, not aves
  o["instrument_drift"]=    	1
  o["weighted_drift"]=		1 # if true, use weighting in instrument drift functions
  o["longman"]=             	1
  o["meter_file"]=		"IGNORED"	# ignored for CG-3/CG-5
  o["meter_type"]=		"CG-3"		# compare "ALIOD"
  o["order"]=               	0		# max order for poly drift
  o["start_order"]=             0		# starting order for poly drift
  o["out_name"]=            	"DIR/OUTPUT"
  o["parm_file"]=           	"DIR/PARAMETER"
  o["processed_view"]=      	0
  o["raw_file"]=            	"DIR/INPUT"
  o["raw_view"]=            	0
  o["ref_type"]=            	1
  o["ref_val"]=             	0.
  o["sigma_threshhold"]=	0.05
  o["skip"]=                	3.0
  o["slope_threshhold"]=	2.7e-5
  o["staircase_drift"]=		1
  o["tamura"]=              	0
  o["tare_file"]=           	"DIR/OUTPUT"
  o["temp_correct"]=		0	# if true, correct temp rdgs outside threshhold
  o["temp_correct_debug"]=	0	# if true, print output with orig and new temp/grav
  o["temp_remove"]=		0	# if true, remove points with temps outside temp_threshhold
  o["temp_threshhold"]=		4.0	# threshhold for temp_correct; abs value of temp is used!
  o["temp_threshhold_drop"]=	4.0	# threshhold for temp_remove; temp_remove will use temp_threshhold if temp_threshhold_drop not present
  o["thiele"]=              	0		# if 1, use Thiele
  o["thiele_filt_radius"]=	30	# radius of filter window, in pts
  o["thiele_tolerance"]=	1e-7	# tolerance for "equal" time-series values
  o["tilt"]=                	0
  o["use_gnuplot"]=		0	# use gnuplot for plots
  o["xe"]=                  	0.0
  o["ye"]=                  	0.0

  return(o);

