#!/usr/bin/python

# Given a terrain-corrected exploration file (reduce.py -> reduce2expl
# -> terrain -> exploration.py), compute CBGA at specified densities...

# line width:########################################################
VERSION  =  "##            Version 1.3.0 (Feb 2013)                ##";

# system modules
import sys
import Tkinter
import tkFileDialog, tkMessageBox, tkSimpleDialog
import time
from types import *
import traceback
from math import *

# custom modules
import explfileop
from grav_util import Truth
import grav_util
from grav_data import *
import embeddedViewer, StatusWindow
import logger
import spherical_cap

def debug_status(write):
  return
  write("*** CURRENT STATUS OF CODE ***\n")
  write("Known Issues:\n")
  write("  None\n")
  write("******************************\n")

def main():
  global VERSION
  global write, log
  BatchMode = False
  ##Parse command-line args
  for i in range(len(sys.argv)):
    if sys.argv[i] == '-F':
      # batch mode - get input from control file
      BatchMode = True
      try:
	controlfilename = sys.argv[i+1]
      except:
	sys.stderr.write("Must supply control file name; falling back to interactive mode\n")
	BatchMode=False

  options = {}
  if BatchMode:
    # create logger instance 
    log = logger.Logger("exploration.log")
    # set output function
    write = log.scrn
    options = explfileop.grokCmdFile(controlfilename)
    if not options:
      # scream and die
      print "CANNOT PARSE COMMAND FILE %s. ABORT."%(controlfilename)
      sys.exit(0)

  else:
    ##Convenience variables
    data_filetypes = (("Reduced Data Files","*.exp"), ("All Files","*"))
    out_filetypes  = (("CBGA Files","*.cbga"), ("All Files","*"))

    ##Fire up the GUI
    root = Tkinter.Tk()
    root.title("Calculate CBGA")
    # create status window
    status_win = StatusWindow.StatusWindow(root, 150,30);
    status_win.show()

    # create logger instance 
    log = logger.Logger("exploration.log", window=status_win)

    # set output function
    write = log.win

    # get default control options
    options = explfileop.defaultOptions()

  # timestamp the log files
  write("# Started %s\n"%time.asctime(time.localtime(time.time())))
  write("########################################################\n")
  write("##      Exploration Gravity Anomaly Computations      ##\n")
  write("%s\n"%VERSION)
  write("##                   Paul Gettings                    ##\n")
  write("########################################################\n")
  write("\n*** READ REDUCED GRAVITY DATA ***\n\n")

  if not BatchMode:
    write("#-----------------------\n")
    write("# Select the gravity data file.  The file should be in the format\n")
    write("# output by 'reduce2expl'; comments are OK.\n")
    write("#-----------------------\n")
    root.update()
  #
  # Read Raw Data
  if BatchMode:
    raw_file = options["raw_file"]
  else:
    raw_file = tkFileDialog.askopenfilename(parent=root, title="Gravity Data File", initialdir=".", filetypes=data_filetypes)

  if not raw_file:
    write( "!!! must enter a processed data filename!\n")
    if not BatchMode:
      tkMessageBox.showerror("No File Name", "Must enter a data filename!")
    sys.exit()

  write(">>> Reading station data file\n")
  try:
    data = explfileop.readData(raw_file, write)
  except Exception, e:
    write("!!! error processing data file %s\nException: %s\n"%(raw_file, e))
    if not BatchMode:
      tkMessageBox.showerror("Data File Error", "Error processing data file %s!"%raw_file)
    sys.exit()
  write("--> Processed gravity data file '%s' with %d stations\n"%(raw_file, len(data)))

  options["raw_file"] = raw_file

  write("\n*** GET REDUCTION DENSITIES ***\n\n")
  # Get CBGA densities for computations
  if BatchMode:
    rhoDict = options["densities"]; # get density dict from option file
    write(">>> Using densities: %s\n"%rhoDict.keys())
  else:
    rhoString = tkSimpleDialog.askstring("Density list",
      "Enter a space-separated list of densities, in kg/m^3; density of 0 for FAA",
      parent=root, initialvalue="2670")
    rhoDict={}
    fields=rhoString.split()
    for s in fields:
      rhoDict[float(s)] = 1;
  densities=rhoDict.keys()
  options["densities"]=rhoDict;
  densities.sort(grav_util.num_sort);

  write("\n*** COMPUTE CBGA FOR ALL STATIONS, ALL DENSITIES ***\n\n")
  write(">>> 2% complete for each .\n");
  write("--> ");
  # constants for GRS80, from Hinze et al(2005)
  g_e = 978032.67715 # mGal
  k = 1.931851353e-3
  e2 = 6.69438002290e-3
  # run CBGA for each density...
  n=0; interval = max((1,int(len(data)/50)));
  for stn in data.keys():
    # theoretical gravity:
    # phi is station lat
    # ThG = ge*(1+k*sin^2(phi)) / sqrt(1-e^2*sin^2(phi))
    lat = radians(data[stn].lat)
    s2 = sin(lat) * sin(lat)
    ThG = g_e * (1 + k*s2) / sqrt(1 - e2*s2)

    # free-air correction from Hinze et al(2005)
    FAC = 7.2125e-8*data[stn].elev*data[stn].elev - (0.3087691 - 4.398e-4*s2)*data[stn].elev

    # atmospheric correction; account for height differences
    # empirical equation for elevations in m to 10 km ASL
    ATM = 0.874 - 9.9e-5*data[stn].elev + 3.56e-9*data[stn].elev*data[stn].elev

    ## Curvature correction
    #E = data[stn].elev * 3.048	# convert to feet
    #CC = 4.462e-4*E - 3.282e-8*E*E + 1.27e-15*E*E*E

    data[stn].cbga = {}
    # Bouguer slab
    n+=1;
    if n%interval==0: write(".")
    for rho in densities:
      BSC = 4.19357380320026e-05 * rho * data[stn].elev # mGal, for rho in kg/m^3, elev in m
      # compute correction to Bouguer Slab using LaFehr (1991) closed form
      CC = spherical_cap.calc(data[stn].elev, data[stn].lat, rho);
      # CBGA==> ObsG - model = ObsG - ThG + ATM - FAC - BSC + (TC-CC)*rho/2670
      # following is for emprical curvature correction scale dby rho/2670
      #model = ThG - ATM + FAC + BSC - (data[stn].ttc - CC)*rho/2670;
      # using LaFehr spherical cap correction, don't scale CC by density ratio
      model = ThG - ATM + FAC + BSC + CC - data[stn].ttc*rho/2670;
      data[stn].cbga[rho] = data[stn].gravity - model;

  # linear fits for CBGA vs elevation
  write("\n>>> Linear regressions for CBGA vs. elevation; pick minimum r^2 for interpretation.\n");
  write("->> %8s ==> %12s %12s %12s\n"%("Density", "Slope", "Intercept", "r^2"));
  for rho in densities:
    n=0; X=0; Y=0; XY=0; XX=0; YY=0;
    for k in data.keys():
      x=data[k].elev; y=data[k].cbga[rho]
      X += x; Y += y;
      XY += x*y;
      XX += x*x;
      YY += y*y;
      n += 1;
    if n*XX-X*X!=0:
      m = (n*XY - X*Y)/(n*XX - X*X);
      c = (Y*XX - X*XY)/(n*XX - X*X);
      # r^2 correlation coefficient
      r = ( (n*XY - X*Y)*(n*XY - X*Y) )/( (n*XX-X*X)*(n*YY-Y*Y) );
    else:
      m=0; c=0; r = 0;
    write("--> %8.0f ==> %12.6f %12.6f %12.6f\n"%(rho, m, c, r));

  # write output file to disk
  write("\n*** WRITE OUTPUT FILE ***\n\n")
  if BatchMode:
    out_file = options["out_file"]
  else:
    out_file = tkFileDialog.asksaveasfilename(parent=root, title="Output File", initialdir=".", filetypes=out_filetypes)
  if not out_file:
    write("!!! No output filename given; aborting run!\n")
    if not BatchMode:
      tkMessageBox.showerror("No File Name", "No output filename given; aborting run!")
    sys.exit(1)
  options["out_file"] = out_file

  explfileop.writeData(options["out_file"], data, options);
  write("--> CBGA values written to file '%s'\n"%options["out_file"])
  explfileop.writeCmdFile("exploration.cmd", options);
  write("--> Command file written to 'exploration.cmd'\n")

  write("# Completed at %s\n"%time.asctime(time.localtime(time.time())))

############
# END MAIN #
############

# RUN IT
try:
  main()
except SystemExit:
  sys.exit(1)
except:
  print "Unhandled exception!  Aborting program."
  print "Results not saved; output files suspect!"
  print "Fix problem and try again."
  print "Program execution traceback follows; submit with bug report."
  traceback.print_exc();
  sys.exit(0);

