/*
 * servconf.c
 *
 * Handling server configuration options.
 *
 * Based on code from Tatu Ylonen's SSH 1.2.12.

 Covered by ssh-1.2.12-COPYING

 *
 * 13 Feb 1996, hot@informatik.tu-chemnitz.de
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <syslog.h>
#include <unistd.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <errno.h>
#include <stdarg.h>

#include "servconf.h"
#include "xmalloc.h"
#include "log-server.h"

#if !defined(HAVE_STRSEP)
char *strsep(char **stringp, const char *delim);
#endif
  
/* Initializes the server options to their default values. */

void fill_default_server_options(ServerOptions *options)
{
  memset(options, 0, sizeof(*options));
  options->ssh_command = NULL;
  options->udp_port = 0;
  options->num_udp_fw = 0;
  options->listen_addr.s_addr = htonl(INADDR_LOOPBACK);
  options->with_magic = 0;
  options->packet_magic = 0x6feeddcc;
  options->max_ref_count = 1;
  options->id_string = xstrdup("RPC Proxy Client\n");
  options->quiet_mode = 0;
  options->log_to_syslog = 1;
  options->log_facility = LOG_DAEMON;
  options->insecure_mode = 2;
  options->insecure_port = 0;
  options->insecure_msg = 0;
  options->show_msg = 0;
  options->keep_pmap = 0;
  options->retry_ssh = 1;
  options->retry_limit = 0;
  options->reap_interval = 60;
  options->id = NULL;
  options->tunnel_command = NULL;
  options->host = NULL;
  options->remote_id = NULL;
  options->remote_command = NULL;
}

#define WHITESPACE " \t\r\n"

/* Keyword tokens. */
typedef enum 
{
  sSshCommand, sPort, sUdpForward, sTcpForwards, sListenAddress, sPacketMagic,
  sWithMagic, sIdString, sMaxRefCount, sQuietMode, sLogToSyslog, sLogFacility,
  sInsecure, sInsecurePort, sInsecureMsg, sShowMsg, sReapInterval, 
  sKeepPortmap, sId, sHost, sRemoteId, sTunnelCommand, sRemoteCommand, 
  sRetrySsh, sRetryLimit
} ServerOpCodes;

/* Textual representation of the tokens. */
static struct
{
  const char *name;
  ServerOpCodes opcode;
} keywords[] =
{
  { "SshCommand", sSshCommand },
  { "Port", sPort },
  { "UdpForward", sUdpForward },
  { "ListenAddress", sListenAddress },
  { "WithMagic", sWithMagic },
  { "PacketMagic", sPacketMagic },
  { "MaxRefCount", sMaxRefCount },
  { "IdString",  sIdString },
  { "QuietMode", sQuietMode },
  { "LogToSyslog",  sLogToSyslog },
  { "LogFacility", sLogFacility },
  { "Insecure", sInsecure },
  { "InsecurePort", sInsecurePort },
  { "InsecureMsg", sInsecureMsg },
  { "ShowMsg", sShowMsg },
  { "KeepPortmap", sKeepPortmap },
  { "ReapInterval", sReapInterval },
  { "Id", sId },
  { "Host", sHost },
  { "TunnelCommand", sTunnelCommand },
  { "RemoteId", sRemoteId },
  { "RemoteCommand", sRemoteCommand},
  { "RetrySsh", sRetrySsh},
  { "RetryLimit", sRetryLimit},
  { NULL, 0 }
};

static struct 
{
  const char *name;
  int facility;
} log_facilities[] =
{
  { "DAEMON", LOG_DAEMON },
  { "USER", LOG_USER },
  { "AUTH", LOG_AUTH },
  { "LOCAL0", LOG_LOCAL0 },
  { "LOCAL1", LOG_LOCAL1 },
  { "LOCAL2", LOG_LOCAL2 },
  { "LOCAL3", LOG_LOCAL3 },
  { "LOCAL4", LOG_LOCAL4 },
  { "LOCAL5", LOG_LOCAL5 },
  { "LOCAL6", LOG_LOCAL6 },
  { "LOCAL7", LOG_LOCAL7 },
  { NULL, 0 }
};

/* Returns the number of the token pointed to by cp of length len.
   Never returns if the token is not known. */

static ServerOpCodes parse_token(const char *cp, const char *filename,
                 int linenum)
{
  unsigned int i;

  for (i = 0; keywords[i].name; i++)
    if (strcmp(cp, keywords[i].name) == 0)
      return keywords[i].opcode;

  fprintf(stderr, "%s line %d: Bad configuration option: %s\n", 
      filename, linenum, cp);
  exit(1);
}

/* Read the server configuration file. */

void read_server_config(ServerOptions *options, const char *filename)
{
  FILE *f;
  char line[1024];
  char *cp, **charptr;
  int linenum,i;
  u_int *intptr;
  ServerOpCodes opcode;
#ifndef NI_MAXHOST
#define NI_MAXHOST MAXHOSTNAMELEN
#endif  
  char hostname[NI_MAXHOST];
  char word[MAXHOSTNAMELEN];
  struct hostent *localhost_info;

  f = fopen(filename, "r");
  if (!f)
  {
    fprintf(stderr,"Cannot open %s: %s\n",filename,strerror(errno));
	exit (1);
  }

  linenum = 0;
  
/* Get IP address of host as default ListenAddress */
	  
  if(gethostname(hostname,NI_MAXHOST) == 0)
  {
    if ((localhost_info = gethostbyname(hostname)) != NULL)
    {
      if (localhost_info->h_addrtype == AF_INET)
      {
        options->listen_addr = *(struct in_addr *) localhost_info->h_addr_list[0];
      }
      else
      {
        fprintf(stderr, "Only ipv4 address type accepted\n");
      }
    }
  }
  
  while (fgets(line, sizeof(line), f))
  {
    linenum++;
    cp = line + strspn(line, WHITESPACE);
    if (!*cp || *cp == '#')
    {
      continue;
    }
    cp = strtok(cp, WHITESPACE);
    opcode = parse_token(cp, filename, linenum);
    switch (opcode)
    {
      case sSshCommand:
        charptr = &options->ssh_command;
      store_string:
        cp = strtok(NULL, "\n");
        if (!cp)
        {
          fprintf(stderr, "%s line %d: argument missing.\n",
                  filename, linenum);
          exit(1);
        }
        if (*charptr)
        {
          free(*charptr);
        }
        *charptr = xstrdup(cp);
        continue;

      case sIdString:
        charptr = &options->id_string;
        goto store_string;

      case sTunnelCommand:
        charptr = &options->tunnel_command;
        goto store_string;

      case sRemoteCommand:
        charptr = &options->remote_command;
        goto store_string;

      case sUdpForward:
        i = options->num_udp_fw;
        cp = strtok(NULL, WHITESPACE);
        if (!cp)
        {
          fprintf(stderr, "%s line %d: argument missing.\n",
                  filename, linenum);
          exit(1);
        }
        while (cp && (i < MAX_FORWARD))
        {
          char *current = cp;
          char *token;
          char dummy;
          
          options->udp_forward[i].flags = 0;
          options->udp_forward[i].flags |= SEC_RPC_ACCEPTED;
          options->udp_forward[i].flags |= SEC_RPC_TO_REGISTER;
          options->udp_forward[i].num_uid = 1;
          options->udp_forward[i].uid[0] = getuid();
          options->udp_forward[i].num_gid = 1;
          options->udp_forward[i].gid[0] = getgid();
          options->udp_forward[i].old_port = 0;  
	      
          token = strsep (&current, ",");
          if (!token || !current || !strlen(token) || (sscanf(token,"%ld%c", &options->udp_forward[i].prog, &dummy) != 1))
          {
            fprintf(stderr, "%s line %d: not a program number, version: %s.\n",
                    filename, linenum, token);
            exit(1);
          }

          token = strsep (&current, ":");
          if (!token || !strlen(token) || (sscanf(token, "%ld%c", &options->udp_forward[i].vers, &dummy) != 1))
          {
            fprintf(stderr, "%s line %d: not a version number: %s.\n",
                    filename, linenum, token);
            exit(1);
          }
          if (!current)
            goto next;
	  
          token = strsep (&current, ":");
          if (!token)
          {
            fprintf(stderr, "%s line %d: remote prog ':' without following argument.\n",
                   filename, linenum);
            exit(1);
          }
          else if (!strlen(token))
            options->udp_forward[i].remote_prog = 0;
          else if ((*token) == '-')
          {
            options->udp_forward[i].flags &= ~SEC_RPC_TO_REGISTER;
          }
          else if (sscanf(token,"%ld%c", &options->udp_forward[i].remote_prog, &dummy) != 1)
          {
            fprintf(stderr, "%s line %d: not a program number or -: %s.\n",
                    filename, linenum, token);
            exit(1);
          }
          if (!current)
            goto next;

          token = strsep (&current, ":");
          if (!token)
          {
            fprintf(stderr, "%s line %d: unset pmap ':' without following argument.\n",
                   filename, linenum);
            exit(1);
          }
          else if ((!strlen(token)) || (*token) == 'k')
            options->udp_forward[i].flags &= ~SEC_RPC_UNSET_PMAP;
          else if ((*token) == 'u')
            options->udp_forward[i].flags |= SEC_RPC_UNSET_PMAP;
          else 
          {
            fprintf(stderr, "%s line %d: not  'k', 'u' or '':%s.\n",
                    filename, linenum, token);
            exit(1);
          }
          if (!current)
            goto next;

          token = strsep (&current, ":");
          if (!token)
          {
            fprintf(stderr, "%s line %d: uid list ':' without following argument.\n",
                   filename, linenum);
            exit(1);
          }
          else if ((*token) == '-')
          {
            options->udp_forward[i].flags &= ~SEC_RPC_ACCEPTED;
          }
          else if ((*token) == '*')
          {
            options->udp_forward[i].flags |= SEC_RPC_ANY_UID;
          }
          else if (strlen (token))
            options->udp_forward[i].num_uid += parse_uid_list(token, options->udp_forward[i].uid + 1, options->udp_forward[i].num_uid);

          if (!current)
            goto next;
          token = strsep (&current, ":");
          if (!token)
          {
            fprintf(stderr, "%s line %d: gid list ':' without following argument.\n",
                 filename, linenum);
            exit(1);
          }
	  else if ((*token) == '*')
          {
            options->udp_forward[i].flags |= SEC_RPC_ANY_GID;
          }
          else if (strlen (token))
            options->udp_forward[i].num_gid += parse_gid_list(token, options->udp_forward[i].gid + 1, options->udp_forward[i].num_gid);

        next:
	  i++;
          cp = strtok(NULL, WHITESPACE);
        }
        options->num_udp_fw = i;
/*	exit (0);*/
        continue;

      case sPort:
        intptr = &options->udp_port;
      parse_int:
        cp = strtok(NULL, WHITESPACE);
        if (!cp)
        {
          fprintf(stderr, "%s line %d: missing integer value.\n", 
                  filename, linenum);
          exit(1);
        }
        *intptr = atoi(cp);
        break;
  
      case sReapInterval:
        intptr = &options->reap_interval;
        goto parse_int;

      case sMaxRefCount:
        intptr = &options->max_ref_count;
        goto parse_int;

      case sRetryLimit:
        intptr = &options->retry_limit;
        goto parse_int;

      case sPacketMagic:
        cp = strtok(NULL, WHITESPACE);
        if (!cp)
        {
          fprintf(stderr, "%s line %d: missing long value.\n", 
                  filename, linenum);
          exit(1);
        }
        if (!sscanf(cp, "0x%lx", &options->packet_magic))
        {
          options->packet_magic = atol(cp);
        }
        break;

      case sListenAddress:
        cp = strtok(NULL, WHITESPACE);
	
        if (!cp) 
	  {
	    fprintf(stderr, "%s line %d: missing inet addr.\n",
		    filename, linenum);
	    exit(1);	
	  }
	
	options->listen_addr.s_addr = inet_addr(cp);
	
        break;
  
      case sLogFacility:
        cp = strtok(NULL, WHITESPACE);
        if (!cp)
        {
          fprintf(stderr, "%s line %d: missing facility name.\n",
                  filename, linenum);
          exit(1);
        }
        for (i = 0; log_facilities[i].name; i++)
        {
          if (strcmp(log_facilities[i].name, cp) == 0)
          {
            break;
          }
        }
        if (!log_facilities[i].name)
        {
          fprintf(stderr, "%s line %d: unsupported log facility %s\n",
                  filename, linenum, cp);
          exit(1);
        }
        options->log_facility = log_facilities[i].facility;
        break;

      case sLogToSyslog:
        intptr = &options->log_to_syslog;
      parse_flag:
        cp = strtok(NULL, WHITESPACE);
        if (!cp)
        {
          fprintf(stderr, "%s line %d: missing yes/no argument.\n",
                  filename, linenum);
          exit(1);
        }
        if (strcmp(cp, "yes") == 0)
          *intptr = 1;
        else
          if (strcmp(cp, "no") == 0)
            *intptr = 0;
          else
          {
            fprintf(stderr, "%s line %d: Bad yes/no argument: %s\n",
                    filename, linenum, cp);
            exit(1);
          }
        break;
        
      case sQuietMode:
        intptr = &options->quiet_mode;
        goto parse_flag;
  
      case sWithMagic:
        intptr = &options->with_magic;
        goto parse_flag;
  
      case sKeepPortmap:
        intptr = &options->keep_pmap;
        goto parse_flag;

      case sInsecurePort:
        intptr = &options->insecure_port;
        goto parse_flag;

      case sInsecureMsg:
        intptr = &options->insecure_msg;
        goto parse_flag;
		
      case sInsecure:
        intptr = &options->insecure_mode;
        goto parse_flag;
		
      case sShowMsg:
        intptr = &options->show_msg;
        goto parse_flag;

      case sRetrySsh:
        intptr = &options->retry_ssh;
        goto parse_flag;

      case sId:
        charptr = &options->id;
      parse_word:
        cp = strtok(NULL, WHITESPACE);
        if (!cp)
        {
          fprintf(stderr, "%s line %d: missing argument.\n",
                  filename, linenum);
          exit(1);
        }
        if (*charptr)
        {
          free(*charptr);
        }
        snprintf(word, MAXHOSTNAMELEN, "%s", cp);
        *charptr = xstrdup(word);
        continue;
		
      case sHost:
        charptr = &options->host;
        goto parse_word;

      case sRemoteId:
        charptr = &options->remote_id;
        goto parse_word;

	  default:
        fprintf(stderr, "%s line %d: Missing handler for opcode %s (%d)\n",
                filename, linenum, cp, opcode);
        exit(1);
    }

    if (strtok(NULL, WHITESPACE) != NULL)
    {
      fprintf(stderr, "%s line %d: garbage at end of line.\n",
              filename, linenum);
      exit(1);
    }
  }
  if (options->insecure_mode < 2)
  {
    options->insecure_port = options->insecure_mode;
    options->insecure_msg = options->insecure_mode;
  }
  fclose(f);
}

u_int parse_uid_list(char *string, uid_t *uid_table, u_int num_uid)
{
  char *current = string;
  char *token;
  u_int added_uid = 0;
  int nb_tokens;
  char dummy;

  if (num_uid > MAX_ID)
  {
    fprintf(stderr, "Too many uids specified (limit is %d)\n",MAX_ID);
    exit(1);
  }
  
  while (current && (num_uid + added_uid  < MAX_ID))
  {
    struct passwd *passwd_entry;
    uid_t new_uid;

    token = strsep (&current, ",");
    if (!token || !strlen(token))
    {
      fprintf(stderr, "need a uid after ','\n");
      exit (1);
    }
    nb_tokens = sscanf(token, "%u%c", &new_uid, &dummy);
    if (nb_tokens == 1)
      goto next;
    if (nb_tokens != 0)
    {
      fprintf(stderr, "not a numeric uid: %s.\n", token);
      exit(1);
    }
    if ((passwd_entry = getpwnam(token)))
    {
      new_uid = passwd_entry->pw_uid;
    }
    else
    {
      fprintf(stderr, "cannot find uid for %s.\n", token);
      exit(1);
    }
  next:
    (*(uid_table + num_uid + added_uid - 1)) = new_uid;
    added_uid++;
  }
  if (current)
    fprintf(stderr, "too many uids: %s\n", current);
  return added_uid;
}

u_int parse_gid_list(char *string, gid_t *gid_table, u_int num_gid)
{
  char *current = string;
  char *token;
  u_int added_gid = 0;
  int nb_tokens;
  char dummy;
  if (num_gid > MAX_ID)
  {
    fprintf(stderr, "Too many uids specified (limit is %d)\n",MAX_ID);
    exit(1);
  }
  while (current && (num_gid + added_gid  < MAX_ID - 1))
  {
    struct group *group_entry;
    gid_t new_gid;
    token = strsep (&current, ",");
    if (!token || !strlen(token))
    {
      fprintf(stderr, "need a gid after ','\n");
      exit (1);
    }
    nb_tokens = sscanf(token, "%u%c", &new_gid, &dummy);
    if (nb_tokens == 1)
      goto next;
    if (nb_tokens != 0)
    {
      fprintf(stderr, "not a numeric gid: %s.\n", token);
      exit(1);
    }
    if ((group_entry = getgrnam(token)))
    {
      new_gid = group_entry->gr_gid;
    }
    else
    {
      fprintf(stderr, "cannot find gid for %s.\n", token);
      exit(1);
    }
  next:
    (*(gid_table + num_gid + added_gid - 1)) = new_gid;
    added_gid++;
  }
 if (current)
   fprintf(stderr, "too many gids: %s\n", current);
  return added_gid;
}


void parse_command(char * command, char **arg)
{
  char *cp,    /* current pointer in arg string */
       *cpw=NULL;   /* current write position when removing escapes */

  int arg_idx, /* index of the current arg */
      in_arg,  /* if true, then we scan an arg */
      esc;     /* if true, the current char was escaped */
  char c;       /* current char in the arg string */

  cp = command;
  esc = in_arg = arg_idx = 0;

  while ((c = *cp))
  {
    if (esc)            /* escaped ? */
    {
      *cpw++ = c;       /* yes, store it */
      cp++;
      esc = 0;          /* switch back to normal */
      if (!in_arg)
      {
        in_arg = 1;
        if (arg_idx < MAX_ARGS - 1)
        {
          arg[arg_idx++] = cpw - 1; /* add another arg */
        }
        else
        {
          fatal("command line %s has too many args", command);
        }
      }
    }
    else                /* unescaped char */
    {
      switch (c)
      {
        case '\\':
          esc = 1;      /* escape */
          if (cpw == NULL)
          {             /* first escape for the current arg */
            cpw = cp++;
            continue;
          }
          break;

        case '\t':
        case ' ':
          if (in_arg)   /* it's the end of the current arg */
          {
            in_arg = 0; /* no longer in an arg */
            if (cpw == NULL)
            {
              *cp++ = 0;  /* mark the end of the arg here */
            }
            else
            {
              *cpw = 0;  /* mark the end of the arg here */
              cp++;
              cpw = NULL;
            }
            continue;
          }
          break;        /* skip whitespace around args */

        default:        /* char without special meaning */
          if (!in_arg)
          {
            in_arg = 1;
            if (arg_idx < MAX_ARGS - 1)
            {
              arg[arg_idx++] = cp; /* add another arg */
            }
            else
            {
              fatal("command line %s has too many args", command);
            }
          }
          else
          {
            if (cpw != NULL)
            {
              *cpw++ = c;
            }
          }
       }
       cp++;
     }
  }
  if (in_arg)   /* it's the end of the string, and the end of an arg */
  {
    if (cpw == NULL)
    {
      *cp = 0;  /* mark the end of the arg here */
    }
    else
    {
      *cpw = 0;  /* mark the end of the arg here */
    }
  }

  arg[arg_idx++] = NULL; /* end of args list */
  
}

int duplicate_command(char** to_command, char** from_command)
{
  char ** from_cmd = from_command,
       ** to_cmd = to_command;
  int idx = 0;
  
  if (!to_command || !from_command)
    fatal("bug: in duplicate_command NULL pointer");
  
  while ((*from_cmd) && (idx < MAX_ARGS - 1))
  {    
    *to_cmd = *from_cmd;
    to_cmd++;
    from_cmd++;
    idx++;
  }
  if (*from_cmd)
      fatal("command too long, ending with: %s", *from_cmd);
  *to_cmd = NULL;
  return idx;
}

int size_command(char **command)
{
  char** cmd = command;
  int size = 0;
  if (!command)
    fatal("bug:size_command NULL command");
  
  while (*cmd)
  {
    size++;
    if (size > MAX_ARGS - 1)
      fatal ("size of %p too long: %d", command, size);
    cmd++;
  }
  return size;
}

int join_commands(char **result_command, char** added_command)
{
  char** result_cmd;
  char** added_cmd = added_command; 
  int size;
 
  if (!added_command)
    fatal ("bug: join_commands NULL added_command pointer");
  size = size_command(result_command);
  if (size)
  {
    result_cmd = result_command + size - 1;
	if (!*result_cmd)
      fatal("bug: join_commands NULL at the end -1 of the joined command\n");
  }
  result_cmd = result_command + size;
  if (*result_cmd)
    fatal("bug: join_commands not NULL at the end of the joined command\n");
  
  while ((*added_cmd) && (size < MAX_ARGS - 1))
  {
    *result_cmd = *added_cmd;
    result_cmd++;
    added_cmd++;
    size++;
  }
  if (*added_cmd)
    fatal("command too long, ending with: %s", *added_cmd);
  *result_cmd = NULL;
  return size;
}

void concatenate_commands(char ** result_command, int number, ...)
{
  va_list commands;
  char **command_added = NULL;
  if (!result_command || !number)
  {
    fatal("bug: in concatenate_commands result_command NULL or number=0");
  }
  va_start(commands, number);
  while (number--)
  {
    command_added = va_arg(commands, char **);
    join_commands(result_command, command_added);
  }
  va_end(commands);
}

void debug_command(char **command)
{
  char **cmd = command;
  int idx = 0;

  if (command == NULL)
    fatal("bug: in debug_command command is NULL");
  
  debug("Command line");
  while (*cmd)
  {
    debug("arg %3d: %s", idx, *cmd);
    cmd++;
    idx++;
  }
}

void dump_forward(forward *msg_forward, u_int num_fw)
{
  int i;
  if (num_fw == 0)
  {
    debug ("Nothing registered");
    return;
  }
  debug ("rpc forward configuration:");
  for (i = 0; i < num_fw; i++)
  {
    forward *program = &msg_forward[i];
    debug("%d: prog:%lu, vers:%lu, remote_prog:%lu, registered:%d, accepted:%d, pmap unset: %d,  all uid:%d, all gid:%d", 
        i, program->prog,  program->vers, program->remote_prog,
        (program->flags & SEC_RPC_TO_REGISTER), (program->flags & SEC_RPC_ACCEPTED),
        (program->flags & SEC_RPC_UNSET_PMAP), (program->flags & SEC_RPC_ANY_UID),  
        (program->flags & SEC_RPC_ANY_GID));
    if (program->flags & SEC_RPC_ACCEPTED)
    {
      if (! (program->flags & SEC_RPC_ANY_UID)) 
        debug_uid_list(program->uid, program->num_uid);
      if (! (program->flags & SEC_RPC_ANY_GID))
        debug_gid_list(program->gid, program->num_gid);
    }
  }
}

void debug_uid_list(uid_t *uid_list, u_int num_uid)
{
  int i;
  for (i = 0; i < num_uid; i++)
    debug("uid %d: %d", i, uid_list[i]);
}

void debug_gid_list(gid_t *gid_list, u_int num_gid)
{
  int i;        
  for (i = 0; i < num_gid; i++)
    debug("gid %d: %d", i, gid_list[i]);
}

