/*
 * rpc_to_str.c
 *
 * Analyse an RPC message, convert several parts to an ASCII string.
 *
 * 18 Jan 1996, hot@informatik.tu-chemnitz.de
 */

#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    <unistd.h>
#include    <sys/time.h>
#include    <errno.h>
#include    <rpc/rpc.h>
#include    <ctype.h>

#include    "log-server.h"
#include    "map_xid.h"
#include    "servconf.h"
#include    "rpc_to_str.h"

/*
 * some global data
 */
#define DUMP_SIZE 1000         /* maximum length of packet dumps for debug */

static char result[DUMP_SIZE]; /* the result */

/*
 * convert the message body to ASCII string
 */

void
rpc_message_to_str(char *resulting_message, u_char *message, int request_len, int show_message)
{
  int i, len, remain;
  char *cp;

  cp = resulting_message + (len = strlen(resulting_message)); 
  remain = DUMP_SIZE - 1 - len;

  if (request_len < remain)
  {
    remain = request_len;
  }

  if (show_message)
  {
    for (i = 0; i < remain; i++)
    {
      *cp++ = isprint(message[i]) ? message[i] : '.';
    }
  }
  else
  {
    for (i = 0; i < remain; i++)
    {
      *cp++ = '.';
    }
  }
  *cp = 0;
}

/* 
 * convert a request to ASCII string
 */
char *
rpc_request_to_str(u_char *request, int req_len, int show_message)
{
  struct rpc_msg_32 *req = (struct rpc_msg_32 *) request;
  struct call_body_32 *cb = &req->ru.RM_cmb;

  sprintf(result, "xid=0x%08lx, prog=%ld, vers=%ld, proc=%ld: ",
          (long) ntohl(req->rm_xid), (long) ntohl(cb->cb_prog), 
		  (long) ntohl(cb->cb_vers), (long) ntohl(cb->cb_proc));

  rpc_message_to_str(result, request, req_len, show_message);

  return result;
}

int
check_rpc_request(u_char *request, int req_len, forward *forwarded_progs, u_int num_forwarded_progs, char *result_string, u_long *xid, int show_message)
{
  int i, j, gid_accepted, uid_accepted;
  XDR xdr;
  struct rpc_msg req;
  struct authunix_parms authunix;
  int prog_index = -1;
  forward *forwarded_prog = NULL;
  memset((void *) &req, 0, sizeof(struct rpc_msg));
  memset((void *) &authunix, 0, sizeof(struct authunix_parms));
 
  if (result_string)
    strcpy(result_string, "request not decoded, program not forwarded or no auth_unix");
  xdrmem_create(&xdr, (char *) request, req_len, XDR_DECODE);
  if (!xdr_callmsg(&xdr, &req))
  {
    debug ("cannot decode packet");
    return SEC_RPC_NO_REPLY;
  }
  if (req.rm_call.cb_rpcvers != RPC_MSG_VERSION)
    return SEC_RPC_NO_REPLY;
  (*xid) = req.rm_xid;
  for (i = 0; i < num_forwarded_progs; i++)
  {
    if ((forwarded_progs[i].prog == req.rm_call.cb_prog) &&
            (forwarded_progs[i].vers == req.rm_call.cb_vers))
    {
      prog_index = i;
      forwarded_prog = &forwarded_progs[i]; 
      break;
    }
  }
  if ((prog_index < 0) || (!(forwarded_prog->flags & SEC_RPC_ACCEPTED)))
  {
    debug ("%ld, %ld not forwarded(%d)", req.rm_call.cb_prog, req.rm_call.cb_vers, prog_index);
    return SEC_RPC_REPLY_ERROR;
  }

  if ((forwarded_prog->flags & SEC_RPC_ANY_UID) && (forwarded_prog->flags & SEC_RPC_ANY_GID))
  {
    if (result_string)
    {
      if (req.rm_call.cb_cred.oa_flavor != AUTH_UNIX)
      {
        sprintf(result_string,
          "xid=0x%08lx, prog=%ld, vers=%ld, proc=%ld: ",
          req.rm_xid, req.rm_call.cb_prog, req.rm_call.cb_vers, 
          req.rm_call.cb_proc);
      }
      else
      {
         xdrmem_create(&xdr, req.rm_call.cb_cred.oa_base, req.rm_call.cb_cred.oa_length, XDR_DECODE);
        if (!xdr_authunix_parms(&xdr, &authunix))
        {
           debug ("Couldn't decode auth");
        }
        else
        {
           sprintf(result_string, 
              "xid=0x%08lx, prog=%ld, vers=%ld, proc=%ld, computer=%s, uid=%d, gid=%d: ",
              req.rm_xid, req.rm_call.cb_prog, req.rm_call.cb_vers, req.rm_call.cb_proc,
              authunix.aup_machname, authunix.aup_uid, authunix.aup_gid);
        }
      }
    }
  }
  else
  {
    if (req.rm_call.cb_cred.oa_flavor != AUTH_UNIX)
    {
      return SEC_RPC_REPLY_ERROR;
    }
     xdrmem_create(&xdr, req.rm_call.cb_cred.oa_base, req.rm_call.cb_cred.oa_length, XDR_DECODE);
    if (!xdr_authunix_parms(&xdr, &authunix))
    {
      debug ("Couldn't decode auth");
	  return SEC_RPC_REPLY_ERROR;
    }
    if (result_string)
      sprintf(result_string, 
        "xid=0x%08lx, prog=%ld, vers=%ld, proc=%ld, computer=%s, uid=%d, gid=%d: ",
        req.rm_xid, req.rm_call.cb_prog, req.rm_call.cb_vers, req.rm_call.cb_proc,
        authunix.aup_machname, authunix.aup_uid, authunix.aup_gid);
  }
  if (result_string)
  {
    rpc_message_to_str(result_string, request, req_len, show_message);
  }
  
  uid_accepted = 0;
  if (!(forwarded_prog->flags & SEC_RPC_ANY_UID))
  {
    for (i = 0; i < forwarded_prog->num_uid; i++)
    {
      if (authunix.aup_uid == forwarded_prog->uid[i])
      { 
        uid_accepted = 1;
        break;
      }
    }
    if (!uid_accepted)
    {
      debug ("uid %d not accepted", authunix.aup_uid);
      return SEC_RPC_REPLY_ERROR;
    }
  }
  gid_accepted = 0;
  if (!(forwarded_prog->flags & SEC_RPC_ANY_GID))  
  {
    for (i = 0; i < forwarded_prog->num_gid; i++)
    {
      if (authunix.aup_gid == forwarded_prog->gid[i])
      {
        gid_accepted = 1;
        break;
      }
    }
    if (!gid_accepted)
    {
      for (i = 0; i < forwarded_prog->num_gid; i++)
      {
        for (j = 0; j < authunix.aup_len; j++)
          if (authunix.aup_gids[j] == forwarded_prog->gid[i])
          {
            gid_accepted = 1;
            break;
          }
        if (gid_accepted)
          break;
      }
      if (!gid_accepted)
      {
        debug("gid %d not accepted", authunix.aup_gid);
        return SEC_RPC_REPLY_ERROR;
      }
    }
  }

  return prog_index;
}
/* 
 * convert a reply to ASCII string
 */
char *
rpc_reply_to_str(u_char *reply, int rep_len, int show_message)
{
  struct rpc_msg_32 *rep = (struct rpc_msg_32 *) reply;
  struct reply_body *rb = &rep->ru.RM_rmb;

  sprintf(result, "xid=0x%08lx, stat=%ld: ", 
          (long) ntohl(rep->rm_xid), (long) ntohl(rb->rp_stat));

  rpc_message_to_str(result, reply, rep_len, show_message);

  return result;
}    


