/*
 * rpc_psrv.c
 * Copyright (C) 2001 Holger Trapp and John C. Bowman 
 *
 * RPC proxy server
 *
 * This daemon pretends to be an RPC server. Actually it only forwards 
 * the requests and replies between clients and the real server.
 *
 * UDP datagrams with Sun RPC messages from RPC clients sent to this 
 * daemon are forwarded to the actual RPC server via a secure SSH connection.
 * Answers sent back via the SSH connection are delivered to the original
 * client.
 *
 * To multiplex many clients the XID of each request is mapped to a new,
 * globally unique one before forwarding the message via SSH and mapped
 * back before sending the answer to the client.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Original author: 20 Feb 1996, Holger Trapp (hot@informatik.tu-chemnitz.de)
 * Adapted for NFS: 09 May 2001, John Bowman <bowman@math.ualberta.ca>
 *
 * 06 Dec 2001, rune.saetre@netcom-gsm.no
 *  - Modified to support HPUX
 */

#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    <strings.h>
#include    <unistd.h>
#include    <signal.h>
#include    <fcntl.h>
#include    <sys/time.h>
#include    <sys/socket.h>
#include    <netinet/in.h>
#include    <arpa/inet.h>
#include    <errno.h>
#include    <rpc/rpc.h>
#include    <rpc/pmap_clnt.h>
#include    <rpc/pmap_prot.h>
#include    <sys/wait.h>
#include    <pwd.h>
#if defined sun || defined hppa
#include    <rpc/clnt_soc.h>
#endif

#include    "log-server.h"
#include    "map_xid.h"
#include    "buffer.h"
#include    "xmalloc.h"
#include    "servconf.h"
#include    "rpc_to_str.h"

/*
 * some global data
 */
static ServerOptions options; /* the options */
static int s_port;           /* UDP server port */
static int debug_flag = 0;    /* if true, print debug info */
static int quiet = 0;         /* if true, don't print process id */
static int root = 0;         /* if true, run as root */
static int force = 0;        /* if true force unsetting portmap */
static int show_msg = 0;     /* if true, show rpc msg contents in debug mode */
static char *prog_name;       /* program name for syslog */
static int ssh_child_status = 0; /* exit status of the ssh child */
static int ssh_child_signaled = 0; /* true if ssh child caught a signal */
static int ssh_pid;           /* pid of the ssh client (child) */
static int clean_flag = 1;        /* if true do cleaning */
static int pid = 0;             /* pid of the child if operating in the background */
static char * default_tunnel_command[] = { "ssh", "ssh", "-c", "blowfish", "-x",
     "-oFallBackToRsh no", NULL };
static char * default_remote_command[] = { "rpc_pcl", NULL };
static char request_dump[DUMP_SIZE]; /* a request debug info */

#define MIN_RPC_REQ_LEN 40    /* minimum length of a UDP datagram to be
                                 accepted as RPC request */
#define MIN_RPC_REPL_LEN 20   /* minimum length of a UDP datagram to be
                                 accepted as RPC reply */
#define SEC_RPC_MIN_WAIT 10   /* minimum waiting time before retrying ssh */
#define SEC_RPC_MAX_WAIT 60  /* maximum waiting time before retrying ssh */
/*
 * error codes
 */

#define SEC_RPC_EXIT_SSH_ERROR 255
#define SEC_RPC_EXIT_NO_CLEAN 254
#define SEC_RPC_EXIT_CHILD_SIGNALED 253
#define SEC_RPC_EXIT_CHILD_EXITED 252

/*
 * show the usage
 */
void usage(char *prog)
{
  fprintf(stderr, "Usage: %s [options] config_file\n", prog);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "   -b       run in background\n");
  fprintf(stderr, "   -d       run in debug mode\n");
  fprintf(stderr, "   -r       run as root (overrides configuration file)\n");
  fprintf(stderr, "   -q       quiet mode (don't echo process id)\n");
  fprintf(stderr, "   -s       show messages contents in debug mode\n");
  fprintf(stderr, "   -e file  redirect stderr to file (if in background)\n");
  exit(1);
}

/*
 * signal handler to reap stale mappings 
 */
void
reap_handler(int sig)
{
  grim_reap(options.reap_interval);
  signal(SIGALRM, reap_handler);
  alarm(options.reap_interval);
}

void
portmap_register_handler(int sig)
{
  struct sockaddr_in pmap_addr;
  int i, registered_port;
  u_long prog, vers;
  
  bzero((char *) &pmap_addr, sizeof(pmap_addr));
  pmap_addr.sin_family      = AF_INET;
  pmap_addr.sin_addr.s_addr = options.listen_addr.s_addr;
  pmap_addr.sin_port        = htons(PMAPPORT);
  for (i = 0; i < options.num_udp_fw; i++)
  {
    prog = options.udp_forward[i].prog,
    vers = options.udp_forward[i].vers;

    if (! (options.udp_forward[i].flags & SEC_RPC_TO_REGISTER))
      continue;

    registered_port = pmap_getport (&pmap_addr, prog, vers, IPPROTO_UDP);
    if (registered_port)
    {
      if (registered_port != s_port)
      {
        debug ("in portmap_register_handler unsetting %u registered for %u,%u",
          registered_port, prog, vers);
        pmap_unset(prog, vers);
      }
      else 
        continue;
    }
    if (!pmap_set(prog, vers, IPPROTO_UDP, s_port))
      debug ("in portmap_register_handler couldn't register port %u for %u,%u",
          s_port, prog, vers);
  }
      
  signal(SIGUSR1, portmap_register_handler);
}

/*
 * forward RPC requests and replies (UDP only)
 */
int 
handle_forwarding(int sockfd, int ssh_write_side, int ssh_read_side)
{
  int nread,          /* number of bytes read */
      nwritten,       /* number of bytes written */
      nready,         /* number of ready descriptors after select */
      max_rfd,        /* maximum read fd for select */
      check_res = 0;  /* return status of check_rpc_request */

  fd_set readfds;     /* set of descriptors for select */ 

  u_long 
      packet_length,  /* length of the packet sent via SSH */ 
      packet_magic = htonl(options.packet_magic),
                      /* the magic to detect when being out of sync */
      xid;            /* xid of the error reply */

  struct sockaddr_in 
      cli_addr,       /* the client's address */
      cli_error_addr; /* the client address for the error reply */

  XDR xdr;

  u_int cli_len;        /* length of the client's address */

  struct timeval tv;  /* for timeout with select */

  /*
   * buffers
   */
  Buffer requ_buffer; /* buffer for the requests received from the client */

  u_char 
      request[UDPMSGSIZE],
                      /* a single request datagram */
      reply[UDPMSGSIZE],
                      /* a single reply datagram */
      reply_error[UDPMSGSIZE],
                      /* a reply datagram in case of auth error */
      header[8];      /* header (magic and length) */

  int exp_replen,     /* expected length for a reply */
      rep_len,        /* current reply length */
      head_len,       /* header length for the reply */
      buflen;         /* buffer length */
  
  struct rpc_msg msg_error;
  msg_error.rm_xid = 0;
  msg_error.rm_direction = REPLY;
  msg_error.ru.RM_rmb.rp_stat = MSG_DENIED;
  msg_error.ru.RM_rmb.ru.RP_dr.rj_stat = AUTH_ERROR;
  msg_error.ru.RM_rmb.ru.RP_dr.ru.RJ_why = AUTH_TOOWEAK;

  /*
   * read and write the pipe in non-blocking mode
   */
  if (fcntl(ssh_write_side, F_SETFL, O_NONBLOCK) < 0)
      error("fcntl O_NONBLOCK: %.100s", strerror(errno));
  if (fcntl(ssh_read_side, F_SETFL, O_NONBLOCK) < 0)
      error("fcntl O_NONBLOCK: %.100s", strerror(errno));

  /*
   * initialize the buffers
   */
  buffer_init(&requ_buffer); 
  exp_replen = rep_len = 0;
  head_len = options.with_magic ? 0 : 4;

  alarm(options.reap_interval);
  signal(SIGALRM, reap_handler);

  /*
   * determine the maximum number of elements in fd sets for select
   */
  max_rfd = sockfd > ssh_read_side ? sockfd+1: ssh_read_side+1;

  /*
   * central server loop
   */
  debug("starting central server loop");
  for (;;) 
  {
    /*
     * prepare the fd sets for select
     */
    FD_ZERO(&readfds);

    FD_SET(sockfd, &readfds);
    FD_SET(ssh_read_side, &readfds);

    /*
     * step zero: if there was an authentication error, send the response
     */
    
    if (check_res == SEC_RPC_REPLY_ERROR)
    {
      /* get the xid and encode the error response*/
      msg_error.rm_xid = xid;
      xdrmem_create(&xdr, (char *) reply_error, UDPMSGSIZE, XDR_ENCODE);
      if (xdr_replymsg(&xdr, &msg_error))
      {
        int reply_error_len = XDR_GETPOS (&xdr);
        if (debug_flag)
        {
          debug("sendto addr/port %s/%d: %s (%d)", inet_ntoa(cli_error_addr.sin_addr),
           ntohs(cli_error_addr.sin_port), 
           rpc_reply_to_str(reply_error, reply_error_len, show_msg), reply_error_len);
        }
        /* send the error msg */
        if (sendto(sockfd, reply_error, reply_error_len, 0, 
               (struct sockaddr *) &cli_error_addr, 
          sizeof(cli_error_addr)) != reply_error_len)
        {
          error("couldn't send error reply to client: %s", strerror(errno));
        }
        else
        {
          debug("reply sent to client, %d bytes", reply_error_len);
        }
      }
      else
      {
        error("couldn't encode error reply");
      }
      check_res = 0;
    }
    
    /*
     * step one: if there is data to be written, write as much as possible
     */
 
    /* reply for any client complete ? */
    if (exp_replen && exp_replen == rep_len)
    {
      /* yes, but check the length first */
      debug("reply for client complete, %d bytes", rep_len);
      if (exp_replen < MIN_RPC_REPL_LEN)
      {
        debug("server reply too small: %d bytes, dropped", exp_replen);
      }
      else
      {
        /* length OK, then send it to the client */
        if (demap_XID(reply, &cli_addr)) 
        {
          if (debug_flag)
          {
            debug("sendto addr/port %s/%d: %s", inet_ntoa(cli_addr.sin_addr),
                  ntohs(cli_addr.sin_port), rpc_reply_to_str(reply, rep_len, show_msg));
          }
    
          if (sendto(sockfd, reply, rep_len, 0, 
		      (struct sockaddr *) &cli_addr, sizeof(cli_addr)) != rep_len)
          {
            error("couldn't send reply to client: %s", strerror(errno));
          }
          else
          {
            debug("reply sent to client, %d bytes", rep_len);
            /* clear the buffer */
            exp_replen = rep_len = 0;
            head_len = options.with_magic ? 0 : 4;
          }
        }
        else 
        {
          debug("couldn't demap XID");
          /* clear the buffer */
          exp_replen = rep_len = 0;
          head_len = options.with_magic ? 0 : 4;
        }
      }
    }

    /* data for server available ? */
    if ((buflen = buffer_len(&requ_buffer)) > 0)
    {
      /* yes, send as much as possible */
      debug("data for server available");
      nwritten = write(ssh_write_side, buffer_ptr(&requ_buffer), buflen);
      debug("%d bytes written to SSH", nwritten);

      if (nwritten == -1 && errno != EAGAIN)
      {
        error("write to pipe failed: %s", strerror(errno));
      }
      else
      {
        if (nwritten > 0)
        {
          buffer_consume(&requ_buffer, nwritten);
        }
      }    
    }

    /* 
     * step two: test whether we can read data
     * 
     * if there is no data left to be written then
     * block in select
     */

    if ((exp_replen && exp_replen == rep_len) || buffer_len(&requ_buffer)
          || (check_res == SEC_RPC_REPLY_ERROR))
    {
      /* we should write as soon as possible, therefore don't
       * block in select
       */
      tv.tv_sec = tv.tv_usec = 0;
      debug("select without blocking (%d descriptors)", max_rfd);
      nready = select(max_rfd, &readfds, NULL, NULL, &tv);
    }
    else
    {
      /* wait until data is available to be read, because there is 
       * no other job to be done
       */
      debug("select with blocking (%d descriptors)", max_rfd);
      nready = select(max_rfd, &readfds, NULL, NULL, NULL);
    }

    if (nready == -1 && errno != EINTR)
    {
      /* there was an error */
      error("select failed: %s", strerror(errno));
    }

    if (nready > 0)
    {
      /* yes, we can read 
       * find out from where
       */
      if (FD_ISSET(sockfd, &readfds)) 
      {
        /* read a datagram from the net */
        debug("can read datagram");
        
        cli_len = sizeof(cli_addr);
        nread = recvfrom(sockfd, (char *) request, sizeof(request), 0, 
		                 (struct sockaddr *) &cli_addr, &cli_len);

        if (nread <= 0) 
        {
          error("error when reading a datagram: %s", strerror(errno));
        }
        else 
        {
          char *request_debug = NULL;
          if (debug_flag)
          {
          /* prepare debug message, which differs depending on security set or
           * not */
            request_debug = request_dump;
            if (options.insecure_msg)
              debug("recvfrom got %d bytes from %s/%d: %s", nread,
                inet_ntoa(cli_addr.sin_addr), ntohs(cli_addr.sin_port),
                rpc_request_to_str(request, nread, show_msg));
            else 
              debug("recvfrom got %d bytes from %s/%d", nread,
                  inet_ntoa(cli_addr.sin_addr), ntohs(cli_addr.sin_port));
          }
          /* only accept requests coming from a reserved port */
          if (!options.insecure_port && ntohs(cli_addr.sin_port) > IPPORT_RESERVED)
          {
            debug("request coming from unreserved port, dropped");
          }
          /* throw away small messages */
          else if (nread < MIN_RPC_REQ_LEN)
          {
            debug("client request too small: %d bytes, dropped", nread);
          }
          /* check rpc credentials */
          else if (!options.insecure_msg && ((check_res = check_rpc_request(request, nread, options.udp_forward, options.num_udp_fw, request_debug, &xid, show_msg)) < 0))
          {
            debug("authentication failed: %s", request_debug);
            /* get the client adress in case we have to send back an error
             * reply */
            memcpy((void *) &cli_error_addr, (void *) &cli_addr, sizeof(cli_addr));
          }
          else
          {
            /* map the XID */
            if (debug_flag && !options.insecure_msg)
              debug ("request: %s", request_debug);
            if (map_XID(request, &cli_addr))
            {
              /* form a packet and append it to the buffer */
              if (options.with_magic)
              {
                buffer_append(&requ_buffer, (u_char *) &packet_magic, 4);
              }
              packet_length = htonl(nread);
              buffer_append(&requ_buffer, (u_char *) &packet_length, 4);
              buffer_append(&requ_buffer, request, nread);
              debug("packet appended to request buffer");
              if (!options.insecure_msg)
              /* a packet was successfully sent for the program 
               * mark it as unset, such that the program remains registered
               * with the same port, and next rpc_psrv reuse this port */
              {
                options.udp_forward[check_res].flags &= ~SEC_RPC_SET;
              }
            }
            else
            {
              /* drop the packet silently */
              debug("map_XID failed, packet dropped");
            }
          }
        }
      }

      if (FD_ISSET(ssh_read_side, &readfds)) 
      {
        debug("can read from SSH");
        /* receive via SSH */
        if (head_len < 4)
        {
          /* try to get the magic */
          debug("try to get the magic (%d bytes)", 4 - head_len);
          nread = read(ssh_read_side, header + head_len, 4 - head_len);

          if (!nread) /* EOF */
          {
            error("remote partner closed the SSH connection");
            break;
          }

          if (nread < 0) /* error */
          {
            error("reading magic from pipe failed: %s", strerror(errno));
          }
          else /* complete the magic */
          {
            head_len += nread;
            if (head_len == 4 && *((u_long *) header) != packet_magic)
            {
              error("out of sync (bad magic: 0x%08lx)", *((u_long *) header));
              break;
            }
          }
          
        }
        else
        {
          if (head_len < 8)
          {
            /* try to get the length of the reply */
            debug("try to get the length (%d bytes)", 8 - head_len);
            nread = read(ssh_read_side, header + head_len, 8 - head_len);

            if (!nread) /* EOF */
            {
              error("remote partner closed the SSH connection");
              break;
            }

            if (nread < 0) /* error */
            {
              error("reading length from pipe failed: %s", strerror(errno));
            }
            else /* complete the header */
            {
              head_len += nread;
              if (head_len == 8)
              {
                debug("header completely received");
                exp_replen = ntohl(*((u_long *) (header + 4)));
                /* check the length */
                if (exp_replen > sizeof(reply))
                {
                  fatal("got giant packet: %d bytes", exp_replen);
                }
                debug("expecting reply with %d bytes", exp_replen);
              }
            }
          }
          else
          {
            /* normal data, append it to the reply */
            debug("getting normal data (%d bytes)", exp_replen - rep_len);
            nread = read(ssh_read_side, reply + rep_len, exp_replen - rep_len);

            if (!nread) /* EOF */
            {
              error("remote partner closed the SSH connection");
              break;
            }

            if (nread < 0) /* error */
            {
              error("reading data from pipe failed: %s", strerror(errno));
            }
            else /* complete the reply */
            {
              debug("got %d bytes normal data", nread);
              rep_len += nread;
            }
          }
        }
      }
    }
    /* stop forwarding if ssh child is dead. */
    if (ssh_pid == 0)
    {
      debug ("ssh child is dead, stopping forwarding.");
      break;
    }
  } /* for(;;) */
  return 0;
}

/*
 * start ssh to establish an SSH connection
 */
int 
start_ssh(int *read_side, int *write_side, char** args, int sockfd)
{
    int pin[2],  /* pipe, stdin for SSH */
        pout[2]; /* pipe, stdout for SSH */

    /* create a pipe for communicating with ssh */
    if (pipe(pin) < 0) 
    {
      error("pipe: failed: %s", strerror(errno));
      return 0;
    }
    if (pipe(pout) < 0) 
    {
      error("pipe: failed: %s", strerror(errno));
      return 0;
    }

    /* fork a child to execute ssh */
    if ((ssh_pid = fork()) < 0)
    {
      error("can't fork SSH: %s", strerror(errno));
      return 0;
    }
    
    /* set clean_flag to 0 such that parent don't do any cleaning in case
     * it catch a signal before they are reset */
    clean_flag = 0;
    
    if (!ssh_pid)
    {
      /* child */

      /* the parent should do the cleanup if the child exited
       * therefore we don't handle any signal */
      signal (SIGTERM,SIG_DFL);
      signal (SIGHUP,SIG_DFL);
      signal (SIGINT,SIG_DFL);
      signal (SIGQUIT,SIG_DFL);
      
      /* FIXME: shouldn't it be fatal ? */
      if (close(pin[1]) < 0)
        error("close pin[1] failed: %s", strerror(errno));
      if (close(pout[0]) < 0)
        error("close pout[0] failed: %s", strerror(errno));
      if (dup2(pin[0], 0) < 0)
        error("dup2(pin[0], 0) failed: %s", strerror(errno));
      if (dup2(pout[1], 1) < 0)
        error("dup2(pout[1], 1) failed: %s", strerror(errno));
      if (close(pin[0]) < 0)
        error("close pin[0] failed: %s", strerror(errno));
      if (close(pout[1]) < 0)
        error("close pout[1] failed: %s", strerror(errno));
      if (close(sockfd) <0)
        error("close udp socket failed: %s", strerror(errno));
      debug ("close/dup2 done");

      if (chdir("/") < 0)
        error("ssh process cannot chdir to /: %s",  strerror(errno));

      if (debug_flag)
      {
        debug_command(args);
      }
      debug("pid of the ssh process: %d",getpid());
      execvp(args[0], args+1);
      fatal("execvp failed: %s", strerror(errno));
    }

    /* parent, closes the other side, and returns the local side */

    clean_flag = 1;
    
    if(close(pin[0]) < 0)
      error("parent of ssh close(pin[0]) failed :%s", strerror(errno));
    *write_side = pin[1];
    if(close(pout[1]) < 0)
      error("parent of ssh close(pout[1]) failed :%s", strerror(errno));
    *read_side = pout[0];
    return 1;
}

/*
 * Translate exit status for error report
 */

char *
translate_exit_status(int exit_status)
{
  char *error_string ;
  char temp_string [1024];
  if (exit_status == 1)
  {
    error_string = xstrdup ("ambiguous exit code from child (1). Check the logs ;-)");
  }
  else if (exit_status == SEC_RPC_EXIT_NO_CLEAN)
  {
    snprintf(temp_string, 1024, "child exited without cleaning (or ssh exited with status %d)", SEC_RPC_EXIT_NO_CLEAN);
    error_string = xstrdup (temp_string);
  }
  else if (exit_status == SEC_RPC_EXIT_CHILD_SIGNALED)
  {
    snprintf(temp_string, 1024, "ssh child caught a signal (or ssh exited with status %d)", SEC_RPC_EXIT_CHILD_SIGNALED);
    error_string = xstrdup (temp_string);
  }
  else if (exit_status == SEC_RPC_EXIT_SSH_ERROR)
  {
    snprintf(temp_string, 1024, "ssh exited with status %d, indicating an ssh error", SEC_RPC_EXIT_SSH_ERROR);
    error_string = xstrdup (temp_string);
  }
  else if (exit_status == SEC_RPC_EXIT_CHILD_EXITED)
  {
    snprintf(temp_string, 1024, "psrv daemon caught a fatal error, and ssh child exited with bad error status (or ssh exited with status %d)", SEC_RPC_EXIT_CHILD_EXITED);
    error_string = xstrdup (temp_string);
  }
  else
  {
    snprintf(temp_string, 1024, "ssh exited with status %d", exit_status);
    error_string = xstrdup (temp_string);
  }
  return error_string;
}

/*
 * signal handlers
 */

/* to avoid zombies, and exit. Set in the parent if there is a -b flag */
void
terminate_child_handler(int sig)
{
  int status;
  pid_t child_pid;
  int exit_status;
  
  if ((child_pid = wait(&status)) < 0)
  {
    if (errno == EINTR)
    {
       debug ("bug: terminate_child_handler, wait interrupted by a signal which returned. Recalling terminate_child_handler");
       terminate_child_handler(0);
    }
    else if (errno == ECHILD)
    {
      debug("in terminate_child_handler no child to wait, status should be collected anyway");
      return;
    }
    else
    {
      debug ("Unknown error value for wait: %d, exiting", errno);
      exit (1);
    }
  }
  if (!child_pid)
  {
    debug("bug: ssh child exited with status 0 before sending SIGUSR1, exiting");
    exit (1);
  }
  if (WIFEXITED(status))
  {
    exit_status = WEXITSTATUS(status);
    debug ("Child (forwarding daemon) exited before finalizing ssh tunnel, status %d", WEXITSTATUS(status));
    error ("%s", translate_exit_status (exit_status));
    exit (exit_status);
  }
  if (WIFSIGNALED(status))
    debug ("Child (forwarding daemon) caught signal %d before finalizing ssh tunnel",WTERMSIG(status));
  log("%s[%d] (forwarding daemon parent) caught %d, child died, exiting", prog_name, getpid(),sig);
  exit(1);
}

/* to avoid zombies */
void
child_handler(int sig)
{
  int status;
  pid_t child_pid;

  if (sig == SIGCHLD)
    signal(SIGCHLD, child_handler);
  else if (sig == SIGPIPE)
    signal(SIGPIPE, child_handler);
  
  if ((child_pid = wait(&status)) < 0)
  {
    if (errno == EINTR)
    {
       debug ("Wait interrupted by a signal which returned.");
     }
     else if (errno == ECHILD)
     {
       debug("in child_handler no child to wait for");
       /* sanity check */
       if (ssh_pid)
         debug ("bug ? No child to wait for, but ssh_pid is set..."); 
     }
     else
     {
       debug ("Unknown error value for wait: %d", errno);
     }
  }
  else
  {
    ssh_pid = 0;
    if (WIFEXITED(status))
    {
      ssh_child_status = WEXITSTATUS(status);
      debug ("Child (ssh) exited with status %d", WEXITSTATUS(status));
    }
    if (WIFSIGNALED(status))
    {
      ssh_child_signaled = 1;
      debug ("Child (ssh) caught signal %d",WTERMSIG(status));
    }
  }
}

/* to handle SIGTERM, SIGQUIT, SIGINT, SIGHUP, SIGPIPE and SIGCHLD */
void
terminate_handler(int sig)
{
  int i;
  int status;
  pid_t wait_result, child_pid;
  u_long prog, vers;
  
  signal (SIGALRM, SIG_DFL);

  if (!clean_flag)
  {
    debug ("Exit without cleaning");
    if ((sig > 0) && (sig != SIGCHLD))
    {
      log("%s going down on signal %d", prog_name, sig);
      signal(sig,SIG_DFL);
      raise (sig);
    }
    exit(SEC_RPC_EXIT_NO_CLEAN);
  }
  if (sig > 0)
    debug("process %d caught %d.",getpid(),sig);
  else 
    debug("process %d got a fatal error", getpid());

  /* unset the registered ports with the portmapper */
  for (i = 0; i < options.num_udp_fw; i++)
  {
    prog = options.udp_forward[i].prog,
    vers = options.udp_forward[i].vers;
    
	if (options.udp_forward[i].flags & SEC_RPC_SET)
    {
      debug ("Unsetting %d,%d",prog, vers);
      /* make sure we don't unset the portmap entry even if a signal 
         is caught after the call to pmap_unset */
      options.udp_forward[i].flags &= ~SEC_RPC_SET ;
      pmap_unset(prog, vers);
      if (options.udp_forward[i].old_port)
      {
        debug("resetting %u,%u with %d", prog, vers, 
            options.udp_forward[i].old_port);
        pmap_set(prog, vers, IPPROTO_UDP, options.udp_forward[i].old_port);
      }
    }
  }
  
  if (ssh_pid > 0)
  {
    wait_result = waitpid(ssh_pid, &status, WNOHANG|WUNTRACED);
    if (wait_result < 0)
    {
      /* If we have been interrupted, the interrupting 
         function should never return */
      if (errno == EINTR)
      {
        debug("bug: waitpid interrupted by a signal which returned");
      }
      else if ((errno == ECHILD) && (sig == SIGCHLD))
      {
        debug("SIGCHLD caught but %d is not our child", ssh_pid); 
      }
      else
      {
        perror ("waitpid");
      }
    }
    else if (wait_result == 0) /* a child is running, kill it */
    {
      if (sig > 0)
        debug ("killing ssh child %d while trapping %d", ssh_pid, sig);
      else 
        debug ("killing ssh child %d while in fatal error cleanup handler", ssh_pid);
      signal(SIGCHLD, child_handler);
      if (kill(ssh_pid, SIGHUP) < 0)
        debug ("kill ssh child failed: %s", strerror(errno));
      else
      {
        if ((child_pid = wait(&status)) < 0)
        {
          if (errno == EINTR)
          {
            debug ("wait interrupted by a signal which returned.");
          }
          else if (errno == ECHILD)
          {
            debug("no child to wait for ?");
          }
          else
          {
            debug ("Unknown error value for wait: %d", errno);
          }
        }
        else if (!child_pid)
          debug("Warning: killed by SIGHUP, ssh child exited with status 0");
        else
        {
          if (child_pid != ssh_pid)
            debug("bug: child_pid != ssh_pid");
          if (WIFEXITED(status))
          {
            ssh_child_status = WEXITSTATUS(status);
            debug ("Killed by SIGHUP, child (ssh) exited with status %d", WEXITSTATUS(status));
          }
          if (WIFSIGNALED(status))
          {
            ssh_child_signaled = 1;
            if (WTERMSIG(status) == SIGHUP)
            {
              debug ("Child (ssh) caught signal %d",WTERMSIG(status));
            }
            else
            {
              debug ("Killed by SIGHUP, child (ssh) caught signal %d",WTERMSIG(status));
            }
          }
        }
      }
    }
    else
    {
      /* child was already terminated. We report the exit status and also
         a warning if we aren't catching SIGCHLD */
      if (sig != SIGCHLD)
        debug ("Child (ssh) exited and we aren't traping SIGCHLD");
      if (WIFEXITED(status))
      {
        ssh_child_status = WEXITSTATUS(status);
        debug ("Child (ssh) exited with status %d", WEXITSTATUS(status));
      }
      if (WIFSIGNALED(status))
      {
        ssh_child_signaled = 1;
        debug ("Child (ssh) caught signal %d",WTERMSIG(status));
      }
    }
  }
  if ((sig > 0) && (sig != SIGCHLD))
  {
    log("%s[%d] going down on signal %d", prog_name, getpid(),  sig);
    signal(sig,SIG_DFL);
    raise (sig);
  }
  
  if ((sig == 0) && (! ssh_child_status))
  {
    debug("%s[%d] exiting with status 1 because of a fatal error", prog_name, getpid());
    exit (1);
  }
  else if (sig == 0)
  {
    debug("%s[%d] exit with status %d because of fatal error and ssh child exit != 0", prog_name, getpid(), SEC_RPC_EXIT_CHILD_EXITED);
    exit (SEC_RPC_EXIT_CHILD_EXITED);
  }
  else
  /* terminating SIGCHLD caught */
  {
    int exit_status;
    
    if (ssh_child_signaled)
    {
       exit_status = SEC_RPC_EXIT_CHILD_SIGNALED;
    }
    else if (ssh_child_status == 0)
    {
      debug ("ssh child exited with status 0, this shouldn't happen. Exiting with 1");
      exit_status = 1;
    }
    else
    {
      exit_status = ssh_child_status;
    }
    debug("%s[%d] exiting with status %d because of a child death",prog_name, getpid(), exit_status);
    error("%s", translate_exit_status(exit_status));
    exit(exit_status);
  }
}

/* wrapper used for cleanup with fatal */ 

void
cleanup (void * context)
{
  terminate_handler(0);
}

/* signal handler for the parent process when normally signaled by the child */

void
terminate_parent (int sig)
{
  if(!quiet) printf("%d\n",pid);
  debug("%s[%d] (forwarding daemon parent) caught %d, exiting normally", prog_name, getpid(), sig);
  exit(0);
}

/*
 * MAIN
 */
int
main(int argc, char **argv)
{
  struct sockaddr_in 
      serv_addr,        /* UDP server address */
      pmap_addr,        /* portmapper address */
      test_port_addr;   /* address used to test the port registered with the
                           rpc program,version we want */
	  
  int ssh_write_side,   /* write side of the SSH pipe */
      ssh_read_side,    /* read side of the SSH pipe */
      registered_port;  /* port registered with the same programm number */

  u_int addr_len,         /* length of a socket address */
      wait_before_retry = 0,  /* time to wait for before retrying ssh */
      ssh_retries = 0;  /* how many times ssh was retried unsuccessfully */

  int sockfd,           /* UDP socket descriptor */
      sock_test,        /* socket descriptor used to test the port registered
                           with the rpc program,version we want */
      bgr_flag = 0,     /* to run in background ? */
      i, len;

  u_long 
      prog,             /* RPC program number of a request */
      vers;             /* RPC version number of a request */

  char 
      *config_file,     /* name of the config file */
      *id_string,       /* the remote partner's ID string */
      *err_file = NULL; /* if specified and in background, write stderr into this file */
  struct passwd *passwd_entry; /* entry in the passwd database */
  uid_t uid;           /* uid of the user to run server as */
  int error_read = 0;      /* error reading the remote side's ID string */ 

  
  void *context=NULL;  /* needed by fatal_add_cleanup */
  char *args[MAX_ARGS]; 
                        /* the args for execvp (reserve enough entries) */

  /*
   * variables for getopt
   */
  extern char *optarg;
  extern int optind;
  int ch;

  /*
   * process the arguments
   */
  if (strchr(argv[0], '/'))
    prog_name = strrchr(argv[0], '/') + 1;
  else
    prog_name = argv[0];

  while ((ch = getopt(argc, argv, "bdqrfse:")) != EOF)
  {
    switch(ch)
    {
      case 'b':
          bgr_flag = 1;
          break;
      case 'd':
          debug_flag = 1;
          break;
      case 'q':
          quiet = 1;
          break;
      case 'r':
          root = 1;
          break;
      case 'e':
          err_file = optarg;
          break;
      case 'f':
          force = 1;
          break;
      case 's':
          show_msg = 1;
          break;
      case '?':
      default:
          usage(prog_name);
    }
  }
  argc -= optind;
  argv += optind;

  if (argc != 1) 
  {
    usage(prog_name);
  }

  config_file = argv[0];

  /*
   * read the config file and initialize the logging
   */
  fill_default_server_options(&options);
  read_server_config(&options, config_file);

  if (force)
  {
    options.keep_pmap = 1;
  }

  if (!show_msg && options.show_msg)
     show_msg = 1;

  log_init(prog_name, debug_flag, options.log_to_syslog, 
           options.quiet_mode, options.log_facility);

  debug("config file read");

  /*
   * check the completeness of the options
   */
  if (!options.ssh_command && !options.host)
  {
    fprintf(stderr, "config file: no ssh command line or host specified\n");
    exit(1);
  }
  
  /* dump the security specifications associated with forwarded programs */
  if (debug_flag)
    dump_forward(options.udp_forward, options.num_udp_fw);
  
  for(i=0; i < MAX_ARGS; i++) args[i]=0;
  
  /* split the command line into args for execvp */
  if (options.ssh_command)
    parse_command(options.ssh_command, args);
  else if (options.host)
  /* concatenate tunnel_command, host name (or login@host), remote command
   * and "-M local_program:remote_program" options */
  {
    char *tunnel[MAX_ARGS], *remote[MAX_ARGS];
    char *debug_arg[2];
    char * login_host[2];
    char * arg_forwarding[2 * MAX_FORWARD +1];
    char login_host_string[2 * MAXHOSTNAMELEN + 1];
    u_int idx_udp_fw = 0;
    int idx_arg_forwarding = 0;

    if (options.tunnel_command)
      parse_command(options.tunnel_command, tunnel);
    else duplicate_command(tunnel, default_tunnel_command);

    if (options.remote_command)
      parse_command(options.remote_command, remote);
    else duplicate_command(remote, default_remote_command);
	
    if (options.remote_id)
    {
      snprintf(login_host_string, 2 * MAXHOSTNAMELEN + 1, "%s@%s", options.remote_id, options.host);
    }
    else
      snprintf(login_host_string, MAXHOSTNAMELEN, "%s", options.host);
    login_host[0] = login_host_string;
    login_host[1] = NULL;

    debug_arg[0] = "-d";
    debug_arg[1] = NULL;
    
    for (idx_udp_fw = 0; idx_udp_fw < options.num_udp_fw; idx_udp_fw++)
    {
      if (options.udp_forward[idx_udp_fw].remote_prog)
      {
        char forward[60];
        snprintf (forward, 60, "%lu:%lu", options.udp_forward[idx_udp_fw].prog,
                  options.udp_forward[idx_udp_fw].remote_prog);
        arg_forwarding[idx_arg_forwarding] = xstrdup("-M");
        arg_forwarding[idx_arg_forwarding + 1] = xstrdup(forward);
        idx_arg_forwarding = idx_arg_forwarding + 2;
      }
    }
    arg_forwarding[idx_arg_forwarding] = NULL;
    if(debug_flag) 
    {
      concatenate_commands(args, 5, tunnel, login_host, remote,
			   debug_arg, arg_forwarding);
    } else {
      concatenate_commands(args, 4, tunnel, login_host, remote,
			   arg_forwarding);
    }
  }
  else 
    fatal("need a Host or SshCommand");

  /*
   * Change uid
   */
  
  if (options.id && getuid() == 0 && !root)
  {
    if ((passwd_entry = getpwnam (options.id)))
    {
      uid = passwd_entry->pw_uid;
      if (setuid(uid) < 0)
      {
        fatal ("Cannot change to uid: %d, errno: %s",uid,strerror(errno));
      }
      else
      {
        debug ("Changed uid to: %d", uid);
      }
    }
    else 
    {
      fatal ("Problem finding user %s in passwd database", options.id); 
    }
  }
 
  /*
   * create a server socket to listen to client requests
   */
  if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
  {
    fatal("can't open datagram socket for server");
  }
  debug("UDP server socket created");

  /* 
   * setup local address
   */
  bzero((char *) &serv_addr, sizeof(serv_addr));
  serv_addr.sin_family      = AF_INET;
  serv_addr.sin_addr.s_addr = options.listen_addr.s_addr;
  serv_addr.sin_port = htons(options.udp_port);
  
  /*
   * If given, bind the listening port
   */
  if (htons(options.udp_port) != 0)
  {
    if (bind(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0) 
    {
      fatal("can't bind local address for server: %s",strerror(errno));
    }
  }

  /*
   * set up portmap address for server, used to find the port registered
   * with the program,version we want
   */
  
  bzero((char *) &pmap_addr, sizeof(pmap_addr));
  pmap_addr.sin_family      = AF_INET;
  pmap_addr.sin_addr.s_addr = options.listen_addr.s_addr;
  pmap_addr.sin_port        = htons(PMAPPORT);
  
  /*
   * prepare address used to test the port registered by portmap with the
   * program,version we want
   */
  
  bzero((char *) &test_port_addr, sizeof(test_port_addr));
  test_port_addr.sin_family      = AF_INET;
  test_port_addr.sin_addr.s_addr = options.listen_addr.s_addr;

  /*
   * setup signal handlers
   */

  /* maybe we should catch some more signals */
  signal(SIGTERM, terminate_handler);
  signal(SIGQUIT, terminate_handler);
  signal(SIGINT, terminate_handler);
  signal(SIGHUP, terminate_handler);

  /*
   * setup cleanup function
   */

  fatal_add_cleanup(cleanup,context);

  /* 
   * unregister the ports having SEC_RPC_UNSET_PMAP flag set
   * if the flag isn't set, try to bind to the already registered port
   * if successful, use that port if it is the first binding, else unset it
   */
  for (i = 0; i < options.num_udp_fw; i++)
  {
    prog = options.udp_forward[i].prog, 
    vers = options.udp_forward[i].vers;

    if (!(options.udp_forward[i].flags & SEC_RPC_TO_REGISTER))
      continue;

	registered_port = pmap_getport (&pmap_addr, prog, vers, IPPROTO_UDP);
    if (!registered_port)
      continue;

	if (htons(registered_port) == serv_addr.sin_port)
    {
      debug("%u,%u already registered with %u", prog, vers, registered_port);
      options.udp_forward[i].flags |= SEC_RPC_REGISTERED;
      options.udp_forward[i].flags &= ~SEC_RPC_SET;
      continue;
    }

    if ((options.udp_forward[i].flags & SEC_RPC_UNSET_PMAP) || !options.keep_pmap)
    {
      debug("Unsetting %u,%u, storing old port %u", prog, vers, registered_port);
      pmap_unset(prog, vers);
      options.udp_forward[i].old_port = registered_port;
    }
    else if (!serv_addr.sin_port)
    /* try to bind to the port which was originally registered with the 
    * program number */
	{
      serv_addr.sin_port = htons(registered_port);
      if (bind(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) == 0)
      {
        debug ("bound to port already registered %u for %u,%u", 
             registered_port, prog, vers);
        options.udp_forward[i].flags |= SEC_RPC_REGISTERED;
        options.udp_forward[i].flags &= ~SEC_RPC_SET;
      }
      else 
      {
        fatal ("%u,%u already registered with port %u", prog, vers, 
              registered_port);
      }
    }
    else
    /* test if the port associated with the prog,vers we want is still in use
     * by binding to it. If the binding is successful, unset it */
    {

      if ((sock_test = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
        error ("can't open socket to test registered port %u for %u,%u",
                registered_port, prog, vers);
      else 
      {
        test_port_addr.sin_port = htons(registered_port);
        if (bind(sock_test, (struct sockaddr *) &test_port_addr, sizeof(test_port_addr)) == 0)
        {
          if (close(sock_test) < 0)
            error("can't close socket testing registered port %u for %u,%u",
                  registered_port, prog, vers);
          debug("unsetting %u,%u after having bind to the registered port %u",
                  prog, vers, registered_port);
          pmap_unset(prog, vers);
          options.udp_forward[i].old_port = registered_port;
        }
        else
          fatal("%u,%u already registered with port %u", prog, vers,
                registered_port);
      }
    }
  }
	
  if (!serv_addr.sin_port)
  {
    /* 
     * get a free port 
     */
    serv_addr.sin_port = 0;
    if (bind(sockfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0) 
    {
      fatal("can't bind local address for server: %s",strerror(errno));
    }
  }
  
  /* 
   * read the binding back 
   */
  addr_len = sizeof(serv_addr);
  getsockname(sockfd, (struct sockaddr *) &serv_addr, &addr_len);
  s_port = ntohs(serv_addr.sin_port);

  debug("listening socket bound to address %s, port %d", 
        inet_ntoa(serv_addr.sin_addr), ntohs(serv_addr.sin_port));

  /* 
   * register the ports having SEC_RPC_TO_REGISTER flag set with the portmapper 
   */
  for (i = 0; i < options.num_udp_fw; i++)
  {
    prog = options.udp_forward[i].prog, 
    vers = options.udp_forward[i].vers;

    if (!(options.udp_forward[i].flags & SEC_RPC_TO_REGISTER) 
       || (options.udp_forward[i].flags & SEC_RPC_REGISTERED))
      continue;

    if (!pmap_set(prog, vers, IPPROTO_UDP, s_port))
    {
      fatal("couldn't register UDP prog %d, vers %d, port %d with portmapper",
            prog, vers, s_port);
    }
    else
    {
      options.udp_forward[i].flags |=  SEC_RPC_SET;
    }
    debug("registered UDP prog %d, vers %d, port %d with portmapper", 
          prog, vers, s_port);
  }

  /* 
   * If required put ourselves into the background,
   * and let the child do all the work.
   * set clean_flag to 0 such that the parent don't do any cleaning
   */
  if (bgr_flag)
  {
    clean_flag = 0;
    if ((pid = fork()) < 0)
    {
      fatal("can't fork the proxy");
    }
    else
    {
      /*
       * parent. Wait to receive SIGUSR1 from the child once the child is
       * ready to forward, or for the child to die.
       */
      if (pid)
      {
        int status;
        int child_pid;
        int exit_status;
        signal (SIGUSR1,terminate_parent);
        signal (SIGTERM,SIG_DFL);
        signal (SIGHUP,SIG_DFL);
        signal (SIGINT,SIG_DFL);
        signal (SIGQUIT,SIG_DFL);
        signal(SIGCHLD, terminate_child_handler);
        /* wait for the child to die, or a signal interrupting the wait */
        if ((child_pid = wait(&status)) < 0)
        {
          if (errno == EINTR)
          {
            debug ("bug: in parent wait interrupted by a signal which returned. Exit with status 1");
            exit (1);
          }
          else if (errno == ECHILD)
          {
            debug("bug: in parent no child to wait for ?");
            exit (1);
          }
          else
          {
            debug ("Unknown error value for wait: %d, exiting", errno);
            exit (1);
          }
        }
        if (!child_pid)
        {
          debug("bug: ssh child exited with status 0 before sending SIGUSR1, exiting");
          exit (1);
        }
        if (WIFEXITED(status))
        {
          exit_status = WEXITSTATUS(status);
          debug ("Child (forwarding daemon) exited before finalizing ssh tunnel, status %d", WEXITSTATUS(status));
          error ("%s", translate_exit_status (exit_status));
          exit (exit_status);
        }
        if (WIFSIGNALED(status))
        {
          debug ("Child (forwarding daemon) caught signal %d before finalizing ssh tunnel",WTERMSIG(status));
          raise (WTERMSIG(status));
        }
        debug("parent exiting normally, should never happen");
        exit(0); /* never go here, either killed or exit in terminate_child_handler */
      }
    }

    clean_flag = 1;

    /* redirect stdin, stdout to /dev/null, and stderr to /dev/null or a file
     * specified with -e */
    if (freopen("/dev/null", "r", stdin) == NULL)
       error("Cannot reopen stdin to /dev/null: %s", strerror(errno));
    if (freopen("/dev/null", "w", stdout) == NULL)
       error("Cannot reopen stdout to /dev/null: %s", strerror(errno));
    if (err_file)
    {
      if (freopen(err_file, "w", stderr) == NULL)
      {
        error("Cannot open debug file, redirect to /dev/null: %s", strerror(errno));
        if (freopen("/dev/null", "w", stderr) == NULL)
          error("Cannot reopen stderr to /dev/null: %s", strerror(errno));
      }
    }
    else if (freopen("/dev/null", "w", stderr)  == NULL)
        error("Cannot reopen stderr to /dev/null: %s", strerror(errno));

    if (setsid() < 0) /* set process to session leader */
      fatal("setsid failed: %s",strerror(errno));

    /* initialize the log in the child again */
    log_init(prog_name, debug_flag, options.log_to_syslog,
             options.quiet_mode, options.log_facility);
  }

  if (options.retry_ssh)
  {
    signal(SIGPIPE, child_handler);
    signal(SIGCHLD, child_handler);
  }
  else
  {
    signal(SIGPIPE, terminate_handler);
    signal(SIGCHLD, terminate_handler);
  }
  
  /* chdir to / in case the cwd has to be unmounted */
  if (chdir("/") < 0)
    error("rpc forwarder cannot chdir to /: %s",  strerror(errno));

  /* ssh reconnect loop */
  do
  {

    debug ("Begining retry ssh loop; wait_before_retry: %u", wait_before_retry);
	
    /* if wait_before_retry is set it means that a tunnel was successfully 
     * set up once, and that ssh disconnected at least once */
    if (wait_before_retry)
    {
      /*
       * close pipe opened to communicate with ssh
       */	
      if (close(ssh_write_side)<0)
        error("close(ssh_write_side) failed :%s", strerror(errno));
      if (close(ssh_read_side)<0)
        error("close(ssh_read_side) failed :%s", strerror(errno));
      ssh_retries++;
      if (options.retry_limit && ssh_retries > options.retry_limit)
        fatal("Ssh retry limit exceded, giving up");
      debug ("sleeping %d", wait_before_retry);
      /* sanity check */
      if (!options.retry_ssh)
        fatal("bug: wait_before_retry set at the ssh loop begining but no retry_ssh");
      sleep (wait_before_retry);
      /* double the time we wait before retrying connection such that we don't
       * overflow the network in case of an unreachable server */
      if (2 * wait_before_retry < SEC_RPC_MAX_WAIT)
        wait_before_retry *= 2;
      else
        wait_before_retry = SEC_RPC_MAX_WAIT;
    }

    /* sanity check */
    if (ssh_pid)
      fatal("bug: ssh_pid != 0 before the ssh connection");
	
    /*
     * start ssh and reset ssh variables
     */

    error_read = 0;
    ssh_child_signaled = 0;
    ssh_child_status = 0; /* could there be something better ? a negative
                           * value ? */

    if (start_ssh(&ssh_read_side, &ssh_write_side, args, sockfd))
    {
      debug("ssh started");
    }
    else
    {
      error ("Couldn't start ssh child");
      continue;
    }
  
    /*
     * wait for the identification string
     */
    len = strlen(options.id_string);
    id_string = xmalloc(len+1);
    for (i = 0; i < len; i++)
    {
      /* note: read is still blocking here */
      if (read(ssh_read_side, &id_string[i], 1) != 1) 
      {
        error_read = 1;
        error("error reading the remote side's ID string: %s", strerror(errno));
        break;
      }
    }
    if (error_read)
    {
      free (id_string);
      continue;
    }
  
    id_string[len]='\0';

    if (memcmp(id_string, options.id_string, len))
    {
      error("ID string is incorrect");
      free (id_string);
      continue;
    }
    debug("ID String correctly read: %s", id_string);
    free (id_string);
  
    /*
     * Do things that have to be done once if wait_before_retry == 0
     */

    if (!wait_before_retry)
	{
    /*
     * if in the child, kill the parent now that we're ready to forward
     */

      if ((bgr_flag) && (!pid))
      { 
        kill(getppid(), SIGUSR1);
      }

    /* 
     * initialize the map table
     */
      map_init(options.max_ref_count);
      debug("map table initialized");

    /*
     * Mark the programs which are registered, but are not accepted. As
     * no packet will be accepted for them they cannot be marked as
     * not set when a packet is send thrrough them. Instead, if they unset
     * pmap, they are left set but we want this program to be unset on exit
     * and thus we set old_port = 0. If they don't unset pmap we mark them 
     * as not set such that they are not unregistered.
     */
      for (i = 0; i < options.num_udp_fw; i++)
      {
        if ((options.udp_forward[i].flags & SEC_RPC_TO_REGISTER) && 
            (options.udp_forward[i].flags & ~SEC_RPC_ACCEPTED))
        {
          if (options.udp_forward[i].flags & SEC_RPC_UNSET_PMAP)
            options.udp_forward[i].old_port = 0;
          else
            options.udp_forward[i].flags &= ~SEC_RPC_SET;
        }
      }
    /*
     * Setup a signal handler for SIGUSR1 to reregister with portmap
     */

     signal(SIGUSR1, portmap_register_handler);


    }

    /*
     * the tunnel was successfully set up. If it is the first time the tunnel
     * is set up, set the time waited before trying to reconnect, such that 
     * this variable is not 0 if there was at least one successful
     * conection.
     * if it isn't the first tunnel, reset the time waited before trying to 
     * reconnect, if there had been a deconnection, the time waited may have
     * been raised, now we are sure that the server can be reached.
     */
    wait_before_retry = SEC_RPC_MIN_WAIT;
    ssh_retries = 0; 
    /*
     * forward all requests and replies. return if ssh die, or connection lost
     */

    handle_forwarding(sockfd, ssh_write_side, ssh_read_side);

    /* there was an error which stopped the forwarding */
    signal(SIGALRM, SIG_IGN);

    debug ("Forwarding stopped");
    
    if (!options.retry_ssh)
      fatal("Remote side closed the connection");

    /*
     * Kill the ssh child if it isn't already terminated.
     */

    while (ssh_pid != 0)
    {
      int status;
      pid_t wait_result;
      
      wait_result = waitpid(ssh_pid, &status, WNOHANG|WUNTRACED);
      if (wait_result < 0)
      {
        if (errno == EINTR)
        {
          debug("waitpid interrupted by a signal which returned");
          continue;
        }
        else if (errno == ECHILD)
        {
          debug("ssh child seems already dead");
          ssh_pid = 0;
        }
        else
        {
          perror ("waitpid");
          /* FIXME what to do now ?*/
        }
      }
      else if (wait_result == 0) /* a child is running, kill it */
      {
        debug ("killing ssh child %d before retrying connection", ssh_pid);
        if (kill(ssh_pid, SIGHUP) < 0)
          debug ("kill ssh child failed: %s", strerror(errno));
      }
      else
      {
        /* child was already terminated. */
        ssh_pid = 0;
        if (WIFEXITED(status))
        {
          ssh_child_status = WEXITSTATUS(status);
          debug ("Child (ssh) exited with status %d", WEXITSTATUS(status));
        }
        if (WIFSIGNALED(status))
        {
          ssh_child_signaled = 1;
          debug ("Child (ssh) caught signal %d",WTERMSIG(status));
        }
      }
      /* sleep in case child takes time to die. In normal cases we shall
       * catch SIGCHLD. */
      sleep (2);
    }
  }
  /* if wait_before_retry is 0 it means that the tunnel wasn't set up for the 
   * first time. In that case we exit without retrying the ssh connection. 
   * Otherwise, if retry_ssh is set, we retry the ssh connection */
  while (wait_before_retry && options.retry_ssh);
  
  fatal ("other side closed the connection before end of tunnel setup");
  
  return 0;
}
