/*
 * rpc_pcl.c
 * Copyright (C) 2001 Holger Trapp and John C. Bowman 
 *
 * RPC proxy client
 *
 * This client should run on the host where the real RPC server lives.
 * It gets the requests from the RPC proxy server, fowards them to
 * the RPC server and sends the replies back. It communicates with
 * the RPC proxy server via a secure SSH connection. This client sees
 * the SSH pipe as stdin and stdout.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Original author: 26 Feb 1996, Holger Trapp (hot@informatik.tu-chemnitz.de)
 * Adapted for NFS: 09 May 2001, John C. Bowman <bowman@math.ualberta.ca>
 *
 * 06 Dec 2001, rune.saetre@netcom-gsm.no
 *  - Modified to support HPUX
 *  - "-h" option added to support remote rpc server
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <signal.h>
#include <syslog.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <sys/stat.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <rpc/pmap_prot.h>
#if defined sun || defined hppa
#include <rpc/clnt_soc.h>
int bindresvport(int sd, struct sockaddr_in *sin);
#endif

#include "log-server.h"
#include "buffer.h"
#include "xmalloc.h"
#include "rpc_to_str.h"

#define PACKET_MAGIC 0x6feeddcc
#define ID_STRING    "RPC Proxy Client\n"
#define MAX_PMAP 100
#define MAX_HOSTNAME 100
#define MAX_ERRFILE 1024
#define DEFAULT_BASE_DIR "/var/log/rpc_pcl"
#define USER_DIR ".rpc_pcl"

/*
 * global variables
 */
static ClntOptions options;           /* the security options */
static int debug_flag = 0;            /* if true, run in debug mode */
static struct pmap pm_list[MAX_PMAP]; /* internal portmapper list */
static int n_pmap = 0;                /* number of entries in pm_list */
static struct sockaddr_in serv_addr;  /* address of the RPC server */
static int dump_interval = 60;        /* when to dump the portmapper again */
static int nmap=0;
static u_long mapfrom[MAX_FORWARD];
static u_long mapto[MAX_FORWARD];
static char request_dump[DUMP_SIZE]; /* a request debug info */
static int security_flag = 1;        /* if true security checks */
/*
 * show the usage
 */
void usage(char *prog)
{
  fprintf(stderr, "Usage: %s [options]\n", prog);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "   -d       run in debug mode\n");
  fprintf(stderr, "   -e file  redirect stderr to file relative to dir (only in debug mode)\n");
  fprintf(stderr, "   -b dir   base dir for error file (only in debug mode)\n");
  fprintf(stderr, "   -i secs  portmapper dump interval (in secs)\n");
  fprintf(stderr, "   -l       don't log to syslog\n");
  fprintf(stderr, "   -m       with magic in each packet\n");
  fprintf(stderr, "   -p       ask portmapper for each request\n");
  fprintf(stderr, "   -q       quiet logging (no logging)\n");
  fprintf(stderr, "   -h host  hostname of real RPC server\n");
  fprintf(stderr, "   -P port  port to bind to (-1 to request a free privileged port)\n");
  fprintf(stderr, "   -M x:y   map incoming RPC program number x to y\n");
  exit(1);
}

/*
 * dump the server portmapper (UDP services only)
 */
void
pmap_dump(struct sockaddr_in *pm_addr)
{
  struct pmaplist *plp;

  debug("pmap_dump called");
  n_pmap = 0;

  plp = pmap_getmaps(pm_addr);
  debug("pmap_getmaps OK");

  while (plp != NULL && n_pmap < MAX_PMAP)
  {
    if (plp->pml_map.pm_prot == IPPROTO_UDP)
    {
      pm_list[n_pmap++] = plp->pml_map;
    }
    plp = plp->pml_next;
  }
  debug("pmap_dump dumped %d entries", n_pmap);
}

/*
 * search for a service in the internal portmapper list 
 */
u_long
pmap_search(u_long prog, u_long vers)
{
  int i;

  for (i = 0; i < n_pmap; i++)
  {
    if (pm_list[i].pm_prog == prog &&
        pm_list[i].pm_vers == vers)
    {
      return pm_list[i].pm_port;
    }
  }

  return 0; /* not found */
}

/*
 * forward RPC requests and replies (UDP only)
 */
void
handle_forwarding(int sockfd, int with_magic, int ask_pm)
{
  int nread,          /* number of bytes read */
      nwritten,       /* number of bytes written */
      nready,         /* number of ready descriptors after select */
      max_rfd,        /* maximum read fd for select */
      check_res = 0;  /* return status of check_rpc_request */

  u_long
      vers,           /* RPC version number of a request */
      packet_length,  /* length of the packet sent via SSH */ 
      packet_magic = htonl(PACKET_MAGIC),
                      /* the magic to detect when being out of sync */
      xid;            /* xid of the error reply */
	  

  fd_set readfds;     /* set of descriptors for select */ 

  struct sockaddr_in 
      dummy_addr;     /* the RPC server's address when getting replies */

  u_short 
      server_port;    /* the port where the RPC server listens on */

  u_int dummy_len;      /* length of the dummy address */

  struct timeval tv;  /* for timeout with select */
  int i;

  XDR xdr;
  
  /*
   * buffers
   */
  Buffer repl_buffer; /* buffer for the replies received from the server */

  u_char 
      request[UDPMSGSIZE],
                      /* a single request datagram */
      reply[UDPMSGSIZE],
                      /* a single reply datagram */
      reply_error[UDPMSGSIZE],
                      /* a reply datagram in case of auth error */
      header[8];      /* header (magic and length) */

  int exp_reqlen,     /* expected length for a request */
      req_len,        /* current reply length */
      head_len,       /* header length for the reply */
      buflen;         /* buffer length */
  
  /*
   * file descriptors to read from and write to SSH
   */
  int ssh_read_side  = fileno(stdin),
      ssh_write_side = fileno(stdout);

  struct rpc_msg msg_error;
  msg_error.rm_xid = 0;
  msg_error.rm_direction = REPLY;
  msg_error.ru.RM_rmb.rp_stat = MSG_DENIED;
  msg_error.ru.RM_rmb.ru.RP_dr.rj_stat = AUTH_ERROR;
  msg_error.ru.RM_rmb.ru.RP_dr.ru.RJ_why = AUTH_TOOWEAK;

  /*
   * read and write the pipe in non-blocking mode
   */
  if (fcntl(ssh_write_side, F_SETFL, O_NONBLOCK) < 0)
      error("fcntl O_NONBLOCK: %.100s", strerror(errno));
  if (fcntl(ssh_read_side, F_SETFL, O_NONBLOCK) < 0)
      error("fcntl O_NONBLOCK: %.100s", strerror(errno));

  /*
   * initialize the buffers
   */
  buffer_init(&repl_buffer); 
  exp_reqlen = req_len = 0;
  head_len = with_magic ? 0 : 4;

  /*
   * determine the maximum number of elements in fd sets for select
   */
  max_rfd = sockfd > ssh_read_side ? sockfd+1: ssh_read_side+1;

  /*
   * central forwarding loop
   */
  debug("starting central forwarding loop");
  for (;;)
  {
    /*
     * prepare the fd sets for select
     */
    FD_ZERO(&readfds);

    FD_SET(sockfd, &readfds);
    FD_SET(ssh_read_side, &readfds);


    /*
     * step one: if there is data to be written, write as much as possible
     */
	
    /* request from any client complete ? */
    if (exp_reqlen && exp_reqlen == req_len)
    {
      char *request_debug = NULL;
      /* yes, then send it to the server */

      u_int prog=((struct rpc_msg_32 *)request)->rm_call.cb_prog;

      for(i=0; i < nmap; i++) 
      {
        if(mapfrom[i] == prog) 
        {
          ((struct rpc_msg_32 *)request)->rm_call.cb_prog=mapto[i];
	      break;
        }
      }

      if (debug_flag)
      {
        debug("request from client complete, %d bytes", req_len);
        debug("got %s", rpc_request_to_str(request, req_len, options.show_msg));
      }

      if (security_flag)
      {
        /*
         * test whether the request is allowed or not
         */
        if (debug_flag)
        {
          request_debug = request_dump;
        }
        if ((check_res = check_rpc_request(request, req_len, options.accepted_forward, options.num_acc_fw, request_debug, &xid, options.show_msg)) < 0)
        {
          debug("authentication failed: %s", request_debug);
          if (check_res == SEC_RPC_REPLY_ERROR)
          /* reply an error message */
          {
            msg_error.rm_xid = xid;
            xdrmem_create(&xdr, (char *) reply_error, UDPMSGSIZE, XDR_ENCODE);
            if (xdr_replymsg(&xdr, &msg_error))
            {
              int reply_error_len = XDR_GETPOS (&xdr);
              /* form a packet and append it to the buffer */
              if (with_magic)
              {
                buffer_append(&repl_buffer, (u_char *) &packet_magic, 4);
  			}
              packet_length=htonl(reply_error_len);
              buffer_append(&repl_buffer, (u_char *) &packet_length, 4);
              buffer_append(&repl_buffer, reply_error, reply_error_len);
              debug("error reply packet appended to reply buffer: %s (%d)",
                rpc_reply_to_str(reply_error, reply_error_len, options.show_msg), reply_error_len);
            }
            else
            {
              error("couldn't encode error reply");
	        }
          }
        }
        else
        {
          debug("authentication passed");
        }
      }
      /*
       * determine program and version number of the request
       */
      prog = ntohl(((struct rpc_msg_32 *)request)->rm_call.cb_prog);
      vers = ntohl(((struct rpc_msg_32 *)request)->rm_call.cb_vers);

      /*
       * find out the server port
       * requests to unknown servers are silently dropped
       */
      if (ask_pm)
      {
        /* ask the portmapper on the rpc server for the server port */
        dummy_addr = serv_addr;
        server_port = pmap_getport(&dummy_addr, prog, vers, IPPROTO_UDP);
      }
      else
      {
        /* search in the internal list */
        server_port = pmap_search(prog, vers);
      }

      /* 
       * now actually call the server
       */
      if (server_port)
      {
        serv_addr.sin_port = htons(server_port);

        if (debug_flag)
        {
          debug("sendto addr/port %s/%d: %s", inet_ntoa(serv_addr.sin_addr),
                ntohs(serv_addr.sin_port), 
                rpc_request_to_str(request, req_len, options.show_msg));
        }
  
        if (sendto(sockfd, request, req_len, 0, 
		    (struct sockaddr *) &serv_addr, sizeof(serv_addr)) != req_len)
        {
          error("couldn't send request to server");
        }
        else
        {
          debug("request sent to server, %d bytes", req_len);
          /* clear the buffer */
          req_len = exp_reqlen = 0;
          head_len = with_magic ? 0 : 4;
        }
      }
      else
      {
        if (debug_flag)
        {
          debug("no server available for prog %d/ vers %d, request dropped",
                prog, vers);
        }
        /* clear the buffer */
        req_len = exp_reqlen = 0;
        head_len = with_magic ? 0 : 4;
      }
    }

    /* data for client available ? */
    if ((buflen = buffer_len(&repl_buffer)) > 0)
    {
      /* yes, send as much as possible */
      debug("data for client available");
      nwritten = write(ssh_write_side, buffer_ptr(&repl_buffer), buflen);
      debug("%d bytes written to SSH", nwritten);

      if (nwritten == -1 && errno != EAGAIN)
      {
        error("write to pipe failed");
      }
      else
      {
        if (nwritten > 0)
        {
          buffer_consume(&repl_buffer, nwritten);
        }
      }    
    }

    /* 
     * step two: test whether we can read data
     * 
     * if there is no data left to be written then
     * block in select
     */

    if ((exp_reqlen && exp_reqlen == req_len) || buffer_len(&repl_buffer))
    {
      /* we should write as soon as possible, therefore don't
       * block in select
       */
      tv.tv_sec = tv.tv_usec = 0;
      debug("select without blocking (%d descriptors)", max_rfd);
      nready = select(max_rfd, &readfds, NULL, NULL, &tv);
    }
    else
    {
      /* wait until data is available to be read, because there is 
       * no other job to be done
       */
      debug("select with blocking (%d descriptors)", max_rfd);
      nready = select(max_rfd, &readfds, NULL, NULL, NULL);
    }

    if (nready == -1 && errno != EINTR)
    {
      /* there was an error */
      error("select failed");
    }

    if (nready > 0)
    {
      /* yes, we can read 
       * find out from where
       */
      if (FD_ISSET(sockfd, &readfds)) 
      {
        /* read a datagram from the net */
        debug("can read datagram");
        
        dummy_len = sizeof(dummy_addr);
        nread = recvfrom(sockfd, reply, sizeof(reply), 0, 
		                 (struct sockaddr *) &dummy_addr, &dummy_len);

        if (nread <= 0) 
        {
          error("error when reading a datagram");
        }
        else 
        {
          if (debug_flag)
          {
            debug("recvfrom got %d bytes from %s/%d: %s", nread,
                  inet_ntoa(dummy_addr.sin_addr), ntohs(dummy_addr.sin_port), 
                  rpc_reply_to_str(reply, nread, options.show_msg));
          }

          /* was it sent by a server on localhost ? */
          if (dummy_addr.sin_addr.s_addr != serv_addr.sin_addr.s_addr)
          { 
            /* bad server */
            debug("reply didn't come from a server on specified host, dropped");
          }
          else
          {
            /* form a packet and append it to the buffer */
            if (with_magic)
            {
              buffer_append(&repl_buffer, (u_char *) &packet_magic, 4);
            }
            packet_length=htonl(nread);
            buffer_append(&repl_buffer, (u_char *) &packet_length, 4);
            buffer_append(&repl_buffer, reply, nread);
            debug("packet appended to reply buffer");
          }
        }
      }

      if (FD_ISSET(ssh_read_side, &readfds)) 
      {
        debug("can read from SSH");
        /* receive via SSH */
        if (head_len < 4)
        {
          /* try to get the magic */
          debug("try to get the magic (%d bytes)", 4 - head_len);
          nread = read(ssh_read_side, header + head_len, 4 - head_len);

          if (!nread) /* EOF */
          {
            fatal("remote partner closed the SSH connection"); 
          }

          if (nread < 0) /* error */
          {
            error("reading magic from pipe failed");
          }
          else /* complete the magic */
          {
            head_len += nread;
            if (head_len == 4 && *((u_long *) header) != packet_magic)
            {
              fatal("out of sync (bad magic: %d)", *((u_long *) header));
            }
          }
        }
        else
        {
          if (head_len < 8)
          {
            /* try to get the length of the reply */
            debug("try to get the length (%d bytes)", 8 - head_len);
            nread = read(ssh_read_side, header + head_len, 8 - head_len);

            if (!nread) /* EOF */
            {
              fatal("remote partner closed the SSH connection");
            }

            if (nread < 0) /* error */
            {
              error("reading length from pipe failed");
            }
            else /* complete the header */
            {
              head_len += nread;
              if (head_len == 8)
              {
                debug("header completely received");
                exp_reqlen = ntohl(*((u_long *) (header + 4)));
                /* check the length */
                if (exp_reqlen > sizeof(reply))
                {
                  fatal("got giant packet: %d bytes", exp_reqlen);
                }
                debug("expecting request with %d bytes", exp_reqlen);
              }
            }
          }
          else
          {
            /* normal data, append it to the reply */
            debug("getting normal data (%d bytes)", exp_reqlen - req_len);
            nread = read(ssh_read_side, request + req_len, 
                         exp_reqlen - req_len);

            if (!nread) /* EOF */
            {
              fatal("remote partner closed the SSH connection");
            }

            if (nread < 0) /* error */
            {
              error("reading data from pipe failed");
            }
            else /* complete the reply */
            {
              debug("got %d bytes normal data", nread);
              req_len += nread;
            }
          }
        }
      }
    }
  } /* for(;;) */
}

/*
 * signal handler
 */

/* to dump the portmapper from time to time */
void
dump_handler(int sig)
{
  pmap_dump(&serv_addr);
  signal(SIGALRM, dump_handler);
  alarm(dump_interval);
}

/*
 * MAIN
 */

int
main(int argc, char **argv)
{
  int sockfd,             /* socket to communicate with the RPC server */
      on_syslog = 1,      /* if true, log to syslog */
      quiet_flag = 0,     /* quiet logging */
      with_magic = 0,     /* prepend a magic to each packet */
      ask_pm = 0,         /* ask the portmapper for each request */
      setid_status;       /* status for setuid */
  u_int addr_len;         /* length of a socket address */
  
  u_short trusted = 0;    /* is the user a trusted user ? */

  pid_t uid = getuid(),   /* real uid */
    file_uid = geteuid(); /*file/effective uid */
   

  struct sockaddr_in 
      local_addr;         /* local address for us as client */

  struct hostent 
      *hp=NULL;                /* for gethostbyname */

  char 
      *prog_name,         /* program name for syslog */
      *err_file = NULL,   /* if specified, write stderr into this file */
      *with_base_file = NULL, /* alternate file for stderr, relative with base_dir */
      *base_dir = NULL;   /* directory for error file. ignored if no -e
                           and if not trusted user */

  /*
   * variables for getopt
   */
  extern char *optarg;
  extern int optind;
  int ch;
  int local_port=0;
  char name[MAX_HOSTNAME+1];
  char sname[MAX_HOSTNAME+1];
  int has_host_name = 0;    /* Is the host IP specified? */
  int i;

  /*
   * Lower priviledges. We need high priviledges only for opening port
   */
#ifdef _POSIX_SAVED_IDS
  setid_status = seteuid(uid);
#else
  setid_status = setreuid(geteuid(), getuid());
#endif
  if (setid_status <0)
  {
    fprintf(stderr,"seteuid (%d): %s", uid, strerror(errno));
    exit (1);
  }
  /*
   * perform unbuffered I/O
   */
  setbuf(stdin, NULL);
  setbuf(stdout, NULL);
  setbuf(stderr, NULL);
  
  /*
   * Read security configuration
   */
  
  fill_default_rpc_pcl_options(&options);
  read_rpc_pcl_config(&options);
  on_syslog = options.log_to_syslog;
  quiet_flag = options.quiet;
  
  /*
   * test if the user is trusted or insecure is set.
   */
	
   if (options.insecure)
   {
     security_flag = 0;
   }
   else
   {
     for (i = 0; i < options.num_trusted_uid; i++)
     {
       if (uid == options.trusted_uid[i])
       {
         trusted = 1;
         security_flag = 0;
         break;
       }
     }
   }
  
  /*
   * process the arguments
   */
   
  if (strchr(argv[0], '/'))
    prog_name = strrchr(argv[0], '/') + 1;
  else
    prog_name = argv[0];

  while ((ch = getopt(argc, argv, "de:i:lmpqh:P:M:b:")) != EOF)
  {
    switch(ch)
    {
      case 'd':
          debug_flag = 1;
          break;
      case 'e':
          with_base_file = optarg;
          break;
      case 'b':
          base_dir = optarg;
          break;
      case 'i':
          dump_interval = atoi(optarg);
          break;
      case 'l':
          if (trusted)
            on_syslog = 0;
          break;
      case 'm':
          with_magic = 1;
          break;
      case 'p':
          ask_pm = 1;
          break;
      case 'q':
          if (trusted)
            quiet_flag = 1;
          break;
      case 'h':
          strncpy(sname,optarg,MAX_HOSTNAME);
          sname[MAX_HOSTNAME] = 0;
          has_host_name = 1;
          break;
      case 'P':
          local_port = atoi(optarg);
          break;
      case 'M':
          if(nmap >= MAX_FORWARD) {
            fprintf(stderr,
            "Number of mappings exceeds MAX_FORWARD (%d)",MAX_FORWARD);
            exit(1);
          }
          if (2 == sscanf(optarg, "%ld:%ld",&mapfrom[nmap],&mapto[nmap])) {
          nmap++;
          break;
          } /* Else fall through to default */
      case '?':
      default:
          usage(prog_name);
    }
  }
  argc -= optind;
  argv += optind;

  for(i=0; i < nmap; i++) {
    mapfrom[i]=htonl(mapfrom[i]);
    mapto[i]=htonl(mapto[i]);
  }
  
  if (argc) 
  {
    usage(prog_name);
  }
  
  /*
   * initialize the logging
   */
  log_init(prog_name, debug_flag, on_syslog, quiet_flag, LOG_DAEMON);

  /*
   * redirect stderr to debug file if required
   */
 
  if (with_base_file && debug_flag && !quiet_flag)
  {
      char file_with_base_dir [MAX_ERRFILE]; 
      if (trusted)
      {
        if (!base_dir)
          base_dir = xstrdup(DEFAULT_BASE_DIR);
	  }
      else
      {
        struct passwd * passwd_info;
        char user_base_dir [MAX_ERRFILE];
        char * user_dir = USER_DIR;
        struct stat stat_info;
        if (!(passwd_info = getpwuid (uid)))
          fatal ("Cannot get uid information for %d", uid);
        if (!passwd_info->pw_dir)
          fatal("No home dir for %d", uid);
        if (strlen(passwd_info->pw_dir) + strlen(user_dir) > MAX_ERRFILE)
          fatal ("Strange very long user home directory");
		
        snprintf(user_base_dir, MAX_ERRFILE, "%s/%s", passwd_info->pw_dir, user_dir);
        base_dir = xstrdup(user_base_dir);
        if (stat (base_dir, &stat_info) <0)
        {
          if (errno == ENOENT)
          {
            if (mkdir (base_dir, S_IRWXU) <0)
            {
              fprintf (stderr, "Cannot mkdir %s", base_dir); 
              base_dir = NULL;
            }
          }
          else 
            base_dir = NULL;
        }
		else if (! S_ISDIR(stat_info.st_mode))
          base_dir = NULL;
      }
      if (base_dir)
      {
        if (strlen(base_dir) + strlen(with_base_file) > MAX_ERRFILE)
          fatal ("Strange very long error file name");
        snprintf(file_with_base_dir, MAX_ERRFILE, "%s/%s", base_dir, with_base_file);
        err_file = xstrdup(file_with_base_dir);
      }
    if (!err_file || (freopen(err_file, "w", stderr) == NULL))
    {
      fprintf(stderr, "warning: can't open the debug file, using stderr\n");
    }
    setbuf(stderr, NULL);
  }

  /*
   * dump configuration
   */ 

  if (debug_flag)
  {
    u_int uid_idx;
    debug ("Real uid: %d, effective uid: %d, Trusted uid:", uid, file_uid);
    for (uid_idx = 0; uid_idx < options.num_trusted_uid; uid_idx++)
    {
      debug("%u", options.trusted_uid[uid_idx]);
    }
    dump_forward(options.accepted_forward, options.num_acc_fw);
  }
  
  /*
   * resolve localhost
   */
  
  if(gethostname(name,MAX_HOSTNAME) != 0 ||
     (hp = gethostbyname(name)) == NULL)
  {
    fatal("can't resolve localhost");
  }
  debug("localhost resolved");

  bzero((char *) &local_addr, sizeof(local_addr));
  local_addr.sin_family = AF_INET;
  bcopy((char *)hp->h_addr,(char *)&local_addr.sin_addr, hp->h_length);

  /*
   * resolve rpc server host, if desired
   */
  
  if(!has_host_name) strncpy(sname,name,MAX_HOSTNAME);

  if((hp = gethostbyname(sname)) == NULL)
    fatal("can't resolve rpc server host");
  
  debug("rpc server host resolved");

  /*
   * initialize the server address (without port)
   */
  bzero((char *) &serv_addr, sizeof(serv_addr));
  serv_addr.sin_family = AF_INET;
  bcopy((char *)hp->h_addr,(char *)&serv_addr.sin_addr, hp->h_length);

  /*
   * create our local socket and bind it
   */
  if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) 
  {
    fatal("can't open datagram socket for client");
  }

  if(local_port != -1) {
    local_addr.sin_port = htons(local_port);
    if (bind(sockfd, (struct sockaddr *) &local_addr, sizeof(local_addr)) < 0) 
      fatal("can't bind local address");
  } else {
    local_addr.sin_port = 0;
    /* Get possibly higher priviledges */
#ifdef _POSIX_SAVED_IDS
    setid_status = seteuid(file_uid);
#else
    setid_status = setreuid(geteuid(), getuid());
#endif
    if (setid_status <0)
    {
      fatal("seteuid (%d): %s", file_uid, strerror(errno));
    }
    if (bindresvport(sockfd, &local_addr) < 0)
    {
      /* release all priviledges */
#ifdef _POSIX_SAVED_IDS
      setid_status = setuid(uid);
#else
      setid_status = setreuid(geteuid(), getuid());
#endif
      if (setid_status <0)
      {
        fatal("seteuid (%d): %s", uid, strerror(errno));
      }
      fatal("can't bind local address with privileged port");
    }
#ifndef _POSIX_SAVED_IDS
    if (setreuid(geteuid(), getuid()) <0)
    {
      fatal("setreuid: %s", strerror(errno));
    }		
#endif
  }
  
  /*
   * release all priviledges
   */

#ifdef _POSIX_SAVED_IDS
  if (setuid(uid) < 0)
  {
    fprintf(stderr,"setuid (%d): %s", uid, strerror(errno));
    exit (1);
  }
  debug ("Changed uid to: %d", getuid());
#endif
  debug ("euid: %d, uid: %d", geteuid(), getuid());

   
   if (!security_flag)
   {
     if (trusted)
       debug ("No security check, trusted user %d", uid);
     else
     {
       debug ("No security check, insecure is set");
     }
   }

  /* 
   * read the binding back 
   */
   
  addr_len = sizeof(local_addr);
  getsockname(sockfd, (struct sockaddr *) &local_addr, &addr_len);

  debug("socket bound to address %s, port %d", 
        inet_ntoa(local_addr.sin_addr), ntohs(local_addr.sin_port));

  /*
   * write the ID string
   */
  printf("%s", ID_STRING);
  debug("ID string sent");

  /*
   * by default: dump the portmapper
   */
  if (!ask_pm)
  {
    pmap_dump(&serv_addr);
    signal(SIGALRM, dump_handler);
    alarm(dump_interval);
    debug("dump_handler installed, interval: %d secs", dump_interval);
  }

  
  /* chdir to / in case the cwd has to be unmounted */
  if (chdir("/") < 0)
    error("rpc forwarder cannot chdir to /: %s",  strerror(errno));
  
  /*
   * start the game
   */
  handle_forwarding(sockfd, with_magic, ask_pm);
  return 0;
}
