/*
 * map_xid.c
 *
 * Mapping RPC XIDs to a unique new one and vice versa.
 *
 * 10 Feb 1996, hot@informatik.tu-chemnitz.de
 */

#include "log-server.h"
#include "map_xid.h"
#include "xmalloc.h"

/*
 * derive HSIZE and HMASK from HASH_ENTRIES
 */

#define HSIZE (HASH_ENTRIES & ~1)
#define HMASK (HSIZE-1)

/*
 * Hash table for mappings
 */

typedef struct hash_entry
{
  map_item *mi;             /* pointer to the mapping */
  struct hash_entry *next;  /* pointer to the next element */
} hash_entry; 

hash_entry *ht_forw[HSIZE], /* hash table for mapping forward */
           *ht_back[HSIZE]; /* hash table for mapping back */

/*
 * the mappings themselves (circular list)
 */

map_item map_list;          /* static anchor of the list (dummy node) */

/*
 * current unique XID 
 */

static u_int current_XID = 0;

/*
 * maximum reference count
 */
static u_int max_ref_count;

/*
 * our (simple) hash function
 */

#define hash(x) ((u_int)((x)&HMASK))

/*
 * initialize the mapping mechanism
 */

void 
map_init(int max_rc)
{
  int i;

  max_ref_count = max_rc;
  map_list.next = map_list.prev = &map_list;

  for (i = 0; i < HSIZE; i++)
  {
    ht_forw[i] = NULL;
    ht_back[i] = NULL;
  }
}

/* 
 * search for an existing specific mapping
 */

map_item *
find_XID_mapping(u_int xid, struct sockaddr_in *socket_addr)
{
  map_item *iptr;
  hash_entry *he;

  for (he = ht_forw[hash(xid)]; he; he = he->next) 
  {
    iptr = he->mi;
    if ((iptr->orig_XID == xid) &&
        (iptr->socket_addr.sin_addr.s_addr == socket_addr->sin_addr.s_addr) &&
        (iptr->socket_addr.sin_port == socket_addr->sin_port))
        /*
         * note: the byte order in the sockaddr_in structure doesn't matter 
         *       because it is the same for both operands and we only test 
         *       for equality
         */
    {
      return(iptr);
    }
  }
  return(NULL);
}

/* 
 * add a new entry to a hash table
 */

static void
add_to_hash_table(hash_entry *ht[], int xid, map_item *item)
{
  hash_entry *he, *pred;
  u_int hval;

  /* is it already there? */
  pred = NULL;
  for (he = ht[hval = hash(xid)]; he; he = (pred = he)->next)
  {
    if (he->mi == item)
    {
      return; /* yes, it's there */
    }
  }

  if (pred)
  {
    he = pred->next = xmalloc(sizeof(hash_entry));
  }
  else
  {
    he = ht[hval] = xmalloc(sizeof(hash_entry));
  }

  he->next = NULL;
  he->mi = item;
}

/*
 * delete a specific entry from a hash table
 */

static void
delete_from_hash_table(hash_entry *ht[], int xid, map_item *item)
{
  hash_entry *he, *pred;
  u_int hval;

  pred = NULL;
  for (he = ht[hval = hash(xid)]; he; he = (pred = he)->next)
  {
    if (he->mi == item)
    {
      if (pred)
      {
        pred->next = he->next;
      }
      else
      {
        ht[hval] = he->next; 
      }
      xfree(he);
      return;
    }
  }
}

/*
 * append a new mapping at the tail of the list 
 */

static void
append_map_item(map_item *item)
{
  map_item *ip;

  (item->next = (ip = map_list.prev)->next)->prev = item;
  (ip->next = item)->prev = ip;
}

/*
 * delete a mapping from the list 
 */

static void
delete_map_item(map_item *item)
{
  (item->prev->next = item->next)->prev = item->prev;
  xfree(item);
}

/*
 * map an original XID to a unique one
 */

int 
map_XID(u_char *buffer, struct sockaddr_in *socket_addr)
{
  u_int xid;
  map_item *iptr, *new_item;

  xid = * (u_int *) buffer; /* byte order doesn't matter,
                              * it's only a bitstring for us
                              */

  if ((iptr = find_XID_mapping(xid, socket_addr))) 
  {
    /* this mapping already exists */
    if (iptr->ref_count >= max_ref_count)
    {
      debug("max_ref_count reached, not reusing 0x%08x ==> 0x%08x",
            xid, iptr->unique_XID);
      return 0;
    }

    /* the existing mapping will be reused */
    debug("Reusing XID: 0x%08lx ==> 0x%08lx", xid, iptr->unique_XID);
    * (u_int *) buffer = iptr->unique_XID;
    iptr->ref_count++;
    iptr->last_ref = time(NULL);
  }
  else 
  {
    /* a new mapping must be created */
    debug("mapping XID: 0x%08x ==> 0x%08x", xid, current_XID); 
    * (u_int *) buffer = current_XID;

    new_item = xmalloc(sizeof(map_item));

    new_item->socket_addr = *socket_addr;
    new_item->orig_XID = xid;
    new_item->unique_XID = current_XID++;
    new_item->ref_count = 1;
    new_item->last_ref = time(NULL);

    /* add the mapping to the list and the hash tables */
    append_map_item(new_item);
    add_to_hash_table(ht_forw, xid, new_item);
    add_to_hash_table(ht_back, new_item->unique_XID, new_item);
  }
  return 1;
}

/*
 * map the unique XID back to the original one
 */

int
demap_XID(u_char *buffer, struct sockaddr_in *socket_addr)
{
    u_int xid;
    map_item *iptr;
    hash_entry *he;

    xid = * (u_int *) buffer; /* byte order doesn't matter */

    debug("Trying to demap XID 0x%08lx", xid); 

    for (he = ht_back[hash(xid)]; he; he = he->next) 
    {
      iptr = he->mi;
      if (iptr->unique_XID == xid) 
      {
        * (u_int *) buffer = iptr->orig_XID;
        debug("demapping XID: 0x%08x <== 0x%08x", iptr->orig_XID, xid); 

        *socket_addr = iptr->socket_addr;

        /* throw away this element if it is no longer referenced */
        if (!(--iptr->ref_count)) 
        {
          delete_from_hash_table(ht_forw, iptr->orig_XID, iptr);
          delete_from_hash_table(ht_back, xid, iptr);
          delete_map_item(iptr);
        }

        /* successful demapping */
        return 1;
      }
    }

    /* demapping failed */
    return 0;
}

/* 
 * reap stale mapping entries
 */

int 
grim_reap(int reap_interval)
{
    time_t curr_time;
    map_item *iptr, *next;
    int checked = 0, reaped = 0;

    curr_time = time(NULL);

    for (iptr = map_list.next; iptr != &map_list; iptr = next)
    {
      checked++;
      next = iptr->next;
      if (curr_time - iptr->last_ref > reap_interval)
      {
        delete_from_hash_table(ht_forw, iptr->orig_XID, iptr);
        delete_from_hash_table(ht_back, iptr->unique_XID, iptr);
        delete_map_item(iptr);
        reaped++;
      }
    }
    debug("grim_reap checked %d mappings and reaped %d mappings", 
          checked, reaped);
    return reaped;
}

#ifdef MAP_TEST

/*
 * some (primitive) test stuff
 */

#include <string.h>
#include <syslog.h>

/* print a mapping */
static void
print_map_item(map_item *iptr)
{
  printf("oxid=0x%08lx, uxid=0x%08lx, addr=%s, port=%d, rc=%d, lr=%s", 
  iptr->orig_XID,
  iptr->unique_XID,
  inet_ntoa(iptr->socket_addr.sin_addr),
  iptr->socket_addr.sin_port,
  iptr->ref_count, ctime(&iptr->last_ref));
}

/* print the entire list */
static void
print_map_list(void)
{
  map_item *iptr;

  for (iptr = map_list.next; iptr != &map_list; iptr = iptr->next)
  {
    print_map_item(iptr);
  }
}

/* print a hash table */
static void
print_hash_table(hash_entry *ht[])
{
  struct hash_entry *he;
  int i;

  for (i = 0; i < HSIZE; i++)
  {
    if (ht[i])
    {
      printf("slot %d:\n", i);
      for (he = ht[i]; he; he = he->next)
      {
        print_map_item(he->mi);         
      }
    }
  }
}

main(int argc, char **argv)
{
  char line[100], *cp;
  u_int buffer;
  struct sockaddr_in si;
  int interval;
  map_item *iptr;
  time_t curr_time;

  setbuf(stdout, NULL);
  printf("map test\n");

  log_init("maptest", 
           1, /* on_stderr */ 
           1, /* on_syslog */ 
           1, /* debug */
           0, /* quiet */
           LOG_DAEMON);

  /* take the first command line argument as maximum reference counter
   *
   * default: 1
   */
  map_init(argc > 1 ? atoi(argv[1]) : 1);

  while(1)
  {
    printf("> ");
    if (!fgets(line, sizeof(line), stdin))
    {
      printf("\n");
      return 0;
    }
    line[strlen(line)-1] = 0;

    cp = line;
    while (isspace(*cp))
    {
      cp++;
    }

    if (!*cp)
    {
      continue;
    }

    if (!strncmp(cp, "map", 3))
    {
      if (3 != sscanf(cp, "%*s %lu %lu %hu", &buffer, &si.sin_addr.s_addr, 
                       &si.sin_port))
      {
        printf("bad args\n");
        continue;
      }
      map_XID((u_char *)&buffer, &si);
      printf("mapped to %u\n", buffer);
    }
    else
    if (!strncmp(cp, "demap", 5))
    {
      if (1 != sscanf(cp, "%*s %lu", &buffer))
      {
        printf("bad args\n");
        continue;
      }
      if (demap_XID((u_char *)&buffer, &si))
      {
        printf("xid=%lu, addr=%lu, port=%u\n", buffer, si.sin_addr.s_addr, 
               si.sin_port);
      }
      else
      {
        printf("failed\n");
      }
    }
    else
    if (!strncmp(cp, "find", 4))
    {
      if (3 != sscanf(cp, "%*s %lu %lu %hu", &buffer, &si.sin_addr.s_addr, 
                      &si.sin_port))
      {
        printf("bad args\n");
        continue;
      }
      iptr = find_XID_mapping(buffer, &si);
      if (iptr)
      {
        printf("xid=%lu, addr=%lu, port=%u\n", 
               iptr->orig_XID,
               iptr->socket_addr.sin_addr.s_addr,
               iptr->socket_addr.sin_port);
      }
      else
      {
        printf("failed\n");
      }
    }
    else
    if (!strncmp(cp, "print", 5))
    {
      print_map_list();
    }
    else
    if (!strncmp(cp, "reap", 4))
    {
      if (1 != sscanf(cp, "%*s %d", &interval))
      {
        printf("bad args\n");
        continue;
      }
      printf("%d entries reaped\n", grim_reap(interval));
    }
    else
    if (!strncmp(cp, "hash", 4))
    {
      printf("ht_forw:\n");
      print_hash_table(ht_forw);
      printf("ht_back:\n");
      print_hash_table(ht_back);
    }
    else
    if (!strncmp(cp, "date", 4))
    {
      curr_time = time(NULL);
      printf("%s", ctime(&curr_time));
    }
    else
    {
      printf("Available commands:\n");
      printf("map   xid addr port\n");
      printf("demap xid\n");
      printf("find  xid addr port\n");
      printf("print\n");
      printf("reap  interval\n");
      printf("hash\n");
      printf("date\n");
      printf("\n");
    }
  }
}

#endif
