/*
 * log-server.c
 *
 * Server-side versions of debug(), log(), etc.  
 * These send the output to stderr and/or the system log.
 *
 * Taken from Tatu Ylonen's SSH 1.2.12 and slightly adapted.

 Covered by ssh-1.2.12-COPYING

 *
 * 16 Jan 1996, hot@informatik.tu-chemnitz.de
 */

#include <stdio.h>
#include <syslog.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include "log-server.h"
#include "xmalloc.h"

static int log_debug = 0;
static int log_quiet = 0;
static int log_on_syslog = 0;

/* Initialize the log.
 *   av0          program name (should be argv[0])
 *   debug        send debugging on stderr
 *   quiet        don\'t log anything
 */

void log_init(char *av0, int debug, int on_syslog,  
              int quiet, int facility)
{
  log_debug = debug;
  log_quiet = quiet;
  log_on_syslog = on_syslog;
  if (quiet || !log_on_syslog)
    return ;
  closelog(); /* Close any previous log. */
  openlog(av0, LOG_PID, facility);
}

/* Log this message (information that usually should go to the log). */

void log(const char *fmt, ...)
{
  va_list args;

  if (log_quiet)
    return;
  if (log_debug) {
      va_start(args, fmt);
	  fprintf(stderr, "log: ");
	  vfprintf(stderr,fmt,args);
	  fprintf(stderr, "\n");
	  va_end(args);
  }
  if (log_on_syslog)
  {
      char buffer[BUFSIZ];
      va_start(args, fmt);
      vsnprintf(buffer,BUFSIZ,fmt,args);
	  va_end(args);
	  syslog(LOG_INFO,"log: %.500s", buffer);
  }
}

/* Debugging messages that should not be logged during normal operation. */

void debug(const char *fmt, ...)
{
  va_list args;
  struct timeval tv;
  struct timezone tz;
  char buffer[BUFSIZ];
  
  if (!log_debug || log_quiet)
    return;

  va_start(args, fmt);
  vsnprintf(buffer,BUFSIZ,fmt,args);
  va_end(args);
  gettimeofday(&tv,&tz);
  fprintf(stderr, "debug: %ld:%ld ",
          tv.tv_sec, tv.tv_usec);
  fprintf(stderr, "%.500s\n", buffer);
  if (log_on_syslog) syslog(LOG_DEBUG, "debug: %.500s", buffer);
}

/* Error messages that should be logged. */

void error(const char *fmt, ...)
{
  va_list args;
  char buffer[BUFSIZ];
  
  if (log_quiet)
    return;
  va_start(args, fmt);
  vsnprintf(buffer,BUFSIZ,fmt,args);
  va_end(args);
  
  if (log_debug) {
    fprintf(stderr, "error: %.500s\n",buffer);
  }
  
  if (log_on_syslog) syslog(LOG_ERR,"error: %.500s", buffer);
}

struct fatal_cleanup
{
  struct fatal_cleanup *next;
  void (*proc)(void *);
  void *context;
};

static struct fatal_cleanup *fatal_cleanups = NULL;

/* Registers a cleanup function to be called by fatal() before exiting. */

void fatal_add_cleanup(void (*proc)(void *), void *context)
{
  struct fatal_cleanup *cu;

  cu = xmalloc(sizeof(*cu));
  cu->proc = proc;
  cu->context = context;
  cu->next = fatal_cleanups;
  fatal_cleanups = cu;
}

/* Removes a cleanup frunction to be called at fatal(). */

void fatal_remove_cleanup(void (*proc)(void *context), void *context)
{
  struct fatal_cleanup **cup, *cu;
  
  for (cup = &fatal_cleanups; *cup; cup = &cu->next)
  {
    cu = *cup;
    if (cu->proc == proc && cu->context == context)
    {
      *cup = cu->next;
      xfree(cu);
      return;
    }
  }
  fatal("fatal_remove_cleanup: no such cleanup function: 0x%lx 0x%lx\n",
    (unsigned long)proc, (unsigned long)context);
}

/* Fatal messages.  This function never returns. */

void fatal(const char *fmt, ...)
{
  va_list args;
  struct fatal_cleanup *cu, *next_cu;
  static int fatal_called = 0;
  char buffer[BUFSIZ];

  if (!log_quiet)
  {
    va_start(args, fmt);
    vsnprintf(buffer,BUFSIZ,fmt,args);
    va_end(args);

    if (log_debug)
      fprintf(stderr, "fatal: %s\n", buffer);
    if (log_on_syslog)
      syslog(LOG_ERR, "fatal %.500s", buffer);
  }
  
  if (fatal_called)
    exit(1);
  fatal_called = 1;

  /* Call cleanup functions. */
  for (cu = fatal_cleanups; cu; cu = next_cu)
  {
    next_cu = cu->next;
    debug("Calling cleanup 0x%lx(0x%lx)",
      (unsigned long)cu->proc, (unsigned long)cu->context);
    (*cu->proc)(cu->context);
  }

  exit(1);
}
