/*
 * clntconf.c
 *
 * Handling rpc_pcl configuration options.
 *
 * Based on code from Tatu Ylonen's SSH 1.2.12 and servconf.c from
 * Holger Trapp.

 Covered by ssh-1.2.12-COPYING

 *
 * 2 Aug 2002, dumas@centre-cired.fr
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/param.h>
#include <errno.h>

#include "servconf.h"
#include "xmalloc.h"

#ifndef RPC_PCL_CONFIG_FILE
#define RPC_PCL_CONFIG_FILE "/etc/rpc_pcl.conf"
#endif

#if !defined(HAVE_STRSEP)
char *strsep(char **stringp, const char *delim);
#endif

/* Initializes the server options to their default values. */

void fill_default_rpc_pcl_options(ClntOptions *options)
{
  memset(options, 0, sizeof(*options));
  options->num_trusted_uid = 0;
  options->num_acc_fw = 0;
  options->insecure = 0;
  options->show_msg = 0;
  options->log_to_syslog = 1;
  options->quiet = 0;
}

#define WHITESPACE " \t\r\n"

/* Keyword tokens. */
typedef enum 
{
  sTrustedId, sForward, sLogToSyslog, sQuiet, sShowMsg
} ClntOpCodes;

/* Textual representation of the tokens. */
static struct
{
  const char *name;
  ClntOpCodes opcode;
} keywords[] =
{
  { "TrustedId", sTrustedId },
  { "Forward", sForward },
  { "LogToSyslog", sLogToSyslog },
  { "Quiet", sQuiet },
  { "ShowMsg", sShowMsg },
  { NULL, 0 }
};

/* Returns the number of the token pointed to by cp of length len.
   Never returns if the token is not known. */

static ClntOpCodes parse_clnt_token(const char *cp, const char *filename,
                 int linenum)
{
  unsigned int i;

  for (i = 0; keywords[i].name; i++)
    if (strcmp(cp, keywords[i].name) == 0)
      return keywords[i].opcode;

  fprintf(stderr, "%s line %d: Bad configuration option: %s\n", 
      filename, linenum, cp);
  exit(1);
}

/* Read the rpc_pcl configuration file. */

void read_rpc_pcl_config(ClntOptions *options)
{
  FILE *f;
  char line[1024];
  char *cp;
  u_int *intptr;
  int linenum,i;
  ClntOpCodes opcode;
#ifndef NI_MAXHOST
#define NI_MAXHOST MAXHOSTNAMELEN
#endif  
  char word[MAXHOSTNAMELEN];
  char *filename = xstrdup(RPC_PCL_CONFIG_FILE);
  char dummy;

  f = fopen(filename, "r");
  if (!f)
  {
    fprintf(stderr,"Cannot open %s: %s\n",filename,strerror(errno));
	exit (1);
  }

  linenum = 0;
  
  while (fgets(line, sizeof(line), f))
  {
    linenum++;
    cp = line + strspn(line, WHITESPACE);
    if (!*cp || *cp == '#')
    {
      continue;
    }
    cp = strtok(cp, WHITESPACE);
    opcode = parse_clnt_token(cp, filename, linenum);
    switch (opcode)
    {
      case sLogToSyslog:
        intptr = &options->log_to_syslog;
      parse_flag:
        cp = strtok(NULL, WHITESPACE);
        if (!cp)
        {
          fprintf(stderr, "%s line %d: missing yes/no argument.\n",
                  filename, linenum);
          exit(1);
        }
        if (strcmp(cp, "yes") == 0)
          *intptr = 1;
        else
          if (strcmp(cp, "no") == 0)
            *intptr = 0;
          else
          {
            fprintf(stderr, "%s line %d: Bad yes/no argument: %s\n",
                    filename, linenum, cp);
          }
        break;	
      case sQuiet:
        intptr = &options->quiet;
        goto parse_flag;	

      case sShowMsg:
        intptr = &options->show_msg;
        goto parse_flag;	

      case sTrustedId:
        i = options->num_trusted_uid;
        cp = strtok(NULL, WHITESPACE);
        if (!cp)
        {
          fprintf(stderr, "%s line %d: argument missing.\n",
                 filename, linenum);
          exit(1);
        }
        while (cp && (i < MAX_TRUSTED))
        {
          struct passwd *passwd_entry;
          uid_t new_uid;
          int nb_tokens;
		  
          snprintf(word, MAXHOSTNAMELEN, "%s", cp);
		  if (strcmp(word, "*") == 0)
          {
            options->insecure = 1;
            break;
          }
          nb_tokens = sscanf(word, "%u%c", &new_uid, &dummy);
          if (nb_tokens == 1)
          {
            options->trusted_uid[i] = new_uid;
          }
          else if (nb_tokens != 0)
          {
            fprintf(stderr, "not an numeric uid: %s.\n", word);
            exit(1);
          }
          else
          {
            if ((passwd_entry = getpwnam(word)))
            {
              options->trusted_uid[i] = passwd_entry->pw_uid;
            }
            else
            {
              fprintf(stderr, "cannot find uid for %s.\n", word);
              exit (1);
            }
          }
          i++;
          cp = strtok(NULL, WHITESPACE);
        }
        options->num_trusted_uid = i;
        continue;

      case sForward:
        i = options->num_acc_fw;
        cp = strtok(NULL, WHITESPACE);
        if (!cp)
        {
          fprintf(stderr, "%s line %d: argument missing.\n",
                  filename, linenum);
          exit(1);
        }
        while (cp && (i < MAX_FORWARD))
        {
          char *current = cp;
          char *token;
          
          options->accepted_forward[i].flags = 0;
          options->accepted_forward[i].flags |= SEC_RPC_ACCEPTED;
          options->accepted_forward[i].num_uid = 1;
          options->accepted_forward[i].uid[0] = getuid();
          options->accepted_forward[i].num_gid = 1;
          options->accepted_forward[i].gid[0] = getgid();
          options->accepted_forward[i].old_port = 0;
	      
          token = strsep (&current, ",");
          if (!token || !current || !strlen(token) || (sscanf(token,"%ld%c", &options->accepted_forward[i].prog, &dummy) != 1))
          {
            fprintf(stderr, "%s line %d: not a programm number, version: %s.\n",
                    filename, linenum, token);
            exit(1);
          }
	  
          token = strsep (&current, ":");
          if (!token || !strlen(token) || (sscanf(token, "%ld%c", &options->accepted_forward[i].vers, &dummy) != 1))
          {
            fprintf(stderr, "%s line %d: not a version number: %s.\n",
                    filename, linenum, token);
            exit(1);
          }
          if (!current)
            goto next;
		  
          token = strsep (&current, ":");
          if (!token)
          {
            fprintf(stderr, "%s line %d: uid list ':' without following argument.\n",
                   filename, linenum);
            exit(1);
          }
          else if ((*token) == '-')
          {
            options->accepted_forward[i].flags &= ~SEC_RPC_ACCEPTED;
          }
          else if ((*token) == '*')
          {
            options->accepted_forward[i].flags |= SEC_RPC_ANY_UID;
          }
          else if (strlen (token))
            options->accepted_forward[i].num_uid += parse_uid_list(token, options->accepted_forward[i].uid + 1, options->accepted_forward[i].num_uid);

          if (!current)
            goto next;
          token = strsep (&current, ":");
          if (!token)
          {
            fprintf(stderr, "%s line %d: gid list ':' without following argument.\n",
                 filename, linenum);
            exit(1);
          }
	  else if ((*token) == '*')
          {
            options->accepted_forward[i].flags |= SEC_RPC_ANY_GID;
          }
          else if (strlen (token))
            options->accepted_forward[i].num_gid += parse_gid_list(token, options->accepted_forward[i].gid + 1, options->accepted_forward[i].num_gid);

        next:
          i++;
          cp = strtok(NULL, WHITESPACE);
        }
        options->num_acc_fw = i;
/*	exit (0);*/
        continue;

	  default:
        fprintf(stderr, "%s line %d: Missing handler for opcode %s (%d)\n",
                filename, linenum, cp, opcode);
        exit(1);
    }

    if (strtok(NULL, WHITESPACE) != NULL)
    {
      fprintf(stderr, "%s line %d: garbage at end of line.\n",
              filename, linenum);
      exit(1);
    }
  }
  fclose(f);
}
