/* gdg_dma.c -- DMA support for the GDG -*- linux-c -*-
 */
/**************************************************************************
 * 
 * Copyright 2003 Tungsten Graphics, Inc., Cedar Park, Texas.
 * All Rights Reserved.
 * 
 **************************************************************************/

#define __NO_VERSION__
#include "gdg.h"
#include "drmP.h"
#include "drm.h"
#include "gdg_drm.h"
#include "gdg_drv.h"


static void gdg_print_status_page(drm_device_t *dev)
{
      	drm_gdg_private_t *dev_priv = dev->dev_private;
	u32 *temp = dev_priv->hw_status_page;

	if (!temp) {
		DRM_DEBUG("no status page\n");
		return;
	}

   	DRM_DEBUG(  "hw_status: Interrupt Status : %x\n", temp[0]);
   	DRM_DEBUG(  "hw_status: LpRing Head ptr : %x\n", temp[1]);
   	DRM_DEBUG(  "hw_status: IRing Head ptr : %x\n", temp[2]);
      	DRM_DEBUG(  "hw_status: Reserved : %x\n", temp[3]);
   	DRM_DEBUG(  "hw_status: Driver Counter : %d\n", temp[5]);

}




/* Really want an OS-independent resettable timer.  Would like to have
 * this loop run for (eg) 3 sec, but have the timer reset every time
 * the head pointer changes, so that EBUSY only happens if the ring
 * actually stalls for (eg) 3 seconds.
 */
int gdg_wait_ring( drm_device_t *dev, int n, const char *caller )
{
   	drm_gdg_private_t *dev_priv = dev->dev_private;
   	drm_gdg_ring_buffer_t *ring = &(dev_priv->ring);
	u32 last_head = GDG_READ(LP_RING + RING_HEAD) & HEAD_ADDR;
	int i;

	for ( i = 0 ; i < 10000 ; i++ ) {
		ring->head = GDG_READ(LP_RING + RING_HEAD) & HEAD_ADDR;
	   	ring->space = ring->head - (ring->tail+8);
		if (ring->space < 0) ring->space += ring->Size;
		if ( ring->space >= n )
			return 0;
		
		dev_priv->sarea_priv->perf_boxes |= GDG_BOX_WAIT;

		if (ring->head != last_head) 
			i = 0;

		last_head = ring->head;
	}

	return DRM_ERR(EBUSY);
}

void gdg_kernel_lost_context(drm_device_t *dev)
{
      	drm_gdg_private_t *dev_priv = dev->dev_private;
   	drm_gdg_ring_buffer_t *ring = &(dev_priv->ring);
      
   	ring->head = GDG_READ(LP_RING + RING_HEAD) & HEAD_ADDR;
     	ring->tail = GDG_READ(LP_RING + RING_TAIL) & TAIL_ADDR;
     	ring->space = ring->head - (ring->tail+8);
     	if (ring->space < 0) ring->space += ring->Size;

	if (ring->head == ring->tail)
		dev_priv->sarea_priv->perf_boxes |= GDG_BOX_RING_EMPTY;
}


int gdg_cleanup(drm_device_t *dev)
{
	/* Make sure interrupts are disabled here because the uninstall ioctl
	 * may not have been called from userspace and after dev_private
	 * is freed, it's too late.
	 */
	if (dev->irq) DRM(irq_uninstall)(dev);

	if (dev->dev_private) {
	   	drm_gdg_private_t *dev_priv = 
	     		(drm_gdg_private_t *) dev->dev_private;
	   
	   	if (dev_priv->ring.virtual_start) {
		   	DRM_IOREMAPFREE( &dev_priv->ring.map, dev);
		}

	   	if (dev_priv->hw_status_page) {
			pci_free_consistent(dev->pdev, PAGE_SIZE,
				      dev_priv->hw_status_page,
				      dev_priv->dma_status_page);
		   	/* Need to rewrite hardware status page */
		   	GDG_WRITE(0x02080, 0x1ffff000);
		}

	   	DRM(free)(dev->dev_private, sizeof(drm_gdg_private_t), 
			 DRM_MEM_DRIVER);

	   	dev->dev_private = NULL;
	}

   	return 0;
}



static int gdg_initialize(drm_device_t *dev, 
			       drm_gdg_private_t *dev_priv,
			       drm_gdg_init_t *init)
{
   	memset(dev_priv, 0, sizeof(drm_gdg_private_t));

	DRM_GETSAREA();
	if(!dev_priv->sarea) {
		DRM_ERROR("can not find sarea!\n");
		dev->dev_private = (void *)dev_priv;
		gdg_cleanup(dev);
		return DRM_ERR(EINVAL);
	}

	DRM_FIND_MAP( dev_priv->mmio_map, init->mmio_offset );
	if(!dev_priv->mmio_map) {
		dev->dev_private = (void *)dev_priv;
		gdg_cleanup(dev);
		DRM_ERROR("can not find mmio map!\n");
		return DRM_ERR(EINVAL);
	}

	dev_priv->sarea_priv = (drm_gdg_sarea_t *)
		((u8 *)dev_priv->sarea->handle +
		 init->sarea_priv_offset);

   	dev_priv->ring.Start = init->ring_start;
   	dev_priv->ring.End = init->ring_end;
   	dev_priv->ring.Size = init->ring_size;
   	dev_priv->ring.tail_mask = dev_priv->ring.Size - 1;
		
	dev_priv->ring.map.offset = init->ring_start;
	dev_priv->ring.map.size = init->ring_size;
	dev_priv->ring.map.type = 0;
	dev_priv->ring.map.flags = 0;
	dev_priv->ring.map.mtrr = 0;

	DRM_IOREMAP( &dev_priv->ring.map, dev );

   	if (dev_priv->ring.map.handle == NULL) {
		dev->dev_private = (void *) dev_priv;
	   	gdg_cleanup(dev);
	   	DRM_ERROR("can not ioremap virtual address for"
			  " ring buffer\n");
	   	return DRM_ERR(ENOMEM);
	}

   	dev_priv->ring.virtual_start = dev_priv->ring.map.handle;
   
	dev_priv->back_offset = init->back_offset;
	dev_priv->front_offset = init->front_offset;
	dev_priv->current_page = 0;
	dev_priv->sarea_priv->pf_current_page = dev_priv->current_page;

	/* We are using separate values as placeholders for mechanisms for
	 * private backbuffer/depthbuffer usage.
	 */
	dev_priv->use_mi_batchbuffer_start = 0;

   	/* Program Hardware Status Page */
   	dev_priv->hw_status_page =
		pci_alloc_consistent( dev->pdev, PAGE_SIZE, 
						&dev_priv->dma_status_page );

   	if (!dev_priv->hw_status_page) {
		dev->dev_private = (void *)dev_priv;
		gdg_cleanup(dev);
		DRM_ERROR("Can not allocate hardware status page\n");
		return DRM_ERR(ENOMEM);
	}
   	memset(dev_priv->hw_status_page, 0, PAGE_SIZE);
	DRM_DEBUG("hw status page @ %p\n", dev_priv->hw_status_page);
   
   	GDG_WRITE(0x02080, dev_priv->dma_status_page);
	DRM_DEBUG("Enabled hardware status page\n");
   
	dev->dev_private = (void *)dev_priv;

   	return 0;
}

int gdg_dma_init( DRM_IOCTL_ARGS )
{
	DRM_DEVICE;
   	drm_gdg_private_t *dev_priv;
   	drm_gdg_init_t init;
   	int retcode = 0;
	
  	DRM_COPY_FROM_USER_IOCTL( init, (drm_gdg_init_t *)data, sizeof(init));
	
   	switch(init.func) {
	 	case GDG_INIT_DMA:
			dev_priv = DRM(alloc)(sizeof(drm_gdg_private_t), 
					      DRM_MEM_DRIVER);
	   		if(dev_priv == NULL) 
				return DRM_ERR(ENOMEM);
	   		retcode = gdg_initialize(dev, dev_priv, &init);
	   	break;
	 	case GDG_CLEANUP_DMA:
	   		retcode = gdg_cleanup(dev);
	   	break;
	 	default:
	   		retcode = -EINVAL;
	   	break;
	}
   
   	return retcode;
}



static int gdg_dispatch_batchbuffer(drm_device_t *dev, 
				    drm_gdg_batchbuffer_t *batch )
{
   	drm_gdg_private_t *dev_priv = dev->dev_private;
   	drm_clip_rect_t box, *boxes = batch->cliprects;
   	int nbox = batch->num_cliprects;
	int i = 0, count;
   	RING_LOCALS;

	if ((batch->start | batch->used) & 0x7) {
		DRM_ERROR("alignment");
		return DRM_ERR(EINVAL);
	}
		
   	gdg_kernel_lost_context(dev);

	count = nbox ? nbox : 1;

	for (i = 0 ; i < count ; i++) {
		if (i < nbox) {
			if (DRM_COPY_FROM_USER_UNCHECKED( &box, 
							  &boxes[i], 
							  sizeof(box) ))
				return EFAULT;

			if (box.y2 <= box.y1 ||
			    box.x2 <= box.x1 ||
			    box.y2 <= 0 ||
			    box.x2 <= 0) {
				DRM_ERROR("Bad box %d,%d..%d,%d\n",
					  box.x1, box.y1, box.x2, box.y2);
				return DRM_ERR(EINVAL);
			}
				

			BEGIN_LP_RING(6);
			OUT_RING( GFX_OP_DRAWRECT_INFO );
			OUT_RING( batch->DR1 );
			OUT_RING( (box.x1 & 0xffff) | (box.y1<<16) );
			OUT_RING( ((box.x2-1) & 0xffff) | ((box.y2-1)<<16) );
			OUT_RING( batch->DR4 );
			OUT_RING( 0 );
			ADVANCE_LP_RING();
		}


		if (dev_priv->use_mi_batchbuffer_start) {
			BEGIN_LP_RING(2);
			OUT_RING( MI_BATCH_BUFFER_START | (2<<6) );
			OUT_RING( batch->start | MI_BATCH_NON_SECURE );
			ADVANCE_LP_RING();
		} 
		else {
			BEGIN_LP_RING(4);
			OUT_RING( MI_BATCH_BUFFER );
			OUT_RING( batch->start | MI_BATCH_NON_SECURE );
			OUT_RING( batch->start + batch->used - 4 );
			OUT_RING( 0 );
			ADVANCE_LP_RING();
		}
	}

	
	dev_priv->sarea_priv->last_enqueue = dev_priv->counter++;

	BEGIN_LP_RING(4);
	OUT_RING( CMD_STORE_DWORD_IDX );
	OUT_RING( 20 );
	OUT_RING( dev_priv->counter );
	OUT_RING( 0 );
	ADVANCE_LP_RING();

	return 0;
}

static int gdg_dispatch_flip( drm_device_t *dev )
{
   	drm_gdg_private_t *dev_priv = dev->dev_private;
	RING_LOCALS;

	DRM_DEBUG( "%s: page=%d pfCurrentPage=%d\n", 
		   __FUNCTION__, 
		   dev_priv->current_page,
		   dev_priv->sarea_priv->pf_current_page);

  	gdg_kernel_lost_context(dev);


	BEGIN_LP_RING( 2 );
    	OUT_RING( INST_PARSER_CLIENT | INST_OP_FLUSH | INST_FLUSH_MAP_CACHE ); 
	OUT_RING( 0 );
	ADVANCE_LP_RING();

	BEGIN_LP_RING( 6 );
	OUT_RING( CMD_OP_DISPLAYBUFFER_INFO | ASYNC_FLIP );	
	OUT_RING( 0 );
	if ( dev_priv->current_page == 0 ) {
		OUT_RING( dev_priv->back_offset );
		dev_priv->current_page = 1;
	} else {
		OUT_RING( dev_priv->front_offset );
		dev_priv->current_page = 0;
	}
	OUT_RING(0);
	ADVANCE_LP_RING();


	BEGIN_LP_RING( 2 );
	OUT_RING( MI_WAIT_FOR_EVENT |
		  MI_WAIT_FOR_PLANE_A_FLIP );
	OUT_RING( 0 );
	ADVANCE_LP_RING();
	

	dev_priv->sarea_priv->last_enqueue = dev_priv->counter++;

	BEGIN_LP_RING(4);
	OUT_RING( CMD_STORE_DWORD_IDX );
	OUT_RING( 20 );
	OUT_RING( dev_priv->counter );
	OUT_RING( 0 );
	ADVANCE_LP_RING();

	dev_priv->sarea_priv->pf_current_page = dev_priv->current_page;
	return 0;
}


static int gdg_quiescent(drm_device_t *dev)
{
   	drm_gdg_private_t *dev_priv = dev->dev_private;

  	gdg_kernel_lost_context(dev);
	return gdg_wait_ring( dev, dev_priv->ring.Size - 8, __FUNCTION__ );
}


int gdg_flush_ioctl( DRM_IOCTL_ARGS )
{
	DRM_DEVICE;

   	if(!_DRM_LOCK_IS_HELD(dev->lock.hw_lock->lock)) {
		DRM_ERROR("gdg_flush_ioctl called without lock held\n");
		return DRM_ERR(EINVAL);
	}

    	return gdg_quiescent(dev); 
}

int gdg_batchbuffer( DRM_IOCTL_ARGS )
{
	DRM_DEVICE;
   	drm_gdg_private_t *dev_priv = (drm_gdg_private_t *)dev->dev_private;
      	u32 *hw_status = dev_priv->hw_status_page;
   	drm_gdg_sarea_t *sarea_priv = (drm_gdg_sarea_t *) 
     					dev_priv->sarea_priv; 
	drm_gdg_batchbuffer_t batch;
	int ret;

	DRM_COPY_FROM_USER_IOCTL( batch, (drm_gdg_batchbuffer_t *)data, 
				  sizeof(batch) );

	DRM_DEBUG("gdg batchbuffer, start %x used %d cliprects %d\n",
		  batch.start, batch.used, batch.num_cliprects);


   	if(!_DRM_LOCK_IS_HELD(dev->lock.hw_lock->lock)) {
		DRM_ERROR("gdg_batchbuffer called without lock held\n");
		return DRM_ERR(EINVAL);
	}

	if (batch.num_cliprects && DRM_VERIFYAREA_READ(batch.cliprects, 
						       batch.num_cliprects *
						       sizeof(drm_clip_rect_t)))
		return DRM_ERR(EFAULT);

	ret = gdg_dispatch_batchbuffer( dev, &batch );

   	sarea_priv->last_dispatch = (int) hw_status[5];   
	return ret;
}



int gdg_do_cleanup_pageflip( drm_device_t *dev )
{
	drm_gdg_private_t *dev_priv = dev->dev_private;

	DRM_DEBUG("%s\n", __FUNCTION__);
	if (dev_priv->current_page != 0)
		gdg_dispatch_flip( dev );

	return 0;
}

int gdg_flip_bufs( DRM_IOCTL_ARGS )
{
	DRM_DEVICE;

	DRM_DEBUG("%s\n", __FUNCTION__);
   	if(!_DRM_LOCK_IS_HELD(dev->lock.hw_lock->lock)) {
		DRM_ERROR("gdg_flip_buf called without lock held\n");
		return DRM_ERR(EINVAL);
	}

	return gdg_dispatch_flip( dev );
}



int gdg_getparam( DRM_IOCTL_ARGS )
{
	DRM_DEVICE;
	drm_gdg_private_t *dev_priv = dev->dev_private;
	drm_gdg_getparam_t param;
	int value;

	if ( !dev_priv ) {
		DRM_ERROR( "%s called with no initialization\n", __FUNCTION__ );
		return DRM_ERR(EINVAL);
	}

	DRM_COPY_FROM_USER_IOCTL(param, (drm_gdg_getparam_t *)data, 
				 sizeof(param));

	switch( param.param ) {
	case GDG_PARAM_IRQ_ACTIVE:
		value = dev->irq ? 1 : 0;
		break;
	default:
		return DRM_ERR(EINVAL);
	}

	if ( DRM_COPY_TO_USER( param.value, &value, sizeof(int) ) ) {
		return DRM_ERR(EFAULT);
	}
	
	return 0;
}


int gdg_setparam( DRM_IOCTL_ARGS )
{
	DRM_DEVICE;
	drm_gdg_private_t *dev_priv = dev->dev_private;
	drm_gdg_setparam_t param;

	if ( !dev_priv ) {
		DRM_ERROR( "%s called with no initialization\n", __FUNCTION__ );
		return DRM_ERR(EINVAL);
	}

	DRM_COPY_FROM_USER_IOCTL( param, (drm_gdg_setparam_t *)data, 
				  sizeof(param) );

	switch( param.param ) {
	case GDG_SETPARAM_USE_MI_BATCHBUFFER_START:
		dev_priv->use_mi_batchbuffer_start = param.value;
		break;
	case GDG_SETPARAM_TEX_LRU_LOG_GRANULARITY:
		dev_priv->tex_lru_log_granularity = param.value;
		break;
	default:
		DRM_ERROR("unknown parameter %d\n", param.param);
		return DRM_ERR(EINVAL);
	}


	return 0;
}
