/*
 * segy.h - header file for SEGY routines
 *
 * This file is part of Automaton.
 *
 * Copyright (C) 2002, 2003
 * Paul Gettings, Dep't of Geology & Geophysics
 * University of Utah
 *
 * This file is released under the terms of the software
 * license in the file "LICENSE" in the root directory of
 * this package.  If this file is missing or corrupt, please
 * contact the author to receive a new copy.
 *
 * Automaton is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  Use at your
 * own risk; your mileage may vary.
 *
 * Suggestions, improvements, and bug reports welcome at
 * <gettings@mines.utah.edu>
 */
#ifndef SEGY_UUTAH
#define SEGY_UUTAH
#include <stdio.h>
#include "autopick.h"
/*
 * SEGY format, from UTAM webpage
 ********************************
 Offset
  0-3	long tracl;	trace sequence number within line
  4-7	long tracr;	trace sequence number within reel
  8-11	long fldr;	field record number 
 12-15	long tracf;	trace number within field record 
 16-19	long ep;	energy source point number 
 20-23	long cdp;	CDP ensemble number 
 24-27	long cdpt;	trace number within CDP ensemble 
 28-29	short trid;	trace identification code:
 			1 = seismic data
 			2 = dead
 			3 = dummy
 			4 = time break
 			5 = uphole
 			6 = sweep
 			7 = timing
 			8 = water break
 			9---, N = optional use (N = 32,767)
 			Following are CWP id flags:
 			 9 = autocorrelation
 			10 = Fourier transformed - no packing
 			     xr[0],xi[0], ..., xr[N-1],xi[N-1]
 			11 = Fourier transformed - unpacked Nyquist
 			     xr[0],xi[0],...,xr[N/2],xi[N/2]
 			12 = Fourier transformed - packed Nyquist
 	 		     even N:
 			     xr[0],xr[N/2],xr[1],xi[1], ...,
 				xr[N/2 -1],xi[N/2 -1]
 				(note the exceptional second entry)
 			     odd N:
 			     xr[0],xr[(N-1)/2],xr[1],xi[1], ...,
 				xr[(N-1)/2 -1],xi[(N-1)/2 -1],xi[(N-1)/2]
 				(note the exceptional second & last entries)
 			13 = Complex signal in the time domain
 			     xr[0],xi[0], ..., xr[N-1],xi[N-1]
 			14 = Fourier transformed - amplitude/phase
 			     a[0],p[0], ..., a[N-1],p[N-1]
 			15 = Complex time signal - amplitude/phase
 			     a[0],p[0], ..., a[N-1],p[N-1]
 			16 = Real part of complex trace from 0 to Nyquist
 			17 = Imag part of complex trace from 0 to Nyquist
 			18 = Amplitude of complex trace from 0 to Nyquist
 			19 = Phase of complex trace from 0 to Nyquist
 			21 = Wavenumber time domain (k-t)
 			22 = Wavenumber frequency (k-omega)
 			30 = Depth-Range (z-x) traces
 			101 = Seismic data packed to bytes (by supack1)
 			102 = Seismic data packed to 2 bytes (by supack2)
 30-31	short nvs;	 number of vertically summed traces (see vscode
 			   in bhed structure) 
 32-33	short nhs;	 number of horizontally summed traces (see vscode
 			   in bhed structure) 
 34-35	short duse;	 data use:
 				1 = production
 				2 = test 
 36-39	long offset;	 distance from source point to receiver
 			   group (negative if opposite to direction
 			   in which the line was shot) 
 40-43	long gelev;	 receiver group elevation from sea level
 			   (above sea level is positive) 
 44-47	long selev;	 source elevation from sea level
 			   (above sea level is positive) 
 48-51	long sdepth;	 source depth (positive) 
 52-55	long gdel;	 datum elevation at receiver group 
 56-59	long sdel;	 datum elevation at source 
 60-63	long swdep;	 water depth at source 
 64-67	long gwdep;	 water depth at receiver group 
 68-69	short scalel;	 scale factor for previous 7 entries
 			   with value plus or minus 10 to the
 			   power 0, 1, 2, 3, or 4 (if positive,
 			   multiply, if negative divide) 
 70-71	short scalco;	 scale factor for next 4 entries
 			   with value plus or minus 10 to the
 			   power 0, 1, 2, 3, or 4 (if positive,
 			   multiply, if negative divide) 
 72-75	long  sx;	 X source coordinate 
 76-79	long  sy;	 Y source coordinate 
 80-83	long  gx;	 X group coordinate 
 84-87	long  gy;	 Y source coordinate 
 88-89	short counit;	 coordinate units code:
 				for previoius four entries
 				1 = length (meters or feet)
 				2 = seconds of arc (in this case, the
 				X values are longitude and the Y values
 				are latitude, a positive value designates
 				the number of seconds east of Greenwich
 				or north of the equator 
 90-91	short wevel;	 weathering velocity 
 92-93	short swevel;	 subweathering velocity 
 94-95	short sut;	 uphole time at source 
 96-97	short gut;	 uphole time at receiver group 
 98-99 	short sstat;	 source static correction 
100-101  short gstat;	 group static correction 
102-103  short tstat;	 total static applied 
104-105  short laga;	 lag time A, time in ms between end of 240-
 			   byte trace identification header and time
 			   break, positive if time break occurs after
 			   end of header, time break is defined as
 			   the initiation pulse which maybe recorded
 			   on an auxiliary trace or as otherwise
 			   specified by the recording system 
106-107  short lagb;	 lag time B, time in ms between the time break
 		   and the initiation time of the energy source,
 		   may be positive or negative 
108-109  short delrt;	 delay recording time, time in ms between
 			   initiation time of energy source and time
 			   when recording of data samples begins
 			   (for deep water work if recording does not
 			   start at zero time) 
110-111  short muts;	 mute time--start 
112-113  short mute;	 mute time--end 
114-115  unsigned short ns;	 number of samples in this trace 
116-117  unsigned short dt;	 sample interval; in micro-seconds 
118-119  short gain;	 gain type of field instruments code:
 				1 = fixed
 				2 = binary
 				3 = floating point
 				4 ---- N = optional use 
120-121  short igc;	 instrument gain constant 
122-123  short igi;	 instrument early or initial gain 
124-125  short corr;	 correlated:
 				1 = no
 				2 = yes 
126-127  short sfs;	 sweep frequency at start 
128-129  short sfe;	 sweep frequency at end 
130-131  short slen;	 sweep length in ms 
132-133  short styp;	 sweep type code:
 				1 = linear
 				2 = cos-squared
 				3 = other 
134-135  short stas;	 sweep trace length at start in ms 
136-137  short stae;	 sweep trace length at end in ms 
138-139  short tatyp;	 taper type: 1=linear, 2=cos^2, 3=other 
140-141  short afilf;	 alias filter frequency if used 
142-143  short afils;	 alias filter slope 
144-145  short nofilf;	 notch filter frequency if used 
146-147  short nofils;	 notch filter slope 
148-149  short lcf;	 low cut frequency if used 
150-151  short hcf;	 high cut frequncy if used 
152-153  short lcs;	 low cut slope 
154-155  short hcs;	 high cut slope 
156-157  short year;	 year data recorded 
158-159  short day;	 day of year 
160-161  short hour;	 hour of day (24 hour clock) 
162-163  short minute;	 minute of hour 
164-165  short sec;	 second of minute 
166-167  short timbas;	 time basis code:
 				1 = local
 				2 = GMT
 				3 = other 
168-169  short trwf;	 trace weighting factor, defined as 1/2^N
 			   volts for the least sigificant bit 
170-171  short grnors;	 geophone group number of roll switch
 			   position one 
172-173  short grnofr;	 geophone group number of trace one within
 			   original field record 
174-175  short grnlof;	 geophone group number of last trace within
 			   original field record 
176-177   short gaps;	 gap size (total number of groups dropped) 
178-179   short otrav;	 overtravel taper code:
 				1 = down (or behind)
 				2 = up (or ahead) 
 	 local assignments 
180-183  float d1;	 sample spacing for non-seismic data 
184-187  float f1;	 first sample location for non-seismic data 
188-191  float d2;	 sample spacing between traces 
192-195  float f2;	 first trace location 
196-199  float ungpow;	 negative of power used for dynamic
 	   range compression 
200-203  float unscale;	 reciprocal of scaling factor to normalize
 	   range 
204-205  short mark;	 mark selected traces 
206-239  short unass[17];	 unassigned--NOTE: last entry causes 
 			   a break in the word alignment, if we REALLY
 			   want to maintain 240 bytes, the following
 			   entry should be an odd number of short/UINT2
 			   OR do the insertion above the "mark" keyword
 			   entry 
*/
/* Structure for SEGY header information
 *
 * Assume that long ints are >= 32 bits, short ints are >= 16 bits
 */
typedef struct segyHeaderStruct {
  long tracl, tracr, fldr, tracf, ep, cdp, cdpt;
  short trid, nvs, nhs, duse;
  long offset, gelev, selev, sdepth, gdel, sdel, swdep, gwdep;
  short scalel, scalco;
  long sx, sy, gx, gy;
  short counit, wevel, swevel, sut, gut, sstat, gstat, tstat, laga, lagb, delrt, muts, mute;
  unsigned short ns, dt;
  short gain, igc, igi, corr, sfs, sfe, slen, styp, stas, stae, tatyp;
  short afilf, afils, nofilf, nofils, lcf, hcf, lcs, hcs, year, day, hour, minute, sec;
  short timbas, trwf, grnors, grnofr, grnlof, gaps, otrav;
  float d1, f1, d2, f2, ungpow, unscale;
  short mark;
} SEGYHeader;

/* Functions
 */
int checkSEGY(FILE *fp, bool swapByte, bool reelHeader, long *nt, long *ns);
int readSEGYSingleShot(FILE *fp, long nt, seismicData *data, bool swapByte, bool reelHeader);
seismicData *readSEGYMultiShot(FILE *fp, long ns, bool swapByte, bool reelHeader);
SEGYHeader grokHeader(unsigned char buf[240], bool swap);
long headerLong(unsigned char *buf, bool swap);
float headerPosition(unsigned char *buf, bool swap, short scale);
float headerFloat(unsigned char *buf, bool swap);
short headerShort(unsigned char *buf, bool swap);
unsigned char *swap4bytes(unsigned char *p);

#endif
