/*
 * segy.cpp - SEGY file utilities and routines
 *
 * This file is part of Automaton.
 *
 * Copyright (C) 2002, 2003
 * Paul Gettings, Dep't of Geology & Geophysics
 * University of Utah
 *
 * This file is released under the terms of the software
 * license in the file "LICENSE" in the root directory of
 * this package.  If this file is missing or corrupt, please
 * contact the author to receive a new copy.
 *
 * Automaton is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  Use at your
 * own risk; your mileage may vary.
 *
 * Suggestions, improvements, and bug reports welcome at
 * <gettings@mines.utah.edu>
 */
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "segy.h"
#include "autopick.h"

#define REEL_HDR_LEN	3600	// length of SEGY reel header, in bytes

/*
 * Check a SEGY file to determine if it has one shot or many,
 * and how many traces it contains.
 *
 * fp		FILE pointer; file need not be at start
 * swapByte	true indicates swap bytes when reading
 * reelHeader	true if need to skip 3600 byte reel header in file
 * nt		set to number of traces
 * ns		set to number of shots
 *
 * Return values:
 *	0	error
 *	1	success
 */
int checkSEGY(FILE *fp, bool swapByte, bool reelHeader, long *nt, long *ns)
{
  unsigned long i, j;
  unsigned short np, dt, k;
  long shot;
  unsigned char hbuf[3600]; // enough space for reel header too.... 
  unsigned char *swapbuf;

  rewind(fp);
  if(reelHeader) {
    // read the reel header
    if(fread(hbuf, 1, REEL_HDR_LEN, fp) < REEL_HDR_LEN) {
      if(feof(fp)) fprintf(stderr, "checkSEGY: unexpected EOF reading reel header\n");
      else fprintf(stderr, "checkSEGY: error reading reel header\n");
      return(0);
      }
    }

  // determine the number of traces
  *nt = 0;
  *ns = 1;
  shot = -1;
  dt = 0;
  while(!feof(fp)) {
    // header
    if(fread(hbuf, 1, 240, fp) < 240) {
      if(feof(fp)) continue; // Done!
      else fprintf(stderr, "checkSEGY: error reading trace header\n");
      return(0);
      }
    // get number of samples in trace
    np = (unsigned short)headerShort(hbuf+114, swapByte);
    k =(unsigned short)headerShort(hbuf+116, swapByte);
    i = headerLong(hbuf+8, swapByte);

    if(shot == -1) shot = i;
    if(dt == 0) dt = k;
    if(i != shot) {
      // shot number changes, so more than one shot!
      (*ns)++;
      shot = i;
      }
    if(k != dt) {
      fprintf(stderr, "checkSEGY: sample interval changes from %d to %d; abort\n", dt, k);
      return(0);
      }
    // read trace data; UTAM uses 32bit trace data, so read ns 4-byte items
    for(i=0; i<np; i++) {
      if(fread(hbuf, 4, 1, fp) < 1) {
	fprintf(stderr, "checkSEGY: error reading trace data while counting traces.\n");
	return(0);
	}
      }
    (*nt)++;
    }
  return(1);
}

/*
 * Read SEGY file with ONLY ONE SHOT into seismicData struct
 *
 * fp		FILE pointer; file need not be at start
 * nt		number of traces in file
 * data		pointer to seismicData variable; will be allocated and
 *		filled by this routine!
 * swapByte	true indicates swap bytes when reading
 * reelHeader	true if need to skip 3600 byte reel header in file
 */
int readSEGYSingleShot(FILE *fp, long nt, seismicData *data, bool swapByte, bool reelHeader)
{
  SEGYHeader header;
  unsigned long i, j, cnt;
  unsigned short ns;
  unsigned char hbuf[3600]; // enough space for reel header too.... 
  unsigned char *swapbuf;

  rewind(fp);
  if(reelHeader) {
    // read the reel header
    if((cnt = fread(hbuf, 1, REEL_HDR_LEN, fp)) < REEL_HDR_LEN) {
      if(feof(fp)) fprintf(stderr, "readSEGYSingleShot: unexpected EOF reading reel header\n");
      else fprintf(stderr, "readSEGYSingleShot: error reading reel header\n");
      return(0);
      }
    }

  // allocate seismicData struct
  *data = newSeismic(nt);

  // read the data and store
  for(i=0; i<data->n; i++) {
    // trace header
    if((cnt = fread(hbuf, 1, 240, fp)) < 240) {
      if(feof(fp)) fprintf(stderr, "readSEGYSingleShot: unexpected EOF reading trace header\n");
      else fprintf(stderr, "readSEGYSingleShot: error reading trace header\n");
      return(0);
      }
    header = grokHeader(hbuf, swapByte);
    data->shot = header.fldr;
    // fill seismic data struct with header info
    if(header.counit == 2) {
      fprintf(stderr, "readSEGYSingleShot: receiver positions in arcseconds; check velocity values!\n");
      }
    data->x[i] = header.gx*pow(10, header.scalco); 
    data->y[i] = header.gy*pow(10, header.scalco);
    data->num[i] = header.tracf;
    data->data[i] = newTrace(header.ns);
    if(data->data[i].n == 0) {
      fprintf(stderr, "readSEGYSingleShot: unable to allocate memory for trace #%d.\n", i+1);
      return(0);
      }
    for(j=0; j<header.ns; j++) {
      if(fread(hbuf, 4, 1, fp) < 1) {
	if(feof(fp)) fprintf(stderr, "readSEGYSingleShot: unexpected EOF reading trace data\n");
	else fprintf(stderr, "readSEGYSingleShot: error reading trace data\n");
	return(0);
	}
      // compute time offset for each sample
      data->data[i].time[j] = i*((float)header.dt/1000.0) + header.delrt;
      // compute amplitude
      if(swapByte) {
        swapbuf = swap4bytes(hbuf);
        data->data[i].amp[j] = *(long *)swapbuf; // UTAM uses 32bit int data
        }
      else {
	data->data[i].amp[j] = *(long *)hbuf;
	}
      }
    }
  return(1);
}

/*
 * Read SEGY file with MANY SHOTS into seismicData struct array
 *
 * fp		FILE pointer; file need not be at start
 * ns		number of shots in file
 * swapByte	true indicates swap bytes when reading
 * reelHeader	true if need to skip 3600 byte reel header in file
 */
seismicData *readSEGYMultiShot(FILE *fp, long ns, bool swapByte, bool reelHeader)
{
  SEGYHeader header;
  unsigned long i, j, k;
  unsigned short np;
  long shot, *nt, *idx, ntt;
  unsigned char hbuf[3600]; // enough space for reel header too.... 
  unsigned char *swapbuf;

  seismicData *S;

  if((S = (seismicData *)malloc(sizeof(seismicData)*ns)) == NULL) {
    fprintf(stderr, "readSEGYMultiShot: cannot allocate memory for seismicData array; abort\n");
    return(NULL);
    }
  if((nt = (long *)malloc(sizeof(long)*ns)) == NULL) {
    fprintf(stderr, "readSEGYMultiShot: cannot allocate memory for nt array; abort\n");
    return(NULL);
    }
  if((idx = (long *)malloc(sizeof(long)*ns)) == NULL) {
    fprintf(stderr, "readSEGYMultiShot: cannot allocate memory for nt array; abort\n");
    return(NULL);
    }

  for(k=0; k<ns; k++) {
    nt[k] = 0;
    idx[k]= 0;
    }

  rewind(fp);
  if(reelHeader) {
    // read the reel header
    if(fread(hbuf, 1, REEL_HDR_LEN, fp) < REEL_HDR_LEN) {
      if(feof(fp)) fprintf(stderr, "readSEGYMultiShot: unexpected EOF reading reel header\n");
      else fprintf(stderr, "readSEGYMultiShot: error reading reel header\n");
      return(NULL);
      }
    }

  // determine number of traces in each shot; probably equal but maybe not
  k = 0;
  while(!feof(fp)) {
    if(fread(hbuf, 1, 240, fp) < 240) { // trace header
      if(feof(fp)) continue; // Done!
      else fprintf(stderr, "readSEGYMultiShot: error reading trace header while counting traces\n");
      return(NULL);
      }
    // get number of samples in trace
    np = (unsigned short)headerShort(hbuf+114, swapByte);
    // get shot number
    shot = headerLong(hbuf+8, swapByte);
    for(j=0; j<k; j++) {
      if(S[j].shot == shot) {
        nt[j]++;
        break;
        }
      }
    if(j == k) { // no match, so set next entry in data array
      S[k].shot = shot;
      nt[k]++;
      k++;
      }
    // read trace data; UTAM uses 32bit trace data, so read ns 4-byte items
    for(i=0; i<np; i++) {
      if(fread(hbuf, 4, 1, fp) < 1) {
	fprintf(stderr, "readSEGYMultiShot: error reading trace data while counting traces\n");
	return(NULL);
	}
      }
    }

  // allocate seismicData struct
  ntt = 0;
  for(i=0; i<ns; i++) {
    shot = S[i].shot;
    S[i] = newSeismic(nt[i]);
    if(S[i].n == 0) {
      fprintf(stderr, "readSEGYMultiShot: cannot allocate memory for shot #%d; abort\n", i+1);
      return(NULL);
      }
    S[i].shot = shot;
    ntt += nt[i];
    }

  // reset file for data read
  rewind(fp);
  if(reelHeader) {
    // read the reel header
    if(fread(hbuf, 1, REEL_HDR_LEN, fp) < REEL_HDR_LEN) {
      if(feof(fp)) fprintf(stderr, "readSEGYMultiShot: unexpected EOF reading reel header\n");
      else fprintf(stderr, "readSEGYMultiShot: error reading reel header\n");
      return(NULL);
      }
    }

  // read the data and store
  for(i=0; i<ntt; i++) {
    // trace header
    if(fread(hbuf, 1, 240, fp) < 240) {
      if(feof(fp)) fprintf(stderr, "readSEGYMultiShot: unexpected EOF reading trace header\n");
      else fprintf(stderr, "readSEGYMultiShot: error reading trace header\n");
      return(NULL);
      }
    header = grokHeader(hbuf, swapByte);
    shot = header.fldr;
    // find index for data array
    for(k=0; k<ns; k++) {
      if(S[k].shot == shot) break;
      }
    if(k == ns) {
      fprintf(stderr, "readSEGYMultiShot: error reading data- shot number %d not in array\n", shot);
      return(NULL);
      }
    // fill seismic data struct with header info
    if(header.counit == 2) {
      fprintf(stderr, "readSEGYMultiShot: receiver positions in arcseconds; check velocity values!\n");
      }
    S[k].x[idx[k]] = header.gx*pow(10, header.scalco); 
    S[k].y[idx[k]] = header.gy*pow(10, header.scalco);
    S[k].num[idx[k]] = header.tracf;
    S[k].data[idx[k]] = newTrace(header.ns);
    if(S[k].data[idx[k]].n == 0) {
      fprintf(stderr, "readSEGYMultiShot: unable to allocate memory for trace #%d.\n", i+1);
      return(NULL);
      }
    for(j=0; j<header.ns; j++) {
      if(fread(hbuf, 4, 1, fp) < 1) {
	if(feof(fp)) fprintf(stderr, "readSEGYMultiShot: unexpected EOF reading trace data\n");
	else fprintf(stderr, "readSEGYMultiShot: error reading trace data\n");
	return(NULL);
	}
      // compute time offset for each sample
      S[k].data[idx[k]].time[j] = j*((float)header.dt/1000.0) + header.delrt;
      // compute amplitude
      if(swapByte) {
        swapbuf = swap4bytes(hbuf);
        S[k].data[idx[k]].amp[j] = *(long *)swapbuf; // UTAM uses 32bit long
        }
      else {
	S[k].data[idx[k]].amp[j] = *(long *)hbuf;
	}
      }
    idx[k]++;
    }
  return(S);
}

/*
 * Break apart SEGY Header block into structure
 */
SEGYHeader grokHeader(unsigned char buf[240], bool swap)
{
  SEGYHeader h;

  h.tracl =	headerLong (buf+  0, swap);
  h.tracr =	headerLong (buf+  4, swap);
  h.fldr =	headerLong (buf+  8, swap);
  h.tracf =	headerLong (buf+ 12, swap);
  h.ep =	headerLong (buf+ 16, swap);
  h.cdp =	headerLong (buf+ 20, swap);
  h.cdpt =	headerLong (buf+ 24, swap);
  h.trid =	headerShort(buf+ 28, swap);
  h.nvs =	headerShort(buf+ 30, swap);
  h.nhs =	headerShort(buf+ 32, swap);
  h.duse =	headerShort(buf+ 34, swap);
  h.offset =	headerLong (buf+ 36, swap);
  h.gelev =	headerLong (buf+ 40, swap);
  h.selev =	headerLong (buf+ 44, swap);
  h.sdepth =	headerLong (buf+ 48, swap);
  h.gdel =	headerLong (buf+ 52, swap);
  h.sdel =	headerLong (buf+ 56, swap);
  h.swdep =	headerLong (buf+ 60, swap);
  h.gwdep =	headerLong (buf+ 64, swap);
  h.scalel =	headerShort(buf+ 68, swap);
  h.scalco =	headerShort(buf+ 70, swap);
  h.sx =	headerLong (buf+ 72, swap);
  h.sy =	headerLong (buf+ 76, swap);
  h.gx =	headerLong (buf+ 80, swap);
  h.gy =	headerLong (buf+ 84, swap);
  h.counit =	headerShort(buf+ 88, swap);
  h.wevel =	headerShort(buf+ 90, swap);
  h.swevel =	headerShort(buf+ 92, swap);
  h.sut =	headerShort(buf+ 94, swap);
  h.gut =	headerShort(buf+ 96, swap);
  h.sstat =	headerShort(buf+ 98, swap);
  h.gstat =	headerShort(buf+100, swap);
  h.tstat =	headerShort(buf+102, swap);
  h.laga =	headerShort(buf+104, swap);
  h.lagb =	headerShort(buf+106, swap);
  h.delrt =	headerShort(buf+108, swap);
  h.muts =	headerShort(buf+110, swap);
  h.mute =	headerShort(buf+112, swap);
  h.ns = (unsigned short)headerShort(buf+114, swap);
  h.dt = (unsigned short)headerShort(buf+116, swap);
  h.gain =	headerShort(buf+118, swap);
  h.igc =	headerShort(buf+120, swap);
  h.igi =	headerShort(buf+122, swap);
  h.corr =	headerShort(buf+124, swap);
  h.sfs =	headerShort(buf+126, swap);
  h.sfe =	headerShort(buf+128, swap);
  h.slen =	headerShort(buf+130, swap);
  h.styp =	headerShort(buf+132, swap);
  h.stas =	headerShort(buf+134, swap);
  h.stae =	headerShort(buf+136, swap);
  h.tatyp =	headerShort(buf+138, swap);
  h.afilf =	headerShort(buf+140, swap);
  h.afils =	headerShort(buf+142, swap);
  h.nofilf =	headerShort(buf+144, swap);
  h.nofils =	headerShort(buf+146, swap);
  h.lcf =	headerShort(buf+148, swap);
  h.hcf =	headerShort(buf+150, swap);
  h.lcs =	headerShort(buf+152, swap);
  h.hcs =	headerShort(buf+154, swap);
  h.year =	headerShort(buf+156, swap);
  h.day =	headerShort(buf+158, swap);
  h.hour =	headerShort(buf+160, swap);
  h.minute =	headerShort(buf+162, swap);
  h.sec =	headerShort(buf+164, swap);
  h.timbas =	headerShort(buf+166, swap);
  h.trwf =	headerShort(buf+168, swap);
  h.grnors =	headerShort(buf+170, swap);
  h.grnofr =	headerShort(buf+172, swap);
  h.grnlof =	headerShort(buf+174, swap);
  h.gaps =	headerShort(buf+176, swap);
  h.otrav =	headerShort(buf+178, swap);
  h.d1 =	headerFloat(buf+180, swap);
  h.f1 =	headerFloat(buf+184, swap);
  h.d2 =	headerFloat(buf+188, swap);
  h.f2 =	headerFloat(buf+192, swap);
  h.ungpow =	headerFloat(buf+196, swap);
  h.unscale =	headerFloat(buf+200, swap);
  h.mark =	headerShort(buf+204, swap);
  return(h);
}

/* Convert 4 bytes of header to long int
 */
long headerLong(unsigned char *buf, bool swap)
{
  long l, *p;

  p=(long *)buf;
  if(!swap){
    l = *p;
    }
  else {
    p = (long *)swap4bytes(buf);
    l = *p;
    }
  return(l);
}

/* Convert long int position to float, using scale factor "scale"
 * calls headerLong to decode the buffer, and rescales to float
 */
float headerPosition(unsigned char *buf, bool swap, short scale)
{
  float x;
  long k;

  k = headerLong(buf, swap);
  x = (float)k * pow(10, scale);
  return(x);
}

/* Convert 4 bytes of header to float
 */
float headerFloat(unsigned char *buf, bool swap)
{
  float f, *p;

  p = (float*)buf;
  if(!swap){
    f = *p;
    }
  else {
    p = (float *)swap4bytes(buf);
    f = *p;
    }
  return(f);
}

/* Convert 4 bytes of header to short int
 */
short headerShort(unsigned char *buf, bool swap)
{
  short s, *p;
  char b[2];

  p=(short*)buf;
  if(!swap){
    s = *p;
    }
  else {
    b[0] = buf[1];
    b[1] = buf[0];
    p = (short*)b;
    s = *p;
    }
  return(s);
}

/* swap 4 bytes from 1234 to 4321
 */
unsigned char *swap4bytes(unsigned char *p)
{
  int i;
  static unsigned char b[4];
  unsigned char *ptr;

  ptr = p+3;
  for(i=0; i<4; i++) {
    b[i] = *ptr;
    ptr--;
    }
  return(b);
}
