/*
 * positionSEGY.cpp - replace position data in SEGY file
 *
 * This file is part of Automaton.
 *
 * Copyright (C) 2002, 2003
 * Paul Gettings, Dep't of Geology & Geophysics
 * University of Utah
 *
 * This file is released under the terms of the software
 * license in the file "LICENSE" in the root directory of
 * this package.  If this file is missing or corrupt, please
 * contact the author to receive a new copy.
 *
 * Automaton is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  Use at your
 * own risk; your mileage may vary.
 *
 * Suggestions, improvements, and bug reports welcome at
 * <gettings@mines.utah.edu>
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "segy.h"
#include "autopick.h"

typedef struct positionStruct {
  long shot;
  long chan;
  float x, y;
  } Position;

int main(int argc, char *argv[])
{
  long i, j, k;
  FILE *pfp, *sfp;
  long ns, shot, chan;
  char record[1024];
  unsigned char buf[3600];
  Position *P;

  bool swapBytes = true; // SEGY is big-endian, and Intel is little-

  if(argc < 3) {
    fprintf(stderr, "usage: %s position_file segy_file [segy_file ...]\n", argv[0]);
    fprintf(stderr, "SEGY files should NOT have reel headers!\n");
    exit(1);
    }

  if((pfp = fopen(argv[1], "rt")) == NULL) {
    fprintf(stderr, "cannot open '%s' for read.\n", argv[1]);
    exit(1);
    }
  // read the position data
  ns = 0;
  while(!feof(pfp)) {
    if(!nextRecord(pfp, record)) {
      continue;
      }
    ns++;
    }
  rewind(pfp);
  if((P = (Position *)malloc(sizeof(Position)*ns)) == NULL) {
    fprintf(stderr, "cannot allocate memory for %d positions.\n", ns);
    exit(1);
    }

  i=0;
  while(!feof(pfp)) {
    if(!nextRecord(pfp, record)) {
      continue;
      }
    sscanf(record, "%d %d %f %f\n", &P[i].shot, &P[i].chan, &P[i].x, &P[i].y);
    i++;
    }
  fclose(pfp);

  for(i=2; i<argc; i++) {
    if((sfp = fopen(argv[i], "r+b")) == NULL) {
      fprintf(stderr, "cannot open '%s' for read/write.\n", argv[i]);
      exit(1);
      }
    // assume there is no reel header
    unsigned long pos;
    while(!feof(sfp)) {
      pos = ftell(sfp);
      // read bits of the trace header to get shot #, chan #
      fseek(sfp, pos+8L, SEEK_SET);
      fread(buf, 4, 1, sfp);
      if(feof(sfp)) continue;
      shot = headerLong(buf, swapBytes);
      fread(buf, 4, 1, sfp);
      chan = headerLong(buf, swapBytes);
//XXX DEBUG
fprintf(stdout, "main: shot:%d chan:%d\n", shot, chan);
      // find position for this trace
      for(j=0; j<ns; j++) {
        if(P[j].shot == shot && P[j].chan == chan) {
          // write position to file
	  long xl, yl;
	  long *p;
	  unsigned char *swapbuf;
	  fseek(sfp, pos+70, SEEK_SET);
	  /* need to write the scale factor for receiver positions
	   * since the scale factor is hard coded in this program
	   * we can explicitly write the two necessary bytes
	   *
	   * These bytes are for a scale factor of -3, stored in
	   * 2's complement form
	   */
	  if(swapBytes) {
	    buf[0] = 0xff; buf[1] = 0xfd;
	    }
	  else {
	    buf[0] = 0xfd; buf[1] = 0xff;
	    }
	  fwrite(buf, 1, 2, sfp);

	  fseek(sfp, pos+80, SEEK_SET);
	  xl = (long) (P[j].x*1000); // assume mm level accuracy
	  yl = (long) (P[j].y*1000);
	  p = &xl;
	  if(swapBytes) {
	    swapbuf = swap4bytes((unsigned char *)p);
	    }
	  else swapbuf = (unsigned char *)p;
	  fwrite(swapbuf, 4, 1, sfp);
	  p = &yl;
	  if(swapBytes) {
	    swapbuf = swap4bytes((unsigned char *)p);
	    }
	  else swapbuf = (unsigned char *)p;
	  fwrite(swapbuf, 4, 1, sfp);
          break;
          }
	}
      // move to next header
      fseek(sfp, pos+114, SEEK_SET);
      fread(buf, 2, 1, sfp);
      unsigned short nsamp;
      nsamp = (unsigned short) headerShort(buf, swapBytes);
      fseek(sfp, pos+240+(nsamp*4), SEEK_SET); // skip trace data
      }
    fclose(sfp);
    }
  exit(0);
}

