/*
 * autopick.h - automatic picker header file
 *
 * This file is part of Automaton.
 *
 * Copyright (C) 2002, 2003
 * Paul Gettings, Dep't of Geology & Geophysics
 * University of Utah
 *
 * This file is released under the terms of the software
 * license in the file "LICENSE" in the root directory of
 * this package.  If this file is missing or corrupt, please
 * contact the author to receive a new copy.
 *
 * Automaton is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  Use at your
 * own risk; your mileage may vary.
 *
 * Suggestions, improvements, and bug reports welcome at
 * <gettings@mines.utah.edu>
 */
#ifndef AUTOPICK
#define AUTOPICK

#include <stdio.h>		// for FILE * declarations

/* Constants
 */
#define PI	3.141592654	// isn't this in math.h?
#define sqr(x)	(x*x)		// this ought to be in math.h


/* DATA TYPES
 */
/* seismic trace
 *
 * n	number of points in the trace
 * time	time offsets, in ms, in global time frame
 * amp	amplitude for each time
 */
typedef struct traceStruct {
  unsigned long n;
  float *time;
  float *amp;
  } trace;

/* seismic data set structure
 *
 * n	number of traces
 * shot shot number, for determining which pick to use
 * data	pointer to array of n traces
 * x,y	position of each trace
 * num	geophone number
 */
typedef struct seismicDataStruct {
  unsigned long n;
  long shot;
  trace *data;
  float *x, *y;
  long *num;
  } seismicData;

/* wavelet structure
 *
 * n	number of points in the wavelet
 * time	time offset in ms, 0 is start of wavelet
 * amp	amplitude for each time offset
 */
typedef struct waveletStruct {
  unsigned long n;
  float *time;
  float *amp;
  } wavelet;

/* known pick data
 *
 * t		seismic trace data
 * sx,sy	position of source
 * x,y		position of trace
 * start	start time in ms from global time start
 * length	length of wavelet in ms
 * w		wavelet
 */
typedef struct knownPickStruct {
  trace t;
  float sx, sy;
  float x, y;
  float start;
  float length;
  wavelet w;
  } knownPick;

/* Parameter structure
 *
 * This holds various values that tweak the process
 */
typedef struct autopickParmStruct {
  double Threshhold;	// membership function threshhold for acceptance
  double AmpConst;	// decay constant for amplitude difference in M()
  double TimeConst;	// decay constant for time offset difference in M()
  double Mw[3];		// weights for membership function M()
  double Velocity;	// a priori velocity for predicted time offsets
  double MinVelocity;	// minimum velocity to compute data window start
  double MaxVelocity;	// maximum velocity to compute data window end
  double Attenuation;	// attenuation of amplitude with offset
                    	// attenuation value must be in (0,inf]
                    	//   att. < 1 slows 1/d^2 drop, >1 speeds
  double WaveTime;	// length, in ms, of wavelet for fitting
  } autopickParameters;

/* FUNCTION PROTOTYPES
 */
/* pick first arrivals
 * this function is the external interface; main() loads the data and
 * checks parms before calling autopick()
 * FILE *out is the output file for successful picks
 * FILE *dbg is for debugging output to help user tweak parameters
 * returns 0 on fatal error
 */
int autopick(seismicData data, autopickParameters parms, knownPick known, FILE
*out, FILE *dbg);

/* Constructors for seismic data, wavelet, and trace types */
seismicData newSeismic(unsigned long np);
wavelet newWavelet(unsigned long np);
trace newTrace(unsigned long np);
void killWavelet(wavelet *w);
void killSeismic(seismicData *s);
void killTrace(trace *t);

/* compute wavelet amplitude at time offset t, using wavelet W */
double wave(wavelet W, double t);

/* least-squares fit for wavelet to trace */
int fit_wavelet(trace T, wavelet W, double start, double end, double *amp, double *dt, double *r);

/* Membership function */
double M(double r, double da, double dt, autopickParameters p);

/* utility functions */
autopickParameters grokCommandFile(char *name);
int computeWavelet(knownPick *k);
void strstrip(char *s);
seismicData readSeismicData(char *name);
knownPick readKnownPick(char *name, seismicData data, autopickParameters p);
int nextRecord(FILE *fp, char *record);


#endif
