/*
 * arsegy.cpp - extract/delete traces from SEGY files
 *
 * This file is part of Automaton.
 *
 * Copyright (C) 2002, 2003
 * Paul Gettings, Dep't of Geology & Geophysics
 * University of Utah
 *
 * This file is released under the terms of the software
 * license in the file "LICENSE" in the root directory of
 * this package.  If this file is missing or corrupt, please
 * contact the author to receive a new copy.
 *
 * Automaton is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  Use at your
 * own risk; your mileage may vary.
 *
 * Suggestions, improvements, and bug reports welcome at
 * <gettings@mines.utah.edu>
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <limits.h>

#include "segy.h"
#include "arsegy.h"

#define PRINT	1
#define COPY	2 // not needed?
#define EXTRACT	4
#define DELETE	8

#define MAXNAMELEN	2048

bool swapBytes = true;
bool reelHeader = false;
#define REEL_HDR_LEN	3600

void usage(void)
{
  printf("arsegy -{p|c|x|d} -i infile -o outfile -t startT-endT -s startS-endS\n");
  printf("  -h	print usage info\n");
  printf("  -p	print segy header fields for all traces in file\n");
  printf("  -x	extract traces from infile to outfile\n");
  printf("  -d	copy infile to outfile without some traces\n");
  printf("\n");
  printf("  -t	range of traces for c/d/x\n");
  printf("  -s	range of shots for c/d/x\n");
  printf("      ranges are of the form: start-end, where the '-' is needed\n");
  printf("\n");
  printf("  -R	skip reel header at start of file; never copied!\n");
  printf("  -S	toggle byte swapping; default is on\n");
}

int main(int argc, char *argv[])
{
  const char *optstr = "hpxdi:o:t:s:S";
  long startT, endT, startS, endS;
  char infile[MAXNAMELEN], outfile[MAXNAMELEN];
  FILE *ifp, *ofp;
  int c;
  unsigned char Mode=PRINT; // default action is to print headers

  strcpy(infile, "-"); strcpy(outfile, "-");

  // not set, use any
  parseRange("any", &startT, &endT);
  parseRange("any", &startS, &endS);

  // deal with cmd-line args
  while((c = getopt(argc, argv, optstr)) != -1) {
    switch(c) {
      case 'p': Mode = PRINT; break;
      case 'x': Mode = EXTRACT; break;
      case 'd': Mode = DELETE; break;
      case 'i': strncpy(infile, optarg, MAXNAMELEN); break;
      case 'o': strncpy(outfile, optarg, MAXNAMELEN); break;
      case 't': if(!parseRange(optarg, &startT, &endT)) {
      		  fprintf(stderr, "Error parsing trace range '%s'.\n", optarg);
      		  exit(1);
      		  }
		break;
      case 's': if(!parseRange(optarg, &startS, &endS)) {
      		  fprintf(stderr, "Error parsing shot range '%s'.\n", optarg);
      		  exit(1);
      		  }
		break;
      case 'R': reelHeader = true; break;
      case 'S': swapBytes = !swapBytes; break;
      case 'h':
      case '?':
      default:
      		usage();
      		exit(1);
      		break;
      }
    }

  // open files
  if(!strcmp(infile, "-")) {
    ifp = stdin;
    }
  else {
    if((ifp = fopen(infile, "rb")) == NULL) {
      fprintf(stderr, "Cannot open '%s' for read.\n", infile);
      exit(1);
      }
    }
  if(!strcmp(outfile, "-")) {
    ofp = stdout;
    }
  else {
    if((ofp = fopen(outfile, "wb")) == NULL) {
      fprintf(stderr, "Cannot open '%s' for write.\n", outfile);
      exit(1);
      }
    }
  // switch on what to do
  switch(Mode) {
    default:
    case PRINT  : printHeaders(ifp); break;
    case EXTRACT: extractSEGY(ifp, ofp, startT, endT, startS, endS); break;
    case DELETE : deleteSEGY(ifp, ofp, startT, endT, startS, endS); break;
    }
  if(ifp != stdin)
    fclose(ifp);
  if(ofp != stdout)
    fclose(ofp);
  exit(0);
}

/*
 * Parse a range argument, with "any" taken as wildcard
 */
int parseRange(char *str, long *s, long *e)
{
  if(!strcasecmp(str, "any")) {
    *s = LONG_MIN;
    *e = LONG_MAX;
    }
  else {
    sscanf(str, "%d-%d", s, e);
    }
  return(1);
}

/*
 * Copy infile to outfile, but only traces in ranges
 */
int extractSEGY(FILE *fp, FILE *out, long sT, long eT, long sS, long eS)
{
  long i, j;
  SEGYHeader H;
  unsigned char hbuf[REEL_HDR_LEN];

  rewind(fp);
  // read the reel header, if any
  if(reelHeader) {
    if(fread(hbuf, 1, REEL_HDR_LEN, fp) < REEL_HDR_LEN) {
      if(feof(fp)) fprintf(stderr, "extractSEGY: unexpected EOF reading reel header\n");
      else fprintf(stderr, "extractSEGY: error reading reel header\n");
      return(0);
      }
    }

  // step through file
  bool flag;
  while(!feof(fp)) {
    flag=false;
    // trace header
    if(fread(hbuf, 1, 240, fp) < 240) {
      if(feof(fp)) continue; // Done!
      else fprintf(stderr, "extractSEGY: error reading trace header\n");
      return(0);
      }
    // parse the header
    H = grokHeader(hbuf, swapBytes);
    if(sS <= H.fldr & H.fldr <= eS & sT <= H.tracf & H.tracf <= eT) {
      fwrite(hbuf, 1, 240, out);
      flag=true;
      }
    else {
      flag = false;
      }
    // read trace data; UTAM uses 32bit trace data, so read ns 4-byte items
    for(i=0; i<H.ns; i++) {
      if(fread(hbuf, 4, 1, fp) < 1) {
	fprintf(stderr, "extractSEGY: error reading trace data while counting traces.\n");
	return(0);
	}
      if(flag) {
	fwrite(hbuf, 4, 1, out);
	}
      }
    }
  return(1);
}

/*
 * Copy infile to outfile, but without traces in ranges
 */
int deleteSEGY(FILE *fp, FILE *out, long sT, long eT, long sS, long eS)
{
  long i, j;
  SEGYHeader H;
  unsigned char hbuf[REEL_HDR_LEN];

  rewind(fp);
  // read the reel header, if any
  if(reelHeader) {
    if(fread(hbuf, 1, REEL_HDR_LEN, fp) < REEL_HDR_LEN) {
      if(feof(fp)) fprintf(stderr, "extractSEGY: unexpected EOF reading reel header\n");
      else fprintf(stderr, "extractSEGY: error reading reel header\n");
      return(0);
      }
    }

  // step through file
  bool flag;
  while(!feof(fp)) {
    flag=true;
    // trace header
    if(fread(hbuf, 1, 240, fp) < 240) {
      if(feof(fp)) continue; // Done!
      else fprintf(stderr, "extractSEGY: error reading trace header\n");
      return(0);
      }
    // parse the header
    H = grokHeader(hbuf, swapBytes);
    if(sS <= H.fldr & H.fldr <= eS & sT <= H.tracf & H.tracf <= eT) {
      flag=false;
      }
    else {
      fwrite(hbuf, 1, 240, out);
      flag=true;
      }
    // read trace data; UTAM uses 32bit trace data, so read ns 4-byte items
    for(i=0; i<H.ns; i++) {
      if(fread(hbuf, 4, 1, fp) < 1) {
	fprintf(stderr, "extractSEGY: error reading trace data while counting traces.\n");
	return(0);
	}
      if(flag) {
	fwrite(hbuf, 4, 1, out);
	}
      }
    }
  return(1);
}

/*
 * Print out the important info from SEGY trace headers
 *
 * fp		FILE pointer; file need not be at start
 *
 * Return values:
 *	0	error
 *	1	success
 */
int printHeaders(FILE *fp)
{
  SEGYHeader H;
  unsigned char hbuf[3600]; // enough space for reel header too.... 

  rewind(fp);
  // read the reel header, if any
  if(reelHeader) {
    printf("Skipping reel header\n");
    if(fread(hbuf, 1, REEL_HDR_LEN, fp) < REEL_HDR_LEN) {
      if(feof(fp)) fprintf(stderr, "printHeader: unexpected EOF reading reel header\n");
      else fprintf(stderr, "printHeader: error reading reel header\n");
      return(0);
      }
    }

  // print column headers
  printf("line   reel   field  field  scle source source group  group  numbr delta\n");
  printf("trace  trace  record trace  10^X    X      Y      X      Y   sampl t(us)\n");

  // step through file
  while(!feof(fp)) {
    // trace header
    if(fread(hbuf, 1, 240, fp) < 240) {
      if(feof(fp)) continue; // Done!
      else fprintf(stderr, "printHeader: error reading trace header\n");
      return(0);
      }
    // parse the header
    H = grokHeader(hbuf, swapBytes);
    // read trace data; UTAM uses 32bit trace data, so read ns 4-byte items
    long i;
    for(i=0; i<H.ns; i++) {
      if(fread(hbuf, 4, 1, fp) < 1) {
	fprintf(stderr, "printHeader: error reading trace data while counting traces.\n");
	return(0);
	}
      }
    // print header info
    // Use just over 80 columns, but that's too bad
    printf("%6d %6d %6d %6d %4hd %6d %6d %6d %6d %5hu %5hu\n", 
           H.tracl, H.tracr, H.fldr, H.tracf, H.scalco, H.sx, H.sy, H.gx, H.gy, H.ns, H.dt);
    }
  return(1);
}

